<?php
/**
 * BeautyAgency functions and definitions
 *
 * Set up the theme and provides some helper functions, which are used in the
 * theme as custom template tags. Others are attached to action and filter
 * hooks in WordPress to change core functionality.
 *
 * When using a child theme you can override certain functions (those wrapped
 * in a function_exists() call) by defining them first in your child theme's
 * functions.php file. The child theme's functions.php file is included before
 * the parent theme's file, so the child theme functions would be used.
 *
 * @link https://codex.wordpress.org/Theme_Development
 * @link https://codex.wordpress.org/Child_Themes
 *
 * Functions that are not pluggable (not wrapped in function_exists()) are
 * instead attached to a filter or action hook.
 *
 * For more information on hooks, actions, and filters,
 * {@link https://codex.wordpress.org/Plugin_API}
 */

define( 'BEAUTYAGENCY_NAME', 'beautyagency' );

/* not used yet */
define( 'BEAUTYAGENCY_VERSION', '1.0.28' );
define( 'BEAUTYAGENCY_PATH', esc_url(get_template_directory_uri()) . '/' );
/* /not used yet */

// You may request this 'REMOTE_DEMO_ID' value from this theme author to get a colorized demo content.
// See the Theme support service contacts information.
define( 'REMOTE_DEMO_ID', ''); // as example: '12345678'
define( 'REMOTE_DEMO_VERSION', '1.0.28');

include( get_template_directory() . '/inc/init.php' );

/**
 * Set the content width based on the theme's design and stylesheet.
 */
if ( ! isset( $content_width ) ) {
	$content_width = 960;
}

if ( version_compare( $GLOBALS['wp_version'], '4.1-alpha', '<' ) ) {
	require get_template_directory() . '/inc/back-compat.php';
}

$kses_list = '';

if ( ! function_exists( 'beautyagency_setup' ) ) :
	/**
	 * Sets up theme defaults and registers support for various WordPress features.
	 *
	 * Note that this function is hooked into the after_setup_theme hook, which
	 * runs before the init hook. The init hook is too late for some features, such
	 * as indicating support for post thumbnails.
	 */
	function beautyagency_setup() {

		/*
		 * Make theme available for translation.
		 * Translations can be filed in the /languages/ directory.
		 * If you're building a theme based on BeautyAgency, use a find and replace
		 * to change 'beautyagency' to the name of your theme in all the template files
		 */
		load_theme_textdomain( 'beautyagency', get_template_directory() . '/languages' );

		// Add default posts and comments RSS feed links to head.
		add_theme_support( 'automatic-feed-links' );

		/*
		 * Let WordPress manage the document title.
		 * By adding theme support, we declare that this theme does not use a
		 * hard-coded <title> tag in the document head, and expect WordPress to
		 * provide it for us.
		 */
		add_theme_support( 'title-tag' );

		/*
		 * Enable support for Post Thumbnails on posts and pages.
		 *
		 * See: https://codex.wordpress.org/Function_Reference/add_theme_support#Post_Thumbnails
		 */
		add_theme_support( 'post-thumbnails' );
		set_post_thumbnail_size( 960, 590, true );

		// This theme uses wp_nav_menu() in two locations.
		register_nav_menus( array(
			'primary' => esc_html__( 'Primary Menu', 'beautyagency' ),
			'social'  => esc_html__( 'Social Menu', 'beautyagency' ),
			'footer'  => esc_html__( 'Footer Menu', 'beautyagency' ),
		) );

		/*
		 * Switch default core markup for search form, comment form, and comments
		 * to output valid HTML5.
		 */
		add_theme_support( 'html5', array(
			'search-form',
			'comment-form',
			'comment-list',
			'gallery',
			'caption'
		) );

		/*
		 * Enable support for Post Formats.
		 *
		 * See: https://codex.wordpress.org/Post_Formats
		 */
		add_theme_support( 'post-formats', array(
			'aside',
			'image',
			'video',
			'quote',
			'link',
			'gallery',
			'status',
			'audio',
			'chat'
		) );

		// Setup the WordPress core custom background feature.
		add_theme_support( 'custom-background', apply_filters( 'beautyagency_custom_background_args', array(
			'default-attachment' => 'fixed',
		) ) );

		/*
		 * This theme styles the visual editor to resemble the theme style,
		 * specifically font, colors, icons, and column width.
		 */
		add_editor_style( array( 'css/editor-style.css', beautyagency_fonts_url() ) );
	}
endif; // beautyagency_setup
add_action( 'after_setup_theme', 'beautyagency_setup' );

if ( ! function_exists( 'beautyagency_widgets_init' ) ) :
	/**
	 * Register widget area.
	 *
	 * @link https://codex.wordpress.org/Function_Reference/register_sidebar
	 */
	function beautyagency_widgets_init() {
		register_sidebar( array(
			'name'          => esc_html__( 'Sidebar', 'beautyagency' ),
			'id'            => 'sidebar-1',
			'description'   => esc_html__( 'Add widgets here to appear in your sidebar.', 'beautyagency' ),
			'before_widget' => '<aside id="%1$s" class="widget %2$s">',
			'after_widget'  => '</aside>',
			'before_title'  => '<h2 class="widget-title">',
			'after_title'   => '</h2>',
		) );


		register_sidebar( array(
			'name'          => esc_html__( 'Widgets After Header (Full-Width)', 'beautyagency' ),
			'id'            => 'sidebar-alfter-header',
			'description'   => esc_html__( 'Add widgets here to appear full-width after header.', 'beautyagency' ),
			'before_widget' => '<div id="%1$s" class="widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widget-title">',
			'after_title'   => '</h3>',
		) );


		register_sidebar( array(
			'name'          => esc_html__( 'Widgets Before Content', 'beautyagency' ),
			'id'            => 'sidebar-top',
			'description'   => esc_html__( 'Add widgets here to appear over content.', 'beautyagency' ),
			'before_widget' => '<div id="%1$s" class="widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widget-title">',
			'after_title'   => '</h3>',
		) );

		register_sidebar( array(
			'name'          => esc_html__( 'Widgets Before Loop (Loop-Width)', 'beautyagency' ),
			'id'            => 'sidebar-before-loop',
			'description'   => esc_html__( 'Add widgets here to appear over loop.', 'beautyagency' ),
			'before_widget' => '<div id="%1$s" class="widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widget-title">',
			'after_title'   => '</h3>',
		) );

		register_sidebar( array(
			'name'          => esc_html__( 'Widgets After Loop (Loop-Width)', 'beautyagency' ),
			'id'            => 'sidebar-after-loop',
			'description'   => esc_html__( 'Add widgets here to appear after loop.', 'beautyagency' ),
			'before_widget' => '<div id="%1$s" class="widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widget-title">',
			'after_title'   => '</h3>',
		) );

		register_sidebar( array(
			'name'          => esc_html__( 'Widgets After Content', 'beautyagency' ),
			'id'            => 'sidebar-bottom',
			'description'   => esc_html__( 'Add widgets here to appear under content.', 'beautyagency' ),
			'before_widget' => '<div id="%1$s" class="widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widget-title">',
			'after_title'   => '</h3>',
		) );

		register_sidebar( array(
			'name'          => esc_html__( 'Widgets Before Footer (Full-Width)', 'beautyagency' ),
			'id'            => 'sidebar-before-footer',
			'description'   => esc_html__( 'Add widgets here to appear full-width before footer.', 'beautyagency' ),
			'before_widget' => '<div id="%1$s" class="widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widget-title">',
			'after_title'   => '</h3>',
		) );

		$theme_options = beautyagency_get_theme_mods();
		$footer_layout = ! empty( $theme_options['footer_widgets_layout'] ) ? $theme_options['footer_widgets_layout'] : 3;

		if ( $footer_layout != 0 ) :
			register_sidebar( array(
				'name'          => esc_html__( 'Footer Widget 1', 'beautyagency' ),
				'id'            => 'footer-widget-1',
				'description'   => esc_html__( 'Add Footer Widget 1.', 'beautyagency' ),
				'before_widget' => '<div id="%1$s" class="widget %2$s">',
				'after_widget'  => '</div>',
				'before_title'  => '<h3 class="widget-title">',
				'after_title'   => '</h3>',
			) );

			if ( $footer_layout != 1 ) :
				register_sidebar( array(
					'name'          => esc_html__( 'Footer Widget 2', 'beautyagency' ),
					'id'            => 'footer-widget-2',
					'description'   => esc_html__( 'Add Footer Widget 2.', 'beautyagency' ),
					'before_widget' => '<div id="%1$s" class="widget %2$s">',
					'after_widget'  => '</div>',
					'before_title'  => '<h3 class="widget-title">',
					'after_title'   => '</h3>',
				) );

				if ( $footer_layout == 3 || $footer_layout == 112 || $footer_layout == 121 || $footer_layout == 211 || $footer_layout == 4 ) :
					register_sidebar( array(
						'name'          => esc_html__( 'Footer Widget 3', 'beautyagency' ),
						'id'            => 'footer-widget-3',
						'description'   => esc_html__( 'Add Footer Widget 3.', 'beautyagency' ),
						'before_widget' => '<div id="%1$s" class="widget %2$s">',
						'after_widget'  => '</div>',
						'before_title'  => '<h3 class="widget-title"><span>',
						'after_title'   => '</span></h3>',
					) );

					if ( $footer_layout == 4 ) :
						register_sidebar( array(
							'name'          => esc_html__( 'Footer Widget 4', 'beautyagency' ),
							'id'            => 'footer-widget-4',
							'description'   => esc_html__( 'Add Footer Widget 4.', 'beautyagency' ),
							'before_widget' => '<div id="%1$s" class="widget %2$s">',
							'after_widget'  => '</div>',
							'before_title'  => '<h3 class="widget-title">',
							'after_title'   => '</h3>',
						) );
					endif;
				endif;
			endif;
		endif;
	}
endif;
add_action( 'widgets_init', 'beautyagency_widgets_init' );

if ( ! function_exists( 'beautyagency_footer' ) ) :
	function beautyagency_footer() {
		$theme_options  = beautyagency_get_theme_mods();
		$footer_widgets = beautyagency_get_customizer_option( 'footer_widgets_layout' );
		$widget_widths  = array( '', '', '', '' );
		switch ( $footer_widgets ) {
			case '1':
				$widget_widths = array_replace( $widget_widths, array( 'col-md-12 widget-width__container' ) );
				break;
			case '2':
				$widget_widths = array_replace( $widget_widths, array(
					'col-md-6 col-sm-6 widget-width__loop',
					'col-md-6 col-sm-6 widget-width__loop'
				) );
				break;
			case '3':
				$widget_widths = array_replace( $widget_widths, array(
					'col-md-4 col-sm-6 widget-width__side',
					'col-md-4 col-sm-6 widget-width__side',
					'col-md-4 col-sm-6 widget-width__side'
				) );
				break;
			case '4':
				$widget_widths = array_replace( $widget_widths, array(
					'col-md-3 col-sm-6 widget-width__side',
					'col-md-3 col-sm-6 widget-width__side',
					'col-md-3 col-sm-6 widget-width__side',
					'col-md-3 col-sm-6 widget-width__side'
				) );
				break;
			case '12':
				$widget_widths = array_replace( $widget_widths, array(
					'col-md-4 col-sm-6 widget-width__side',
					'col-md-8 col-sm-12 widget-width__loop'
				) );
				break;
			case '21':
				$widget_widths = array_replace( $widget_widths, array(
					'col-md-8 col-sm-12 widget-width__loop',
					'col-md-4 col-sm-6 widget-width__side'
				) );
				break;
			case '112':
				$widget_widths = array_replace( $widget_widths, array(
					'col-md-3 col-sm-6 widget-width__side',
					'col-md-3 col-sm-6 widget-width__side',
					'col-md-6 col-sm-12 widget-width__loop'
				) );
				break;
			case '121':
				$widget_widths = array_replace( $widget_widths, array(
					'col-md-3 col-sm-6 widget-width__side',
					'col-md-6 col-sm-12 widget-width__loop',
					'col-md-3 col-sm-6 widget-width__side'
				) );
				break;
			case '211':
				$widget_widths = array_replace( $widget_widths, array(
					'col-md-6 col-sm-12 widget-width__loop',
					'col-md-3 col-sm-6 widget-width__side',
					'col-md-3 col-sm-6 widget-width__side'
				) );
				break;
			case '13':
				$widget_widths = array_replace( $widget_widths, array(
					'col-md-3 col-sm-6 widget-width__side',
					'col-md-9 col-sm-12 widget-width__loop'
				) );
				break;
			case '31':
				$widget_widths = array_replace( $widget_widths, array(
					'col-md-9 col-sm-12 widget-width__loop',
					'col-md-3 col-sm-6 widget-width__side'
				) );
				break;
			default:
				break;
		}
		?>

			<footer class="footer">
				<?php
				// check if a sidebar is active and in use
				if ( is_active_sidebar( 'footer-widget-wide' ) ) { ?>
					<div class="container">
						<div class="widget-width__container"><?php dynamic_sidebar( 'footer-widget-wide' ); ?></div>
					</div>
				<?php } ?>
				<?php if ( is_active_sidebar( 'footer-widget-1' ) || is_active_sidebar( 'footer-widget-2' ) || is_active_sidebar( 'footer-widget-3' ) || is_active_sidebar( 'footer-widget-4' ) ) { ?>
					<div class="container">
						<?php if ( ! empty( $widget_widths[0] ) ): ?>
							<div class="row">
								<div
									class="<?php echo esc_attr( $widget_widths[0] ); ?>"><?php dynamic_sidebar( 'footer-widget-1' ); ?></div>
								<?php if ( ! empty( $widget_widths[1] ) ): ?>
									<div
										class="<?php echo esc_attr( $widget_widths[1] ); ?>"><?php dynamic_sidebar( 'footer-widget-2' ); ?></div>
								<?php endif; ?>
								<?php if ( ! empty( $widget_widths[2] ) ): ?>
									<div
										class="<?php echo esc_attr( $widget_widths[2] ); ?>"><?php dynamic_sidebar( 'footer-widget-3' ); ?></div>
								<?php endif; ?>
								<?php if ( ! empty( $widget_widths[3] ) ): ?>
									<div
										class="<?php echo esc_attr( $widget_widths[3] ); ?>"><?php dynamic_sidebar( 'footer-widget-4' ); ?></div>
								<?php endif; ?>
							</div>
						<?php endif; ?>
					</div>
				<?php } ?>
			</footer>

		<?php if ( $theme_options['hide_copyright'] == '' ) { ?>
			<div class="copyright">
				<div class="container-fluid">
					<div class="row copyright-row">
						<div class="col-xs-12">
							<div class="copyright-inner">
								<?php if (is_front_page()) {
									echo wp_kses_post( $theme_options['copyright'] );
								} else {
									echo '<div class="copyright-alt">'. wp_kses_post( $theme_options['copyright-2'] ).'</div>';
								}
								?>
							</div>
						</div>
					</div>
				</div>
			</div>
		<?php } ?>
		<?php
	}
endif;
add_action( 'wp_footer', 'beautyagency_footer' );

if ( ! function_exists( 'beautyagency_login_popup' ) ) :
	function beautyagency_login_popup() { ?>
		<!-- Login -->
		<div class="login-popup">
			<?php
			$unique_id = uniqid();
			if ( is_user_logged_in() ) { ?>
				<a href="<?php echo esc_url( wp_logout_url( home_url( '/' ) ) ); ?>" class="login-popup__logout"
				   title="<?php esc_html_e( 'Log Out', 'beautyagency'); ?>">
					<i class="fa fa-sign-out"></i>
				</a>
			<?php } else { ?>
				<a href="#"
				   class="login-popup__toggle login-popup__toggle<?php echo esc_attr( $unique_id ); ?>"
				   title="<?php esc_attr_e( 'Login', 'beautyagency'); ?>"><i class="fa fa-sign-in"></i></a><span class="separator"></span>
				<a href="<?php echo esc_url( wp_registration_url() ); ?>"
				   class="registration__toggle registration__toggle<?php echo esc_attr( $unique_id ); ?>"
				   title="<?php esc_attr_e( 'Register', 'beautyagency'); ?>"><i class="rt-icon icon-user"></i></a><span class="separator"></span>
			<?php } ?>
			<?php if ( is_user_logged_in() ) {
				// logged in
			} else { ?>
				<div
					class="login-popup__popup login-popup__popup<?php echo esc_attr( $unique_id ); ?>">
					<?php wp_login_form( array(
						'echo'        => true,
						'remember'    => true,
						'form_id'     => 'loginform',
						'id_username' => 'user_login',
						'id_password' => 'user_pass',
						'id_remember' => 'rememberme',
						'id_submit'   => 'wp-submit',

						'label_username' => esc_html__( 'Username', 'beautyagency' ),
						'label_password' => esc_html__( 'Password', 'beautyagency' ),
						'label_remember' => esc_html__( 'Remember Me', 'beautyagency' ),
						'label_log_in'   => esc_html__( 'Log In', 'beautyagency' ),
						'value_username' => '',
						'value_remember' => false
					) ); ?>
					<a class="login-popup__popup__forgot-password"
					   href="<?php echo esc_url( wp_lostpassword_url() ); ?>"
					   title="<?php esc_attr_e( 'Forgot your password?', 'beautyagency'); ?>"><?php esc_html_e('Forgot your password?' ,'beautyagency'); ?></a>
				</div><!-- .login-popup -->
			<?php } ?>
		</div>
		<?php
	}
endif;


if ( ! has_action( 'beautyagency_after_header' ) ) {
	// Add the widget area to the bottom
	add_action( 'beautyagency_after_header', 'beautyagency_add_after_header_widget' );
	function beautyagency_add_after_header_widget() {
		if ( is_front_page() ) {
			echo '<div class="sidebar-after-header widget-width__full">';
			if ( ! function_exists( 'dynamic_sidebar' ) || ! dynamic_sidebar( 'Widgets After Header (Full-Width)' ) ) :
			endif;
			echo '</div>';
		}
	}
}

if ( ! function_exists( 'beautyagency_nav_menu_filter' ) ) :
	function beautyagency_nav_menu_filter( $menu ) {
		$menu = str_replace( "\t", "", $menu );
		$menu = str_replace( "\n", "", $menu );
		$menu = str_replace( "\r", "", $menu );

		return $menu;
	}
endif;
add_filter( 'beautyagency_nav_menu_filter', 'beautyagency_nav_menu_filter', 99 );

if ( ! function_exists( 'beautyagency_body_classes' ) ) :
	function beautyagency_body_classes( $classes ) {
		$dropcaps_style = absint( beautyagency_get_customizer_option( 'dropcaps_style' ) );
		$sidebar_class  = absint( beautyagency_get_customizer_option( 'blog_page_sidebar' ) );
		$body_class     = 'beautyagency ';
		$classes[]      = $body_class;
		$classes[]      = 'post_id_' . get_the_ID();
		if ( ! empty( $dropcaps_style ) ) {
			$classes[] = 'dropcaps' . $dropcaps_style;
		}
		if ( ! empty( $sidebar_class ) ) {
			$classes[] = $sidebar_class;
		}

		return $classes;
	}
endif;
add_filter( 'body_class', 'beautyagency_body_classes' );

if ( ! has_action( 'beautyagency_before_content' ) ) :
	function beautyagency_before_content_wrapper() {
		echo '<div class="container">';
	}

	add_action( 'beautyagency_before_content', 'beautyagency_before_content_wrapper' );

	// Add the widget area to the top
	function beautyagency_add_top_widget() {
		if ( is_front_page() ) {

			if ( is_active_sidebar( 'sidebar-top' ) ) {
				echo '<div class="sidebar-before-content widget-width__container">';
				if ( ! function_exists( 'dynamic_sidebar' ) || ! dynamic_sidebar( 'Widgets Before Content' ) ) :
				endif;
				echo '</div>';
			}
		}
	}
	add_action( 'beautyagency_before_content', 'beautyagency_add_top_widget' );
endif;

if ( ! has_action( 'beautyagency_before_loop' ) ) {
	// Add the widget area to the start of the loop
	add_action( 'beautyagency_before_loop', 'beautyagency_add_before_loop_widget' );
	function beautyagency_add_before_loop_widget() {
		if ( is_front_page() ) {

			if ( is_active_sidebar( 'sidebar-before-loop' ) ) {

				$widget_width = beautyagency_sidebar_class( "container" );
				if ( $widget_width == 'sidebar_none' ) {
					$widget_width = 'widget-width__container';
				} else {
					$widget_width = 'widget-width__loop';
				}
				echo '<div class="sidebar-before-loop ' . esc_attr( $widget_width ) . '">';
				if ( ! function_exists( 'dynamic_sidebar' ) || ! dynamic_sidebar( 'Widgets Before Loop (Loop-Width)' ) ) :
				endif;
				echo '</div>';

			}
		}
	}
}

if ( ! has_action( 'beautyagency_after_loop' ) ) {
	// Add the widget area to the end of the loop
	add_action( 'beautyagency_after_loop', 'beautyagency_add_after_loop_widget' );
	function beautyagency_add_after_loop_widget() {
		if ( is_front_page() ) {
			$widget_width = beautyagency_sidebar_class( "container" );
			if ( $widget_width == 'snone' ) {
				$widget_width = 'widget-width__container';
			} else {
				$widget_width = 'widget-width__loop';
			}
			echo '<div class="sidebar-after-loop ' . esc_attr( $widget_width ) . '">';
			if ( ! function_exists( 'dynamic_sidebar' ) || ! dynamic_sidebar( 'Widgets After Loop (Loop-Width)' ) ) :
			endif;
			echo '</div>';
		}
	}
}

if ( ! has_action( 'beautyagency_after_content' ) ) {
	// Add the widget area to the bottom
	add_action( 'beautyagency_after_content', 'beautyagency_add_bottom_widget' );
	function beautyagency_add_bottom_widget() {
		if ( is_front_page() ) {
			echo '<div class="sidebar-after-content widget-width__container">';
			if ( ! function_exists( 'dynamic_sidebar' ) || ! dynamic_sidebar( 'Widgets After Content' ) ) :
			endif;
			echo '</div>';
		}
	}

	function beautyagency_after_content_wrapper() {
		echo '</div>';
	}

	add_action( 'beautyagency_after_content', 'beautyagency_after_content_wrapper' );
}

if ( ! has_action( 'beautyagency_before_footer' ) ) {
	// Add the widget area to the bottom
	add_action( 'beautyagency_before_footer', 'beautyagency_add_before_footer_widget' );
	function beautyagency_add_before_footer_widget() {
		if ( ! is_front_page() ) {
			// check if a sidebar is active and in use
			if ( is_active_sidebar( 'sidebar-before-footer' ) ) {
				echo '<div class="sidebar-before-footer widget-width__full">';
				if ( ! function_exists( 'dynamic_sidebar' ) || ! dynamic_sidebar( 'Widgets Before Footer (Full-Width)' ) ) :
				endif;
				echo '</div>';
			}
		}
	}
}

if( !has_action( 'beautyagency_blog_before_footer' ) ) {
	// Add the widget area before footer on blog page
	add_action('beautyagency_blog_before_footer', 'beautyagency_add_blog_before_footer_widget');
	function beautyagency_add_blog_before_footer_widget() {
		if ( is_home() ) {
			echo '<div class="blog-sidebar-before-footer">';
			if ( !function_exists('dynamic_sidebar') || !dynamic_sidebar('Blog Before Footer (Full-Width)') ) :
			endif;
			echo '</div>';
			echo '</div>';
		}
	}
}

// Add Prettyphoto to default gallery shortcode
function beautyagency_prettyadd ($content) {
	$images = explode("'", $content);
	if( !empty($images[1]) && !(strpos($images[1], "wp-content") === false) ) {
		$post_id = get_the_ID() ? 'slides'.get_the_ID() : 'slides';
		$content = preg_replace( "/<a/", "<a data-gal=\"prettyPhoto[" . $post_id . "]\"", $content, 1 );
	}
	return $content;
}
add_filter('wp_get_attachment_link', 'beautyagency_prettyadd');

if ( ! function_exists( 'beautyagency_add_font_awesome' ) ) :
	// Add Font awesome
	function beautyagency_add_font_awesome() {
		wp_enqueue_style( 'font-awesome', esc_url(get_template_directory_uri()) . '/vendors/font-awesome/css/font-awesome.min.css' );
	}
	endif;
add_action( 'wp_enqueue_scripts', 'beautyagency_add_font_awesome' );

function _action_theme_render_html($post) {
	if ( comments_open() || get_comments_number() ) :
		comments_template();
	endif;
}
add_action('fw_theme_ext_events_after_content', '_action_theme_render_html');


if ( ! function_exists( 'beautyagency_fonts_url' ) ) :
	/**
	 * Register Google fonts.
	 *
	 * @return string Google fonts URL for the theme.
	 */
	function beautyagency_fonts_url() {
		$fonts_url = '';
		$fonts     = array();
		$subsets   = 'latin,latin-ext';

		/* translators: If there are characters in your language that are not supported by Open Sans, translate this to 'off'. Do not translate into your own language. */
		if ( 'off' !== esc_html_x( 'on', 'Open Sans font: on or off', 'beautyagency' ) ) {
			$fonts[] = 'Open Sans:300,300italic,400,400italic,600,600italic,700,700italic,800,800italic';
		}

		/* Theme Fonts */
		$fonts[] = 'Roboto:400,100,100italic,300,300italic,400italic,500,500italic,700,700italic,900,900italic'; /* font-family: 'Roboto', sans-serif; */

		/* Logo Fonts */
		$fonts[] = 'Herr+Von+Muellerhoff'; /* font-family: 'Herr Von Muellerhoff', cursive; */

		/* translators: To add an additional character subset specific to your language, translate this to 'greek', 'cyrillic', 'devanagari' or 'vietnamese'. Do not translate into your own language. */
		$subset = esc_html_x( 'no-subset', 'Add new subset (greek, cyrillic, devanagari, vietnamese)', 'beautyagency' );

		if ( 'cyrillic' == $subset ) {
			$subsets .= ',cyrillic,cyrillic-ext';
		} elseif ( 'greek' == $subset ) {
			$subsets .= ',greek,greek-ext';
		} elseif ( 'devanagari' == $subset ) {
			$subsets .= ',devanagari';
		} elseif ( 'vietnamese' == $subset ) {
			$subsets .= ',vietnamese';
		}

		if ( $fonts ) {
			$fonts_url = add_query_arg( array(
				'family' => urlencode( implode( '%7C', $fonts ) ),
				// Url encode the | (pipe characters) in the href attribute (%7C):
				'subset' => urlencode( $subsets ),
			), '//fonts.googleapis.com/css' );
		}

		return urldecode( $fonts_url );
	}
endif;

if ( ! function_exists( 'beautyagency_scripts' ) ) :
	/**
	 * Enqueue scripts and styles.
	 */
	function beautyagency_scripts() {

		// Add BeautyAgency preloader
		wp_enqueue_style( 'beautyagency-preloader', esc_url(get_template_directory_uri()) . '/css/preloader.css' );

		// Add custom fonts, used in the main stylesheet.
		wp_enqueue_style( 'beautyagency-fonts', esc_url(beautyagency_fonts_url()), array(), null );

		// Add icomoon icon font stylesheet. (rt-icons-2)
		wp_enqueue_style( 'icomoon', esc_url(get_template_directory_uri()) . '/vendors/icomoon/style-icomoon.css', array(), '2' );

		if ( class_exists( 'SC_Class' ) ) {
			// Load prettyPhoto stylesheet.
			wp_enqueue_style( 'beautyagency-apsc', esc_url(get_template_directory_uri()) . '/css/social-counter.css' );
		}

		if ( class_exists( 'booked_plugin' ) ) {
			// Load Booked stylesheet.
			wp_enqueue_style( 'beautyagency-booked', esc_url(get_template_directory_uri()) . '/css/booked.css' );
		}

		// Load Twitter Bootstrap stylesheet.
		wp_enqueue_style( 'bootstrap', esc_url(get_template_directory_uri()) . '/vendors/bootstrap/css/bootstrap.min.css' );

		// Load Bootstrap Select stylesheet.
		wp_enqueue_style( 'bootstrap-select', esc_url(get_template_directory_uri()) . '/vendors/bootstrap-select/css/bootstrap-select.min.css' );

		// Load animate.css stylesheet.
		wp_enqueue_style( 'animate', esc_url(get_template_directory_uri()) . '/vendors/animate/animate.css' );

		// Load Owl Carousel stylesheet.
		wp_enqueue_style( 'owl-carousel', esc_url(get_template_directory_uri()) . '/vendors/owl-carousel/css/owl.carousel.css' );

		// Load prettyPhoto stylesheet.
		wp_enqueue_style( 'prettyPhoto', esc_url(get_template_directory_uri()) . '/vendors/prettyphoto/css/prettyPhoto.css' );

		// Load our main stylesheet.
		wp_enqueue_style( 'beautyagency-style', esc_url(get_stylesheet_uri()) );

		wp_enqueue_script( 'beautyagency-script-ie', esc_url( get_template_directory_uri() ) . '/vendors/html5.js', array( '' ), 'all' );
		wp_script_add_data( 'beautyagency-script-ie', 'conditional', 'lt IE 9' );

		// Load script BX Slider.
		wp_enqueue_script( 'bx-slider', esc_url(get_template_directory_uri()) . '/vendors/jquery.bxslider.js', array( 'jquery' ), "all", true );

		// Load script to make Menu sticky.
		wp_enqueue_script( 'sticky-script', esc_url(get_template_directory_uri()) . '/vendors/jquery.sticky.js', array( 'jquery' ), "all", true );

		// Load script to make Sidebar sticky.
		wp_enqueue_script( 'sticky-kit', esc_url(get_template_directory_uri()) . '/vendors/jquery.sticky-kit.min.js', array( 'jquery' ), "all", true );

		// Load jQuery Appear javascript.
		wp_enqueue_script( 'jquery-appear', esc_url(get_template_directory_uri()) . '/vendors/jquery.appear.js', array( 'jquery' ), "all", true );

		// Load Bootstrap javascripts.
		wp_enqueue_script( 'bootstrap-dropdown', esc_url(get_template_directory_uri()) . '/vendors/bootstrap/js/dropdown.js', array( 'jquery' ), "all", true );
		wp_enqueue_script( 'bootstrap-tabs', esc_url(get_template_directory_uri()) . '/vendors/bootstrap/js/tab.js', array( 'jquery' ), "all", true );
		wp_enqueue_script( 'bootstrap-transition', esc_url(get_template_directory_uri()) . '/vendors/bootstrap/js/transition.js', array( 'jquery' ), "all", true );

		// Load isotope javascript
		wp_enqueue_script( 'isotope-js', esc_url(get_template_directory_uri()) . '/vendors/isotope.pkgd.min.js', array( 'jquery' ), "all", true );

		// Load Bootstrap Select javascript
		wp_enqueue_script( 'bootstrap-select-js', esc_url(get_template_directory_uri()) . '/vendors/bootstrap-select/js/bootstrap-select.min.js', array( 'jquery' ), "all", true );

		// Load Owl Carousel script.
		wp_enqueue_script( 'owl-carousel', esc_url(get_template_directory_uri()) . '/vendors/owl-carousel/js/owl.carousel.min.js', array(), "all", true );

		// Load prettyPhoto script.
		wp_enqueue_script( 'prettyPhoto', esc_url(get_template_directory_uri()) . '/vendors/prettyphoto/js/jquery.prettyPhoto.js', array(), "all", true );

		wp_enqueue_script( 'beautyagency-skip-link-focus-fix', esc_url(get_template_directory_uri()) . '/vendors/skip-link-focus-fix.js', array(), "all", true );

		if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
			wp_enqueue_script( 'comment-reply' );
		}

		if ( is_singular() && wp_attachment_is_image() ) {
			wp_enqueue_script( 'beautyagency-keyboard-image-navigation', esc_url(get_template_directory_uri()) . '/js/keyboard-image-navigation.js', array( 'jquery' ), "all" );
		}

		wp_localize_script( 'beautyagency-script', 'screenReaderText', array(
			'expand'   => '<span class="screen-reader-text">' . esc_html__( 'expand child menu', 'beautyagency' ) . '</span>',
			'collapse' => '<span class="screen-reader-text">' . esc_html__( 'collapse child menu', 'beautyagency' ) . '</span>',
		) );

		wp_enqueue_script( 'iframetracker-script', esc_url(get_template_directory_uri()) . '/js/jquery.iframetracker.js', array( 'jquery' ), "all", true );
		wp_enqueue_script( 'main-script', esc_url(get_template_directory_uri()) . '/js/main.js', array( 'jquery' ), "all", true );
	}
endif;
add_action( 'wp_enqueue_scripts', 'beautyagency_scripts' );

if ( ! function_exists( 'beautyagency_customizer_enqueue' ) ) :
	function beautyagency_customizer_enqueue() {
		// Load Customizer script.
		wp_enqueue_script( 'beautyagency-customizer-script', esc_url(get_template_directory_uri()) . '/js/customizer.js', array(), "all", true );
	}
endif;
add_action( 'admin_print_styles', 'beautyagency_customizer_enqueue' );


if ( ! function_exists( 'beautyagency_comment_callback' ) ) :
	/**
	 * Callback for customizing comments template.
	 *
	 * @param array $comment Comment to display.
	 * @param int $depth Depth of the menu.
	 * @param array $args wp_nav_menu() arguments.
	 */
	function beautyagency_comment_callback( $comment, $args, $depth ) {
		$GLOBALS['comment'] = $comment;
		extract( $args, EXTR_SKIP );

		if ( 'div' == $args['style'] ) {
			$tag       = 'div';
			$add_below = 'comment';
		} else {
			$tag       = 'li';
			$add_below = 'div-comment';
		}
		?>
		<<?php echo esc_attr( $tag ); ?> <?php comment_class( ! empty( $args['has_children'] ) ? 'parent' : '' ); ?> id="comment-<?php comment_ID(); ?>">
		<?php if ( 'div' != $args['style'] ) : ?>
			<div class="comment-author comment-author-avatar">
				<?php if ( 0 != $args['avatar_size'] ) {
					echo get_avatar( $comment, $args['avatar_size'] );
				} ?>
			</div>
			<article id="div-comment-<?php comment_ID(); ?>" class="comment-body">
		<?php endif; ?>
		<div class="comment-meta">
			<div class="comment-author vcard">
				<?php printf( wp_kses( __( '%s <span class="says">says:</span>', 'beautyagency' ), beautyagency_kses_init() ), get_comment_author_link() ); ?>
			</div>
			<?php if ( '0' == $comment->comment_approved ) : ?>
				<em class="comment-awaiting-moderation"><?php esc_html_e( 'Your comment is awaiting moderation.', 'beautyagency' ) ?></em>
				<br/>
			<?php endif; ?>

			<div class="comment-metadata">
				<?php /* translators: 1: date, 2: time */
				printf( __( '<time datetime="%1$s">%2$s</time>', 'beautyagency' ), esc_attr( get_comment_date( 'c' ) ), get_comment_date( 'd.m.Y - G:i' ) ); ?>
				<?php edit_comment_link( esc_html__( '(Edit)', 'beautyagency' ), '&nbsp;&nbsp;', '' ); ?>
			</div>
		</div><!-- .comment-meta -->

		<div class="comment-content">
			<?php comment_text( get_comment_id(), array_merge( $args, array(
				'add_below' => $add_below,
				'depth'     => $depth,
				'max_depth' => $args['max_depth']
			) ) ); ?>
		</div>

		<?php
		comment_reply_link( array_merge( $args, array(
			'add_below'  => $add_below,
			'depth'      => $depth,
			'max_depth'  => $args['max_depth'],
			'before'     => '<span class="reply">',
			'after'      => '</span>',
			'reply_text' => '',
		) ) );
		?>

		<?php if ( 'div' != $args['style'] ) : ?>
			</article>
		<?php endif; ?>
		<?php
	}
endif;

if ( ! function_exists( 'beautyagency_title_tag' ) ) :
	function beautyagency_title_tag() {
		// title tag
		add_theme_support( 'title-tag' );
	}
endif;
add_action( 'after_setup_theme', 'beautyagency_title_tag' );

if ( ! function_exists( 'beautyagency_excerpt_length' ) ) :
	/**
	 * Cut excerpt length to 55 words.
	 */
	function beautyagency_excerpt_length( $length ) {
		return 55;
	}
endif;
add_filter( 'excerpt_length', 'beautyagency_excerpt_length', 999 );

if ( ! function_exists( 'beautyagency_init_excerpt_on_publish' ) ) :
	/**
	 * Auto Init excerpt on wp_insert_post.
	 */
	function beautyagency_init_excerpt_on_publish( $data, $postarr ) {

		$data['post_excerpt'] = wp_trim_words( strip_tags( trim( strip_shortcodes( $data['post_content'] ) ) ), 55, "&hellip;" ); // 55 words by default

		return $data;
	}
endif;
add_filter( 'wp_insert_post_data', 'beautyagency_init_excerpt_on_publish', '99', 2 );


/**
 * Replace search form button
 */
function beautyagency_search_form( $form ) {

	$form = '<form role="search" method="get" class="search-form" action="' . esc_url( home_url( '/' ) ) . '">
				<label>
					<span class="screen-reader-text">' . esc_html__( 'Search for:', 'beautyagency' ) . '</span>
					<input type="search" class="search-field" placeholder="' . esc_attr__( 'Type keyword here', 'beautyagency' ) . '" value="' . get_search_query() . '" name="s" title="' . esc_attr__( 'Search for:', 'beautyagency' ) . '" />
				</label>
				<label class="submit-button-icon"><input type="submit" class="search-submit" value="' . esc_attr__( 'Search', 'beautyagency' ) . '" /></label>
	</form>';

	return $form;
}

add_filter( 'get_search_form', 'beautyagency_search_form' );


if ( ! function_exists( 'beautyagency_build_gallery_content' ) ) :

	/**
	 * Replace gallery shortcode with owl carousel
	 */

	function beautyagency_build_gallery_content( $attrs ) {
		global $post;
		static $instance = 0;
		$instance ++;

		/*
		Limiting what the user can do by
		locking down most short code options.
		*/
		extract( shortcode_atts( array(
			'id'      => $post->ID,
			'include' => '',
			'exclude' => ''
		), $attrs ) );

		$id = intval( 'id' );

		if ( ! empty( $include ) ) {
			$params       = array(
				'include'        => $include,
				'post_status'    => 'inherit',
				'post_type'      => 'attachment',
				'post_mime_type' => 'image',
				'order'          => 'ASC',
				'orderby'        => 'menu_order ID'
			);
			$_attachments = get_posts( $params );
			$attachments  = array();
			foreach ( $_attachments as $key => $val ) {
				$attachments[ $val->ID ] = $_attachments[ $key ];
			}
		} elseif ( ! empty( $exclude ) ) {
			$params      = array(
				'post_parent'    => $id,
				'exclude'        => $exclude,
				'post_status'    => 'inherit',
				'post_type'      => 'attachment',
				'post_mime_type' => 'image',
				'order'          => 'ASC',
				'orderby'        => 'menu_order ID'
			);
			$attachments = get_children( $params );
		} else {
			$params      = array(
				'post_parent'    => $id,
				'post_status'    => 'inherit',
				'post_type'      => 'attachment',
				'post_mime_type' => 'image',
				'order'          => 'ASC',
				'orderby'        => 'menu_order ID'
			);
			$attachments = get_children( $params );
		}

		if ( empty( $attachments ) ) {
			return '';
		}

		$selector = "gallery-{$instance}";

		$gallery_div = sprintf( "<div id='%s' class='owl-gallery owl-carousel gallery galleryid-%d gallery-columns-1 gallery-size-full'>", $selector, $id );
		$output      = $gallery_div;


		foreach ( $attachments as $id => $attachment ) {
			/*
			Use wp_get_attachment_link to return a photo + link
			to attachment page or image
			http://codex.wordpress.org/Function_Reference/wp_get_attachment_link
			*/
			$img = wp_get_attachment_image( $id, 'full', false );

			$caption = '';

			/*
			Set the caption string if there is one.
			*/

			if ( ! empty( $captiontag ) && trim( $attachment->post_excerpt ) ) {
				$caption = sprintf( "<figcaption class='wp-caption-text gallery-caption'><div>%s</div></figcaption>", wptexturize( $attachment->post_excerpt ) );
			}

			/*
			Set the output for each slide.
			*/
			$output .= sprintf( "<figure class='gallery-icon'>%s\n\t%s</figure>", $img, $caption );
		}
		$output .= '</div>';

		return $output;
	}
endif;

if ( ! function_exists( 'beautyagency_gallery_shortcode' ) ) :
	function beautyagency_gallery_shortcode( $output = '', $attrs ) {
		$return = $output;

		if ( ! empty( $attrs ) && ( ! empty( $attrs['type'] ) || ! empty( $attrs['columns'] ) ) ) {

		} else {
			# Gallery function that returns new markup.
			$gallery = beautyagency_build_gallery_content( $attrs );

			if ( ! empty( $gallery ) ) {
				$return = $gallery;
			}
		}

		return $return;
	}
endif;
add_filter( 'post_gallery', 'beautyagency_gallery_shortcode', 10, 2 );

// Add span tag to categories number
add_filter( 'wp_list_categories', 'beautyagency_add_span_cat_count' );
function beautyagency_add_span_cat_count( $links ) {
	$links = str_replace( '</a> (', '</a><span class="cat-count">(', $links );
	$links = str_replace( ')', ')</span>', $links );

	return $links;
}



/**
 * Change comment form inputs order.
 *
 */
add_filter( 'comment_form_fields', 'beautyagency_reorder_comment_fields' );

if ( ! function_exists( 'beautyagency_reorder_comment_fields' ) ) :
	function beautyagency_reorder_comment_fields( $fields ) {
		//die(print_r( $fields )); // view what have field
		$new_fields = array(); // fields in new order
		$myorder    = array( 'author', 'email', 'comment' );

		foreach ( $myorder as $key ) {
			$new_fields[ $key ] = $fields[ $key ];
			unset( $fields[ $key ] );
		}
		if ( $fields ) {
			foreach ( $fields as $key => $val ) {
				print_r( $key );
			}
		}

		return $new_fields;
	}
endif;

if ( ! function_exists( 'beautyagency_iweb_modest_youtube_player' ) ) :

	/**
	 * Filter a few parameters into YouTube oEmbed requests
	 *
	 * @link http://goo.gl/yl5D3
	 */
	function beautyagency_iweb_modest_youtube_player( $html, $url, $args ) {
		$html = str_replace( '?feature=oembed', '?feature=oembed&amp;modestbranding=0&amp;showinfo=0&amp;rel=0', $html );
		$html = str_replace( 'frameborder="0"', '', $html );

		return $html;
	}
endif;
add_filter( 'oembed_result', 'beautyagency_iweb_modest_youtube_player', 10, 3 );


if ( ! function_exists( 'beautyagency_kses_list' ) ) :
	/**
	 * Initiates allowed tags array for wp_kses functions
	 */
	function beautyagency_kses_init() {

		$allowed_atts = array(
			'align'       => true,
			'aria-hidden' => true,
			'class'       => true,
			'clear'       => true,
			'dir'         => true,
			'id'          => true,
			'lang'        => true,
			'name'        => true,
			'style'       => true,
			'title'       => true,
			'xml:lang'    => true,
		);

		$kses_list = array(
			'address'    => $allowed_atts,
			'a'          => array(
				                'href'         => true,
				                'rel'          => true,
				                'rev'          => true,
				                'target'       => true,
				                'data-content' => true,
			                ) + $allowed_atts,
			'abbr'       => $allowed_atts,
			'acronym'    => $allowed_atts,
			'area'       => array(
				                'alt'    => true,
				                'coords' => true,
				                'href'   => true,
				                'nohref' => true,
				                'shape'  => true,
				                'target' => true,
			                ) + $allowed_atts,
			'aside'      => $allowed_atts,
			'b'          => $allowed_atts,
			'bdo'        => $allowed_atts,
			'big'        => $allowed_atts,
			'blockquote' => array(
				                'cite' => true,
			                ) + $allowed_atts,
			'br'         => $allowed_atts,
			'cite'       => $allowed_atts,
			'code'       => $allowed_atts,
			'del'        => array(
				                'datetime' => true,
			                ) + $allowed_atts,
			'dd'         => $allowed_atts,
			'dfn'        => $allowed_atts,
			'details'    => array(
				                'open' => true,
			                ) + $allowed_atts,
			'div'        => $allowed_atts,
			'dl'         => $allowed_atts,
			'dt'         => $allowed_atts,
			'em'         => $allowed_atts,
			'h1'         => $allowed_atts,
			'h2'         => $allowed_atts,
			'h3'         => $allowed_atts,
			'h4'         => $allowed_atts,
			'h5'         => $allowed_atts,
			'h6'         => $allowed_atts,
			'hr'         => array(
				                'noshade' => true,
				                'size'    => true,
				                'width'   => true,
			                ) + $allowed_atts,
			'i'          => $allowed_atts,
			'img'        => array(
				                'alt'      => true,
				                'border'   => true,
				                'height'   => true,
				                'hspace'   => true,
				                'longdesc' => true,
				                'vspace'   => true,
				                'src'      => true,
				                'usemap'   => true,
				                'width'    => true,
			                ) + $allowed_atts,
			'ins'        => array(
				                'datetime' => true,
				                'cite'     => true,
			                ) + $allowed_atts,
			'kbd'        => $allowed_atts,
			'li'         => array(
				                'value' => true,
			                ) + $allowed_atts,
			'map'        => $allowed_atts,
			'mark'       => $allowed_atts,
			'p'          => $allowed_atts,
			'pre'        => array(
				                'width' => true,
			                ) + $allowed_atts,
			'q'          => array(
				                'cite' => true,
			                ) + $allowed_atts,
			's'          => $allowed_atts,
			'samp'       => $allowed_atts,
			'span'       => $allowed_atts,
			'small'      => $allowed_atts,
			'strike'     => $allowed_atts,
			'strong'     => $allowed_atts,
			'sub'        => $allowed_atts,
			'summary'    => $allowed_atts,
			'sup'        => $allowed_atts,
			'time'       => array(
				                'datetime' => true,
			                ) + $allowed_atts,
			'tt'         => $allowed_atts,
			'u'          => $allowed_atts,
			'ul'         => array(
				                'type' => true,
			                ) + $allowed_atts,
			'ol'         => array(
				                'start'    => true,
				                'type'     => true,
				                'reversed' => true,
			                ) + $allowed_atts,
			'label'      => array(
				                'for'         => true,
				                'type'        => true,
				                'style'        => true,
				                'value'       => true,
				                'required'    => true,
				                'placeholder' => true,
			                ) + $allowed_atts,
			'form'      => array(
				               'action'                 => true,
				               'method'                 => true,
				               'data-fw-ext-forms-type' => true,
				               'data-fw-form-id' => true,
			               ) + $allowed_atts,
			'input'      => array(
				                'for'         => true,
				                'type'        => true,
				                'value'       => true,
				                'required'    => true,
				                'placeholder' => true,
				                'data-constraint' => true,
			                ) + $allowed_atts,
			'textarea'   => array(
				                'for'         => true,
				                'type'        => true,
				                'value'       => true,
				                'required'    => true,
				                'placeholder' => true,
			                ) + $allowed_atts,
			'var'        => $allowed_atts,
		);

		return $kses_list;
	}
endif;

// Unyson CSS
if ( ! function_exists( 'beautyagency_unyson_scripts' ) ) :
	function beautyagency_unyson_scripts() {
		// check if unyson exist
		if ( class_exists( '_Fw' ) ) :
			// Load unyson stylesheet.
			wp_enqueue_style( 'beautyagency-unyson', esc_url(get_template_directory_uri()) . '/css/unyson.css' );
			if ( is_rtl() ) {
				wp_enqueue_style( 'beautyagency-unyson-rtl', esc_url(get_template_directory_uri()) . '/css/unyson-rtl.css' );
			}
		endif;
	}
endif;
add_action( 'wp_enqueue_scripts', 'beautyagency_unyson_scripts', 9999 );

/**
 * Check use Unyson PageBuilder
 */
if ( ! function_exists( 'beautyagency_check_use_unyson' ) ) :
	function beautyagency_check_use_unyson() {
		$check_use_unyson = false;
		if ( defined( "FW" ) && fw_ext( 'page-builder' ) && fw_ext_page_builder_is_builder_post( get_the_ID() ) ) {
			$check_use_unyson = true;
		}

		return $check_use_unyson;
	}
endif;

/**
 * Check use Unyson Portfolio
 */
if ( ! function_exists( 'beautyagency_check_use_unyson_portfolio' ) ) :
	function beautyagency_check_use_unyson_portfolio() {
		$check_use_unyson_portfolio = false;
		if ( defined( "FW" ) && fw_ext( 'portfolio' ) && fw_ext_portfolio_get_gallery_images( get_the_ID() ) ) {
			$check_use_unyson_portfolio = true;
		}

		return $check_use_unyson_portfolio;
	}
endif;

/**
 * Add Unyson Events Comment Form
 */
if( post_type_exists('fw-event') ) {
	add_post_type_support( 'fw-event', 'comments' );
}


/**
 * Implement the Custom Header feature.
 */
require get_template_directory() . '/inc/custom-header.php';

/**
 * Custom template tags for this theme.
 */
require get_template_directory() . '/inc/template-tags.php';

/**
 * Custom color theme.
 */
require get_template_directory() . '/inc/color-scheme.php';

if ( ! defined( 'FW' ) ) {
	function beautyagency_add_color_style() {
		wp_enqueue_style( 'beautyagency-color-style', esc_url( get_template_directory_uri() ) . '/color-style.css' );
	}

	add_action( 'wp_enqueue_scripts', 'beautyagency_add_color_style' );
}

/**
 * Customizer additions.
 */
require get_template_directory() . '/inc/customizer.php';

/**
 * Custom widgets.
 */
require get_template_directory() . '/inc/widgets.php';

include_once( get_template_directory() . '/inc/header_styles.php' );
include_once( get_template_directory() . '/inc/mods.php' );
