<?php
/**
 * Template tags related to models profiles for theme authors to use in their theme templates.
 *
 * @package    CustomContentModels
 * @subpackage Includes
 * @author     Justin Tadlock <justin@justintadlock.com>
 * @copyright  Copyright (c) 2013-2015, Justin Tadlock
 * @license    http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 */

/**
 * Makes sure the post ID is an absolute integer if passed in.  Else, returns the result
 * of `get_the_ID()`.
 *
 * @since  1.0.0
 * @access public
 * @param  int     $post_id
 * @return int
 */
function ccm_get_profile_id( $post_id = '' ) {

	return $post_id ? absint( $post_id ) : get_the_ID();
}

/**
 * Checks if viewing a single profile.
 *
 * @since  1.0.0
 * @access public
 * @param  mixed  $post
 * @return bool
 */
function ccm_is_single_profile( $post = '' ) {

	$is_single = is_singular( ccm_get_profile_post_type() );

	if ( $is_single && $post )
		$is_single = is_single( $post );

	return apply_filters( 'ccm_is_single_profile', $is_single, $post );
}

/**
 * Checks if viewing the profile archive.
 *
 * @since  1.0.0
 * @access public
 * @return bool
 */
function ccm_is_profile_archive() {

	return apply_filters( 'ccm_is_profile_archive', is_post_type_archive( ccm_get_profile_post_type() ) && ! ccm_is_author() );
}

/**
 * Checks if the current post is a profile.
 *
 * @since  1.0.0
 * @access public
 * @param  int     $post_id
 * @return bool
 */
function ccm_is_profile( $post_id = '' ) {

	$post_id = ccm_get_profile_id( $post_id );

	return apply_filters( 'ccm_is_profile', ccm_get_profile_post_type() === get_post_type( $post_id ), $post_id );
}

/**
 * Conditional check to see if a profile has the "sticky" type.
 *
 * @since  1.0.0
 * @access public
 * @param  int    $profile_id
 * @return bool
 */
function ccm_is_profile_sticky( $profile_id = 0 ) {
	$profile_id = ccm_get_profile_id( $profile_id );

	return apply_filters( 'ccm_is_profile_sticky', in_array( $profile_id, ccm_get_sticky_profiles() ), $profile_id );
}

/**
 * Outputs the profile URL.
 *
 * @since  1.0.0
 * @access public
 * @param  int     $post_id
 * @return void
 */
function ccm_profile_url( $post_id = '' ) {

	$url = ccm_get_profile_url( $post_id );

	echo $url ? esc_url( $url ) : '';
}

/**
 * Returns the profile URL.
 *
 * @since  1.0.0
 * @access public
 * @param  int    $post_id
 * @return string
 */
function ccm_get_profile_url( $post_id = '' ) {

	$post_id = ccm_get_profile_id( $post_id );

	return apply_filters( 'ccm_get_profile_url', ccm_get_profile_meta( $post_id, 'url' ), $post_id );
}

/**
 * Displays the profile address.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return void
 */
function ccm_profile_address( $args = array() ) {
	echo ccm_get_profile_address( $args );
}

/**
 * Returns the profile address.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return string
 */
function ccm_get_profile_address( $args = array() ) {

	$html = '';

	$defaults = array(
		'post_id' => ccm_get_profile_id(),
		'text'    => '%s',
		'before'  => '',
		'after'   => '',
		'wrap'    => '<span %s>%s</span>',
	);

	$args = wp_parse_args( $args, $defaults );

	$address = ccm_get_profile_meta( $args['post_id'], 'address' );

	if ( $address ) {

		$text = sprintf( $args['text'], sprintf( '<span class="profile-data">%s</span>', $address ) );

		$html .= $args['before'];
		$html .= sprintf( $args['wrap'], 'class="profile-address"', $text );
		$html .= $args['after'];
	}

	return apply_filters( 'ccm_get_profile_address', $html, $args['post_id'] );
}

/**
 * Prints the profile phone.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return void
 */
function ccm_profile_phone( $args = array() ) {
	echo ccm_get_profile_phone( $args );
}

/**
 * Returns the profile phone.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return string
 */
function ccm_get_profile_phone( $args = array() ) {

	$html = '';

	$defaults = array(
		'post_id' => ccm_get_profile_id(),
		'text'    => '%s',
		'before'  => '',
		'after'   => '',
		'wrap'    => '<span %s>%s</span>',
	);

	$args = wp_parse_args( $args, $defaults );

	$phone = ccm_get_profile_meta( $args['post_id'], 'phone' );

	if ( $phone ) {

		$text = sprintf( $args['text'], sprintf( '<span class="profile-data">%s</span>', $phone ) );

		$html .= $args['before'];
		$html .= sprintf( $args['wrap'], 'class="profile-phone"', $text );
		$html .= $args['after'];
	}

	return apply_filters( 'ccm_get_profile_phone', $html, $args['post_id'] );
}

/**
 * Prints the profile email.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return void
 */
function ccm_profile_email( $args = array() ) {
	echo ccm_get_profile_email( $args );
}

/**
 * Returns the profile email.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return string
 */
function ccm_get_profile_email( $args = array() ) {

	$html = '';

	$defaults = array(
		'post_id' => ccm_get_profile_id(),
		'text'    => '%s',
		'before'  => '',
		'after'   => '',
		'wrap'    => '<span class="profile-email">%s</span>',
	);

	$args = wp_parse_args( $args, $defaults );

	$email = ccm_get_profile_meta( $args['post_id'], 'email' );

	if ( $email ) {

		$html .= $args['before'];
		$html .= sprintf( $args['wrap'], $email );
		$html .= $args['after'];
	}

	return apply_filters( 'ccm_get_profile_email', $html, $args['post_id'] );
}

/**
 * Prints the profile height.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return void
 */
function ccm_profile_height( $args = array() ) {
	echo ccm_get_profile_height( $args );
}

/**
 * Returns the profile height.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return string
 */
function ccm_get_profile_height( $args = array() ) {

	$html = '';

	$defaults = array(
		'post_id' => ccm_get_profile_id(),
		'text'    => '%s',
		'format'  => '',
		'before'  => '',
		'after'   => '',
		'wrap'    => '<span %s>%s</span>',
	);

	$args = wp_parse_args( $args, $defaults );

	$height = ccm_get_profile_meta( $args['post_id'], 'height' );

	if ( $height ) {

		$text = sprintf( '<div class="profile-data">%s</div>', $height );

		$text = sprintf( $args['text'], str_replace( '{ccm_break}', '<br>', $text ) );

		$html .= $args['before'];
		$html .= sprintf( $args['wrap'], 'class="profile-height"', $text );
		$html .= $args['after'];
	}

	return apply_filters( 'ccm_get_profile_height', $html, $args['post_id'] );
}

/**
 * Prints the profile bust.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return void
 */
function ccm_profile_bust( $args = array() ) {
	echo ccm_get_profile_bust( $args );
}

/**
 * Returns the profile bust.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return string
 */
function ccm_get_profile_bust( $args = array() ) {

	$html = '';

	$defaults = array(
		'post_id' => ccm_get_profile_id(),
		'text'    => '%s',
		'format'  => '',
		'before'  => '',
		'after'   => '',
		'wrap'    => '<span %s>%s</span>',
	);

	$args = wp_parse_args( $args, $defaults );

	$bust = ccm_get_profile_meta( $args['post_id'], 'bust' );

	if ( $bust ) {

		$text = sprintf( '<div class="profile-data">%s</div>', $bust );

		$text = sprintf( $args['text'], str_replace( '{ccm_break}', '<br>', $text ) );

		$html .= $args['before'];
		$html .= sprintf( $args['wrap'], 'class="profile-bust"', $text );
		$html .= $args['after'];
	}

	return apply_filters( 'ccm_get_profile_bust', $html, $args['post_id'] );
}

/**
 * Prints the profile waist.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return void
 */
function ccm_profile_waist( $args = array() ) {
	echo ccm_get_profile_waist( $args );
}

/**
 * Returns the profile waist.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return string
 */
function ccm_get_profile_waist( $args = array() ) {

    $html = '';

    $defaults = array(
        'post_id' => ccm_get_profile_id(),
        'text'    => '%s',
        'format'  => '',
        'before'  => '',
        'after'   => '',
        'wrap'    => '<span %s>%s</span>',
    );

    $args = wp_parse_args( $args, $defaults );

    $waist = ccm_get_profile_meta( $args['post_id'], 'waist' );

    if ( $waist ) {

        $text = sprintf( '<div class="profile-data">%s</div>', $waist );

        $text = sprintf( $args['text'], str_replace( '{ccm_break}', '<br>', $text ) );

        $html .= $args['before'];
        $html .= sprintf( $args['wrap'], 'class="profile-waist"', $text );
        $html .= $args['after'];
    }

    return apply_filters( 'ccm_get_profile_waist', $html, $args['post_id'] );
}

/**
 * Prints the profile hips.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return void
 */
function ccm_profile_hips( $args = array() ) {
	echo ccm_get_profile_hips( $args );
}

/**
 * Returns the profile hips.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return string
 */
function ccm_get_profile_hips( $args = array() ) {

	$html = '';

	$defaults = array(
		'post_id' => ccm_get_profile_id(),
		'text'    => '%s',
		'format'  => '',
		'before'  => '',
		'after'   => '',
		'wrap'    => '<span %s>%s</span>',
	);

	$args = wp_parse_args( $args, $defaults );

	$hips = ccm_get_profile_meta( $args['post_id'], 'hips' );

	if ( $hips ) {

		$text = sprintf( '<div class="profile-data">%s</div>', $hips );

		$text = sprintf( $args['text'], str_replace( '{ccm_break}', '<br>', $text ) );

		$html .= $args['before'];
		$html .= sprintf( $args['wrap'], 'class="profile-hips"', $text );
		$html .= $args['after'];
	}

	return apply_filters( 'ccm_get_profile_hips', $html, $args['post_id'] );
}

/**
 * Prints the profile shoe.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return void
 */
function ccm_profile_shoe( $args = array() ) {
	echo ccm_get_profile_shoe( $args );
}

/**
 * Returns the profile shoe.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return string
 */
function ccm_get_profile_shoe( $args = array() ) {

	$html = '';

	$defaults = array(
		'post_id' => ccm_get_profile_id(),
		'text'    => '%s',
		'format'  => '',
		'before'  => '',
		'after'   => '',
		'wrap'    => '<span %s>%s</span>',
	);

	$args = wp_parse_args( $args, $defaults );

	$shoe = ccm_get_profile_meta( $args['post_id'], 'shoe' );

	if ( $shoe ) {

		$text = sprintf( '<div class="profile-data">%s</div>', $shoe );

		$text = sprintf( $args['text'], str_replace( '{ccm_break}', '<br>', $text ) );

		$html .= $args['before'];
		$html .= sprintf( $args['wrap'], 'class="profile-shoe"', $text );
		$html .= $args['after'];
	}

	return apply_filters( 'ccm_get_profile_shoe', $html, $args['post_id'] );
}

/**
 * Prints the profile eyes.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return void
 */
function ccm_profile_eyes( $args = array() ) {
	echo ccm_get_profile_eyes( $args );
}

/**
 * Returns the profile eyes.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return string
 */
function ccm_get_profile_eyes( $args = array() ) {

	$html = '';

	$defaults = array(
		'post_id' => ccm_get_profile_id(),
		'text'    => '%s',
		'format'  => '',
		'before'  => '',
		'after'   => '',
		'wrap'    => '<span %s>%s</span>',
	);

	$args = wp_parse_args( $args, $defaults );

	$eyes = ccm_get_profile_meta( $args['post_id'], 'eyes' );

	if ( $eyes ) {

		$text = sprintf( '<div class="profile-data">%s</div>', $eyes );

		$text = sprintf( $args['text'], str_replace( '{ccm_break}', '<br>', $text ) );

		$html .= $args['before'];
		$html .= sprintf( $args['wrap'], 'class="profile-eyes"', $text );
		$html .= $args['after'];
	}

	return apply_filters( 'ccm_get_profile_eyes', $html, $args['post_id'] );
}

/**
 * Displays the profile facebook link.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return void
 */
function ccm_profile_facebook( $args = array() ) {
	echo ccm_get_profile_facebook( $args );
}

/**
 * Returns the profile facebook link.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return string
 */
function ccm_get_profile_facebook( $args = array() ) {

	$html = '';

	$defaults = array(
		'post_id' => ccm_get_profile_id(),
		'text'    => '%s',
		'before'  => '',
		'after'   => '',
		'wrap'    => '<a %s><i class="fa fa-facebook"></i></a>',
	);

	$args = wp_parse_args( $args, $defaults );

	$facebook = ccm_get_profile_meta( $args['post_id'], 'facebook' );

	if ( $facebook ) {

		$text = sprintf( $args['text'], $facebook );
		$attr = sprintf( 'class="profile-facebook" href="%s"', esc_url( $facebook ) );

		$html .= $args['before'];
		$html .= sprintf( $args['wrap'], $attr, $text );
		$html .= $args['after'];
	}

	return apply_filters( 'ccm_get_profile_facebook', $html, $args['post_id'] );
}

/**
 * Displays the profile twitter link.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return void
 */
function ccm_profile_twitter( $args = array() ) {
	echo ccm_get_profile_twitter( $args );
}

/**
 * Returns the profile twitter link.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return string
 */
function ccm_get_profile_twitter( $args = array() ) {

	$html = '';

	$defaults = array(
		'post_id' => ccm_get_profile_id(),
		'text'    => '%s',
		'before'  => '',
		'after'   => '',
		'wrap'    => '<a %s><i class="fa fa-twitter"></i></a>',
	);

	$args = wp_parse_args( $args, $defaults );

	$twitter = ccm_get_profile_meta( $args['post_id'], 'twitter' );

	if ( $twitter ) {

		$text = sprintf( $args['text'], $twitter );
		$attr = sprintf( 'class="profile-twitter" href="%s"', esc_url( $twitter ) );

		$html .= $args['before'];
		$html .= sprintf( $args['wrap'], $attr, $text );
		$html .= $args['after'];
	}

	return apply_filters( 'ccm_get_profile_twitter', $html, $args['post_id'] );
}

/**
 * Displays the profile googleplus link.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return void
 */
function ccm_profile_googleplus( $args = array() ) {
	echo ccm_get_profile_googleplus( $args );
}

/**
 * Returns the profile googleplus link.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return string
 */
function ccm_get_profile_googleplus( $args = array() ) {

	$html = '';

	$defaults = array(
		'post_id' => ccm_get_profile_id(),
		'text'    => '%s',
		'before'  => '',
		'after'   => '',
		'wrap'    => '<a %s><i class="fa fa-plus"></i></a>',
	);

	$args = wp_parse_args( $args, $defaults );

	$googleplus = ccm_get_profile_meta( $args['post_id'], 'googleplus' );

	if ( $googleplus ) {

		$text = sprintf( $args['text'], $googleplus );
		$attr = sprintf( 'class="profile-googleplus" href="%s"', esc_url( $googleplus ) );

		$html .= $args['before'];
		$html .= sprintf( $args['wrap'], $attr, $text );
		$html .= $args['after'];
	}

	return apply_filters( 'ccm_get_profile_googleplus', $html, $args['post_id'] );
}

/**
 * Displays the profile linkedin link.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return void
 */
function ccm_profile_linkedin( $args = array() ) {
	echo ccm_get_profile_linkedin( $args );
}

/**
 * Returns the profile linkedin link.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return string
 */
function ccm_get_profile_linkedin( $args = array() ) {

	$html = '';

	$defaults = array(
		'post_id' => ccm_get_profile_id(),
		'text'    => '%s',
		'before'  => '',
		'after'   => '',
		'wrap'    => '<a %s><i class="fa fa-linkedin"></i></a>',
	);

	$args = wp_parse_args( $args, $defaults );

	$linkedin = ccm_get_profile_meta( $args['post_id'], 'linkedin' );

	if ( $linkedin ) {

		$text = sprintf( $args['text'], $linkedin );
		$attr = sprintf( 'class="profile-linkedin" href="%s"', esc_url( $linkedin ) );

		$html .= $args['before'];
		$html .= sprintf( $args['wrap'], $attr, $text );
		$html .= $args['after'];
	}

	return apply_filters( 'ccm_get_profile_linkedin', $html, $args['post_id'] );
}

/**
 * Displays the profile pinterest link.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return void
 */
function ccm_profile_pinterest( $args = array() ) {
	echo ccm_get_profile_pinterest( $args );
}

/**
 * Returns the profile pinterest link.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return string
 */
function ccm_get_profile_pinterest( $args = array() ) {

	$html = '';

	$defaults = array(
		'post_id' => ccm_get_profile_id(),
		'text'    => '%s',
		'before'  => '',
		'after'   => '',
		'wrap'    => '<a %s><i class="fa fa-pinterest"></i></a>',
	);

	$args = wp_parse_args( $args, $defaults );

	$pinterest = ccm_get_profile_meta( $args['post_id'], 'pinterest' );

	if ( $pinterest ) {

		$text = sprintf( $args['text'], $pinterest );
		$attr = sprintf( 'class="profile-pinterest" href="%s"', esc_url( $pinterest ) );

		$html .= $args['before'];
		$html .= sprintf( $args['wrap'], $attr, $text );
		$html .= $args['after'];
	}

	return apply_filters( 'ccm_get_profile_pinterest', $html, $args['post_id'] );
}

/**
 * Displays the profile instagram link.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return void
 */
function ccm_profile_instagram( $args = array() ) {
	echo ccm_get_profile_instagram( $args );
}

/**
 * Returns the profile instagram link.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return string
 */
function ccm_get_profile_instagram( $args = array() ) {

	$html = '';

	$defaults = array(
		'post_id' => ccm_get_profile_id(),
		'text'    => '%s',
		'before'  => '',
		'after'   => '',
		'wrap'    => '<a %s><i class="fa fa-instagram"></i></a>',
	);

	$args = wp_parse_args( $args, $defaults );

	$instagram = ccm_get_profile_meta( $args['post_id'], 'instagram' );

	if ( $instagram ) {

		$text = sprintf( $args['text'], $instagram );
		$attr = sprintf( 'class="profile-instagram" href="%s"', esc_url( $instagram ) );

		$html .= $args['before'];
		$html .= sprintf( $args['wrap'], $attr, $text );
		$html .= $args['after'];
	}

	return apply_filters( 'ccm_get_profile_instagram', $html, $args['post_id'] );
}

/**
 * Displays the profile contact form.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return void
 */
function ccm_profile_contact_form( $args = array() ) {
	echo ccm_get_profile_contact_form( $args );
}

/**
 * Returns the profile contact form.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return string
 */
function ccm_get_profile_contact_form( $args = array() ) {

	$html = '';

	$defaults = array(
		'post_id' => ccm_get_profile_id(),
		'text'    => '%s',
		'before'  => '',
		'after'   => '',
		'wrap'    => '<span %s>%s</span>',
	);

	$args = wp_parse_args( $args, $defaults );

	$contact_form = ccm_get_profile_meta( $args['post_id'], 'contact_form' );

	//if ( $contact_form ) {

		$html .= $args['before'];

		$name = '';
		$email = '';
		$comments = '';
		$nameError = '';
		$emailError = '';
		$commentError = '';
		if ( isset( $_POST['submitted'] ) ) {
			if ( trim( $_POST['ccmContactName'] ) === '' ) {
				$nameError = apply_filters( 'ccm_contact_form_name_error', 'Please enter your name.' );
				$hasError  = true;
			} else {
				$name = trim( $_POST['ccmContactName'] );
			}

			if ( trim( $_POST['ccmEmail'] ) === '' ) {
				$emailError = 'Please enter your email address.';
				$hasError   = true;
			} else if ( ! preg_match( "/^[[:alnum:]][a-z0-9_.-]*@[a-z0-9.-]+\.[a-z]{2,4}$/i", trim( $_POST['ccmEmail'] ) ) ) {
				$emailError = apply_filters( 'ccm_contact_form_email_error', 'You entered an invalid email address.' );
				$hasError   = true;
			} else {
				$email = trim( $_POST['ccmEmail'] );
			}

			if ( trim( $_POST['ccmComments'] ) === '' ) {
				$commentError = apply_filters( 'ccm_contact_form_message_error', 'Please enter a message.' );
				$hasError     = true;
			} else {
				if ( function_exists( 'stripslashes' ) ) {
					$comments = stripslashes( trim( $_POST['ccmComments'] ) );
				} else {
					$comments = trim( $_POST['ccmComments'] );
				}
			}

			if ( ! isset( $hasError ) ) {
				$emailTo = ccm_get_profile_meta( $args['post_id'], 'email' );
				/*$emailTo = ccm_get_profile_email( array( 'post_id' => $args['post_id'] ) );*/
				if ( ! isset( $emailTo ) || ( $emailTo == '' ) ) {
					$emailTo = get_option( 'admin_email' );
				}
				$subject = apply_filters( 'ccm_contact_form_subject_prefix', 'Message From ' ) . $name;
				$body    = apply_filters( 'ccm_contact_form_body', "Name: $name \n\nEmail: $email \n\nComments: $comments", array('name' => $name, 'email' => $email, 'comments' => $comments) );
				$headers = 'From: ' . $name . ' <' . $emailTo . '>' . "\r\n" . 'Reply-To: ' . $email;

				wp_mail( $emailTo, $subject, $body, $headers );
				$emailSent = true;
			}
		}

		$html .= '<div>';
		if(isset($emailSent) && $emailSent == true) {
			$html .= apply_filters( 'ccm_contact_form_email_sent', '<div><p>Thanks, your email was sent successfully.</p></div>' );
		} else { ?>
			<?php if(isset($hasError) || isset($captchaError)) {
				$html .= apply_filters( 'ccm_contact_form_error_occured', '<p>Sorry, an error occured.<p>' );
			}
			$html .= '<form action="'.ccm_profile_url( $args['post_id'] ).'" id="ccmContactForm" method="post">
				<ul>';
			$html .= '<li class="ccmContactName">
						<label for="ccmContactName">'.apply_filters( 'ccm_contact_form_label_name', 'Name:' ).'</label>';
			$html .= '<input type="text" name="ccmContactName" id="ccmContactName" value="'.(isset($_POST['ccmContactName'])? $_POST['ccmContactName']:"").'" />';
						if($nameError != '') {
							$html .= '<span>'.$nameError.'</span>';
						}
			$html .= '</li>';
			$html .= '<li class="ccmEmail">
						<label for="ccmEmail">'.apply_filters( 'ccm_contact_form_label_email', 'Email:' ).'</label>';
			$html .= '<input type="text" name="ccmEmail" id="ccmEmail" value="'.(isset($_POST['ccmEmail']) ? $_POST['ccmEmail']:"").'" />';
						if($emailError != '') {
							$html .= '<span>'.$emailError.'</span>';
						}
			$html .= '</li>
					<li class="ccmCommentsText"><label for="ccmCommentsText">'.apply_filters( 'ccm_contact_form_label_message', 'Message:' ).'</label>';
			$html .= '<textarea name="ccmComments" id="ccmCommentsText" rows="8" cols="30">'.(isset($_POST['ccmComments']) ? stripslashes($_POST['ccmComments']) : '').'</textarea>';
						if($commentError != '') {
							$html .= '<span>'.$commentError.'</span>';
						}
			$html .= '</li>
					<li>
						<input type="submit" class="submit" value="'.apply_filters( 'ccm_contact_form_label_submit', 'Send' ).'">
					</li>
				</ul>
				<input type="hidden" name="submitted" id="submitted" value="true" />
			</form>';
		}
		$html .= $args['after'];
	//}
	return apply_filters( 'ccm_get_profile_contact_form', $html, $args['post_id'] );
}

function ccm_email_body( $content, $args ) {
	$defaults = array(
		'name'     => '',
		'email'    => '',
		'comments' => ''
	);

	$args = wp_parse_args( $args, $defaults );

	return "Name: ".$args['name']." \n\nEmail: ".$args['email']." \n\nComments: ".$args['comments'];
}
add_filter( 'ccm_contact_form_body', 'ccm_email_body', 10, 2 );