<?php
/**
 * File for registering custom taxonomies.
 *
 * @package    CustomContentModels
 * @subpackage Includes
 * @author     Justin Tadlock <justin@justintadlock.com>
 * @copyright  Copyright (c) 2013-2015, Justin Tadlock
 * @license    http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 */

# Register taxonomies on the 'init' hook.
add_action( 'init', 'ccm_register_taxonomies', 9 );

# Filter the term updated messages.
add_filter( 'term_updated_messages', 'ccm_term_updated_messages', 5 );

/**
 * Returns the name of the models category taxonomy.
 *
 * @since  1.0.0
 * @access public
 * @return string
 */
function ccm_get_category_taxonomy() {

	return apply_filters( 'ccm_get_category_taxonomy', 'models_category' );
}

/**
 * Returns the name of the models tag taxonomy.
 *
 * @since  1.0.0
 * @access public
 * @return string
 */
function ccm_get_tag_taxonomy() {

	return apply_filters( 'ccm_get_tag_taxonomy', 'models_tag' );
}

/**
 * Returns the capabilities for the models category taxonomy.
 *
 * @since  1.0.0
 * @access public
 * @return array
 */
function ccm_get_category_capabilities() {

	$caps = array(
		'manage_terms' => 'manage_models_categories',
		'edit_terms'   => 'manage_models_categories',
		'delete_terms' => 'manage_models_categories',
		'assign_terms' => 'edit_models_profiles'
	);

	return apply_filters( 'ccm_get_category_capabilities', $caps );
}

/**
 * Returns the capabilities for the models tag taxonomy.
 *
 * @since  1.0.0
 * @access public
 * @return array
 */
function ccm_get_tag_capabilities() {

	$caps = array(
		'manage_terms' => 'manage_models_tags',
		'edit_terms'   => 'manage_models_tags',
		'delete_terms' => 'manage_models_tags',
		'assign_terms' => 'edit_models_profiles',
	);

	return apply_filters( 'ccm_get_tag_capabilities', $caps );
}

/**
 * Returns the labels for the models category taxonomy.
 *
 * @since  1.0.0
 * @access public
 * @return array
 */
function ccm_get_category_labels() {

	$labels = array(
		'name'                       => __( 'Models Categories',                           'custom-content-models' ),
		'singular_name'              => __( 'Category',                             'custom-content-models' ),
		'menu_name'                  => __( 'Categories',                           'custom-content-models' ),
		'name_admin_bar'             => __( 'Category',                             'custom-content-models' ),
		'search_items'               => __( 'Search Categories',                    'custom-content-models' ),
		'popular_items'              => __( 'Popular Categories',                   'custom-content-models' ),
		'all_items'                  => __( 'All Categories',                       'custom-content-models' ),
		'edit_item'                  => __( 'Edit Category',                        'custom-content-models' ),
		'view_item'                  => __( 'View Category',                        'custom-content-models' ),
		'update_item'                => __( 'Update Category',                      'custom-content-models' ),
		'add_new_item'               => __( 'Add New Category',                     'custom-content-models' ),
		'new_item_name'              => __( 'New Category Name',                    'custom-content-models' ),
		'not_found'                  => __( 'No categories found.',                 'custom-content-models' ),
		'no_terms'                   => __( 'No categories',                        'custom-content-models' ),
		'pagination'                 => __( 'Categories list navigation',           'custom-content-models' ),
		'list'                       => __( 'Categories list',                      'custom-content-models' ),

		// Hierarchical only.
		'select_name'                => __( 'Select Category',                      'custom-content-models' ),
		'parent_item'                => __( 'Parent Category',                      'custom-content-models' ),
		'parent_item_colon'          => __( 'Parent Category:',                     'custom-content-models' ),
	);

	return apply_filters( 'ccm_get_category_labels', $labels );
}

/**
 * Returns the labels for the models tag taxonomy.
 *
 * @since  1.0.0
 * @access public
 * @return array
 */
function ccm_get_tag_labels() {

	$labels = array(
		'name'                       => __( 'Tags',                           'custom-content-models' ),
		'singular_name'              => __( 'Tag',                            'custom-content-models' ),
		'menu_name'                  => __( 'Tags',                           'custom-content-models' ),
		'name_admin_bar'             => __( 'Tag',                            'custom-content-models' ),
		'search_items'               => __( 'Search Tags',                    'custom-content-models' ),
		'popular_items'              => __( 'Popular Tags',                   'custom-content-models' ),
		'all_items'                  => __( 'All Tags',                       'custom-content-models' ),
		'edit_item'                  => __( 'Edit Tag',                       'custom-content-models' ),
		'view_item'                  => __( 'View Tag',                       'custom-content-models' ),
		'update_item'                => __( 'Update Tag',                     'custom-content-models' ),
		'add_new_item'               => __( 'Add New Tag',                    'custom-content-models' ),
		'new_item_name'              => __( 'New Tag Name',                   'custom-content-models' ),
		'not_found'                  => __( 'No tags found.',                 'custom-content-models' ),
		'no_terms'                   => __( 'No tags',                        'custom-content-models' ),
		'pagination'                 => __( 'Tags list navigation',           'custom-content-models' ),
		'list'                       => __( 'Tags list',                      'custom-content-models' ),

		// Non-hierarchical only.
		'separate_items_with_commas' => __( 'Separate tags with commas',      'custom-content-models' ),
		'add_or_remove_items'        => __( 'Add or remove tags',             'custom-content-models' ),
		'choose_from_most_used'      => __( 'Choose from the most used tags', 'custom-content-models' ),
	);

	return apply_filters( 'ccm_get_tag_labels', $labels );
}

/**
 * Register taxonomies for the plugin.
 *
 * @since  0.1.0
 * @access public
 * @return void.
 */
function ccm_register_taxonomies() {

	// Set up the arguments for the models category taxonomy.
	$cat_args = array(
		'public'            => true,
		'show_ui'           => true,
		'show_in_nav_menus' => true,
		'show_tagcloud'     => true,
		'show_admin_column' => true,
		'hierarchical'      => true,
		'query_var'         => ccm_get_category_taxonomy(),
		'capabilities'      => ccm_get_category_capabilities(),
		'labels'            => ccm_get_category_labels(),

		// The rewrite handles the URL structure.
		'rewrite' => array(
			'slug'         => ccm_get_category_rewrite_slug(),
			'with_front'   => false,
			'hierarchical' => false,
			'ep_mask'      => EP_NONE
		),
	);

	// Set up the arguments for the models tag taxonomy.
	$tag_args = array(
		'public'            => true,
		'show_ui'           => true,
		'show_in_nav_menus' => true,
		'show_tagcloud'     => true,
		'show_admin_column' => true,
		'hierarchical'      => false,
		'query_var'         => ccm_get_tag_taxonomy(),
		'capabilities'      => ccm_get_tag_capabilities(),
		'labels'            => ccm_get_tag_labels(),

		// The rewrite handles the URL structure.
		'rewrite' => array(
			'slug'         => ccm_get_tag_rewrite_slug(),
			'with_front'   => false,
			'hierarchical' => false,
			'ep_mask'      => EP_NONE
		),
	);

	// Register the taxonomies.
	register_taxonomy( ccm_get_category_taxonomy(), ccm_get_profile_post_type(), apply_filters( 'ccm_category_taxonomy_args', $cat_args ) );
	register_taxonomy( ccm_get_tag_taxonomy(),      ccm_get_profile_post_type(), apply_filters( 'ccm_tag_taxonomy_args',      $tag_args ) );
}

/**
 * Filters the term updated messages in the admin.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $messages
 * @return array
 */
function ccm_term_updated_messages( $messages ) {

	$cat_taxonomy = ccm_get_category_taxonomy();
	$tag_taxonomy = ccm_get_tag_taxonomy();

	// Add the models category messages.
	$messages[ $cat_taxonomy ] = array(
		0 => '',
		1 => __( 'Category added.',       'custom-content-models' ),
		2 => __( 'Category deleted.',     'custom-content-models' ),
		3 => __( 'Category updated.',     'custom-content-models' ),
		4 => __( 'Category not added.',   'custom-content-models' ),
		5 => __( 'Category not updated.', 'custom-content-models' ),
		6 => __( 'Categories deleted.',   'custom-content-models' ),
	);

	// Add the models tag messages.
	$messages[ $tag_taxonomy ] = array(
		0 => '',
		1 => __( 'Tag added.',       'custom-content-models' ),
		2 => __( 'Tag deleted.',     'custom-content-models' ),
		3 => __( 'Tag updated.',     'custom-content-models' ),
		4 => __( 'Tag not added.',   'custom-content-models' ),
		5 => __( 'Tag not updated.', 'custom-content-models' ),
		6 => __( 'Tags deleted.',    'custom-content-models' ),
	);

	return $messages;
}
