<?php
/**
 * File for registering custom post types.
 *
 * @package    CustomContentModels
 * @subpackage Includes
 * @author     Justin Tadlock <justin@justintadlock.com>
 * @copyright  Copyright (c) 2013-2015, Justin Tadlock
 * @license    http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 */

# Register custom post types on the 'init' hook.
add_action( 'init', 'ccm_register_post_types' );

# Filter the "enter title here" text.
add_filter( 'enter_title_here', 'ccm_enter_title_here', 10, 2 );

# Filter the bulk and post updated messages.
add_filter( 'bulk_post_updated_messages', 'ccm_bulk_post_updated_messages', 5, 2 );
add_filter( 'post_updated_messages',      'ccm_post_updated_messages',      5    );

/**
 * Returns the name of the profile post type.
 *
 * @since  1.0.0
 * @access public
 * @return string
 */
function ccm_get_profile_post_type() {

	return apply_filters( 'ccm_get_profile_post_type', 'models_profile' );
}

/**
 * Returns the capabilities for the profile post type.
 *
 * @since  1.0.0
 * @access public
 * @return array
 */
function ccm_get_profile_capabilities() {

	$caps = array(

		// meta caps (don't assign these to roles)
		'edit_post'              => 'edit_models_profile',
		'read_post'              => 'read_models_profile',
		'delete_post'            => 'delete_models_profile',

		// primitive/meta caps
		'create_posts'           => 'create_models_profiles',

		// primitive caps used outside of map_meta_cap()
		'edit_posts'             => 'edit_models_profiles',
		'edit_others_posts'      => 'edit_others_models_profiles',
		'publish_posts'          => 'publish_models_profiles',
		'read_private_posts'     => 'read_private_models_profiles',

		// primitive caps used inside of map_meta_cap()
		'read'                   => 'read',
		'delete_posts'           => 'delete_models_profiles',
		'delete_private_posts'   => 'delete_private_models_profiles',
		'delete_published_posts' => 'delete_published_models_profiles',
		'delete_others_posts'    => 'delete_others_models_profiles',
		'edit_private_posts'     => 'edit_private_models_profiles',
		'edit_published_posts'   => 'edit_published_models_profiles'
	);

	return apply_filters( 'ccm_get_profile_capabilities', $caps );
}

/**
 * Returns the labels for the profile post type.
 *
 * @since  1.0.0
 * @access public
 * @return array
 */
function ccm_get_profile_labels() {

	$labels = array(
		'name'                  => __( 'Profiles',                   'custom-content-models' ),
		'singular_name'         => __( 'Profile',                    'custom-content-models' ),
		'menu_name'             => __( 'Models',                     'custom-content-models' ),
		'name_admin_bar'        => __( 'Profile',                    'custom-content-models' ),
		'add_new'               => __( 'New Profile',                'custom-content-models' ),
		'add_new_item'          => __( 'Add New Profile',            'custom-content-models' ),
		'edit_item'             => __( 'Edit Profile',               'custom-content-models' ),
		'new_item'              => __( 'New Profile',                'custom-content-models' ),
		'view_item'             => __( 'View Profile',               'custom-content-models' ),
		'search_items'          => __( 'Search Profiles',            'custom-content-models' ),
		'not_found'             => __( 'No profiles found',          'custom-content-models' ),
		'not_found_in_trash'    => __( 'No profiles found in trash', 'custom-content-models' ),
		'all_items'             => __( 'Profiles',                   'custom-content-models' ),
		'featured_image'        => __( 'Profile Image',              'custom-content-models' ),
		'set_featured_image'    => __( 'Set profile image',          'custom-content-models' ),
		'remove_featured_image' => __( 'Remove profile image',       'custom-content-models' ),
		'use_featured_image'    => __( 'Use as profile image',       'custom-content-models' ),
		'insert_into_item'      => __( 'Insert into profile',        'custom-content-models' ),
		'uploaded_to_this_item' => __( 'Uploaded to this profile',   'custom-content-models' ),
		'views'                 => __( 'Filter profiles list',       'custom-content-models' ),
		'pagination'            => __( 'Profiles list navigation',   'custom-content-models' ),
		'list'                  => __( 'Profiles list',              'custom-content-models' ),

		// Custom labels b/c WordPress doesn't have anything to handle this.
		'archive_title'         => ccm_get_models_title(),
	);

	return apply_filters( 'ccm_get_profile_labels', $labels );
}

/**
 * Registers post types needed by the plugin.
 *
 * @since  0.1.0
 * @access public
 * @return void
 */
function ccm_register_post_types() {

	// Set up the arguments for the models profile post type.
	$profile_args = array(
		'description'         => ccm_get_models_description(),
		'public'              => true,
		'publicly_queryable'  => true,
		'show_in_nav_menus'   => false,
		'show_in_admin_bar'   => true,
		'exclude_from_search' => false,
		'show_ui'             => true,
		'show_in_menu'        => true,
		'menu_position'       => null,
		'menu_icon'           => 'dashicons-id',
		'can_export'          => true,
		'delete_with_user'    => false,
		'hierarchical'        => false,
		'has_archive'         => ccm_get_models_rewrite_base(),
		'query_var'           => ccm_get_profile_post_type(),
		'capability_type'     => 'models_profile',
		'map_meta_cap'        => true,
		'capabilities'        => ccm_get_profile_capabilities(),
		'labels'              => ccm_get_profile_labels(),

		// The rewrite handles the URL structure.
		'rewrite' => array(
			'slug'       => ccm_get_profile_rewrite_slug(),
			'with_front' => false,
			'pages'      => true,
			'feeds'      => true,
			'ep_mask'    => EP_PERMALINK,
		),

		// What features the post type supports.
		'supports' => array(
			'title',
			'editor',
			'excerpt',
			'author',
			'thumbnail',
			'post-formats',
			//'comments',
			'revisions',

			// Theme/Plugin feature support.
			'custom-background', // Custom Background Extended
			'custom-header',     // Custom Header Extended
		)
	);

	// Register the post types.
	register_post_type( ccm_get_profile_post_type(), apply_filters( 'ccm_profile_post_type_args', $profile_args ) );
}

/**
 * Custom "enter title here" text.
 *
 * @since  1.0.0
 * @access public
 * @param  string  $title
 * @param  object  $post
 * @return string
 */
function ccm_enter_title_here( $title, $post ) {

	return ccm_get_profile_post_type() === $post->post_type ? esc_html__( 'Enter profile title', 'custom-content-models' ) : '';
}

/**
 * Adds custom post updated messages on the edit post screen.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $messages
 * @global object $post
 * @global int    $post_ID
 * @return array
 */
function ccm_post_updated_messages( $messages ) {
	global $post, $post_ID;

	$profile_type = ccm_get_profile_post_type();

	if ( $profile_type !== $post->post_type )
		return $messages;

	// Get permalink and pmodel URLs.
	$permalink   = get_permalink( $post_ID );
	$pmodel_url = function_exists( 'get_pmodel_post_link' ) ? get_pmodel_post_link( $post ) : apply_filters( 'pmodel_post_link', add_query_arg( array( 'pmodel' => true ), $permalink ), $post );

	// Translators: Scheduled profile date format. See http://php.net/date
	$scheduled_date = date_i18n( __( 'M j, Y @ H:i', 'custom-content-models' ), strtotime( $post->post_date ) );

	// Set up view links.
	$pmodel_link   = sprintf( ' <a target="_blank" href="%1$s">%2$s</a>', esc_url( $pmodel_url ), esc_html__( 'Pmodel profile', 'custom-content-models' ) );
	$scheduled_link = sprintf( ' <a target="_blank" href="%1$s">%2$s</a>', esc_url( $permalink ),   esc_html__( 'Pmodel profile', 'custom-content-models' ) );
	$view_link      = sprintf( ' <a href="%1$s">%2$s</a>',                 esc_url( $permalink ),   esc_html__( 'View profile',    'custom-content-models' ) );

	// Post updated messages.
	$messages[ $profile_type ] = array(
		 1 => esc_html__( 'Profile updated.', 'custom-content-models' ) . $view_link,
		 4 => esc_html__( 'Profile updated.', 'custom-content-models' ),
		 // Translators: %s is the date and time of the revision.
		 5 => isset( $_GET['revision'] ) ? sprintf( esc_html__( 'Profile restored to revision from %s.', 'custom-content-models' ), wp_post_revision_title( (int) $_GET['revision'], false ) ) : false,
		 6 => esc_html__( 'Profile published.', 'custom-content-models' ) . $view_link,
		 7 => esc_html__( 'Profile saved.', 'custom-content-models' ),
		 8 => esc_html__( 'Profile submitted.', 'custom-content-models' ) . $pmodel_link,
		 9 => sprintf( esc_html__( 'Profile scheduled for: %s.', 'custom-content-models' ), "<strong>{$scheduled_date}</strong>" ) . $scheduled_link,
		10 => esc_html__( 'Profile draft updated.', 'custom-content-models' ) . $pmodel_link,
	);

	return $messages;
}

/**
 * Adds custom bulk post updated messages on the manage profiles screen.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $messages
 * @param  array  $counts
 * @return array
 */
function ccm_bulk_post_updated_messages( $messages, $counts ) {

	$type = ccm_get_profile_post_type();

	$messages[ $type ]['updated']   = _n( '%s profile updated.',                             '%s profiles updated.',                               $counts['updated'],   'custom-content-models' );
	$messages[ $type ]['locked']    = _n( '%s profile not updated, somebody is editing it.', '%s profiles not updated, somebody is editing them.', $counts['locked'],    'custom-content-models' );
	$messages[ $type ]['deleted']   = _n( '%s profile permanently deleted.',                 '%s profiles permanently deleted.',                   $counts['deleted'],   'custom-content-models' );
	$messages[ $type ]['trashed']   = _n( '%s profile moved to the Trash.',                  '%s profiles moved to the trash.',                    $counts['trashed'],   'custom-content-models' );
	$messages[ $type ]['untrashed'] = _n( '%s profile restored from the Trash.',             '%s profiles restored from the trash.',               $counts['untrashed'], 'custom-content-models' );

	return $messages;
}
