<?php
/**
 * Various functions, filters, and actions used by the plugin.
 *
 * @package    CustomContentModels
 * @subpackage Includes
 * @author     Justin Tadlock <justin@justintadlock.com>
 * @copyright  Copyright (c) 2013-2015, Justin Tadlock
 * @license    http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 */

# Check theme support.
add_action( 'wp_loaded', 'ccm_check_theme_support', 0 );

# Template hierarchy.
add_filter( 'template_include', 'ccm_template_include', 5 );

# Add sticky posts to the front of the line.
add_filter( 'the_posts', 'ccm_posts_sticky_filter', 10, 2 );

# Filter the document title.
add_filter( 'document_title_parts', 'ccm_document_title_parts', 5 );

# Filter the post type archive title.
add_filter( 'post_type_archive_title', 'ccm_post_type_archive_title', 5, 2 );

# Filter the archive title and description.
add_filter( 'get_the_archive_title',       'ccm_get_the_archive_title',       5 );
add_filter( 'get_the_archive_description', 'ccm_get_the_archive_description', 5 );

# Filter the post type permalink.
add_filter( 'post_type_link', 'ccm_post_type_link', 10, 2 );

# Filter the post author link.
add_filter( 'author_link', 'ccm_author_link_filter', 10, 3 );

# Force taxonomy term selection.
add_action( 'save_post', 'ccm_force_term_selection' );

# Filter the Breadcrumb Trail plugin args.
add_filter( 'breadcrumb_trail_args', 'ccm_breadcrumb_trail_args', 15 );

#add Thumbnail image size
add_action( 'after_setup_theme', 'ccm_image_size_setup', 20 );

/**
 * Checks if the theme supports `custom-content-models`.  If not, it runs specific filters
 * to make themes without support work a little better.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function ccm_check_theme_support() {

	if ( ! current_theme_supports( 'custom-content-models' ) )
		add_filter( 'the_content', 'ccm_the_content_filter', 25 );
}

/**
 * Basic top-level template hierarchy. I generally prefer to leave this functionality up to
 * themes.  This is just a foundation to build upon if needed.
 *
 * @since  1.0.0
 * @access public
 * @param  string  $template
 * @return string
 */
function ccm_template_include( $template ) {

	// Bail if not a models page.
	if ( ! ccm_is_models() )
		return $template;

	$templates = array();

	// Author archive.
	if ( ccm_is_author() ) {
		$templates[] = 'models-author.php';
		$templates[] = 'models-archive.php';

	// Category archive.
	} else if ( ccm_is_category() ) {
		$templates[] = 'models-category.php';
		$templates[] = 'models-archive.php';

	// Tag archive.
	} else if ( ccm_is_tag() ) {
		$templates[] = 'models-tag.php';
		$templates[] = 'models-archive.php';

	// Profile archive.
	} else if ( ccm_is_profile_archive() ) {
		$templates[] = 'models-archive.php';

	// Single profile.
	} else if ( ccm_is_single_profile() ) {
		$templates[] = 'models-profile.php';
	}

	// Fallback template.
	$templates[] = 'models.php';

	// Check if we have a template.
	$has_template = locate_template( apply_filters( 'ccm_template_hierarchy', $templates ) );

	// Return the template.
	return $has_template ? $has_template : $template;
}

/**
 * Filter on `the_content` for themes that don't support the plugin.  This filter outputs the basic
 * profile metadata only.
 *
 * @since  1.0.0
 * @access public
 * @param  string  $content
 * @return string
 */
function ccm_the_content_filter( $content ) {

	if ( in_the_loop() && ccm_is_single_profile() && ccm_is_profile() && ! post_password_required() ) {

		$categories = get_categories( array( 'type' => 'models_profile', 'orderby' => 'name', 'taxonomy' => 'models_category' ) );
		$tags = get_categories( array( 'type' => 'models_profile', 'orderby' => 'name', 'taxonomy' => 'models_tag' ) );

		$before = '';
		$after = '';
		$address = ccm_get_profile_meta( get_the_ID(), 'address' );
		$address_formatted = ccm_get_profile_address( array( 'text' => '%s', 'before' => $before, 'after' => $after ) );
		$phone = ccm_get_profile_meta( get_the_ID(), 'phone' );
		$phone_formatted = ccm_get_profile_phone( array( 'text' => '%s', 'before' => $before, 'after' => $after ) );
		$email = ccm_get_profile_meta( get_the_ID(), 'email' );
		$email_formatted = ccm_get_profile_phone( array( 'text' => '%s', 'before' => $before, 'after' => $after ) );

		$height = ccm_get_profile_meta( get_the_ID(), 'height' );
		$height_formatted = ccm_get_profile_height( array( 'text' => '%s', 'before' => $before, 'after' => $after ) );
		$bust = ccm_get_profile_meta( get_the_ID(), 'bust' );
		$bust_formatted = ccm_get_profile_bust( array( 'text' => '%s', 'before' => $before, 'after' => $after ) );
		$waist = ccm_get_profile_meta( get_the_ID(), 'waist' );
		$waist_formatted = ccm_get_profile_waist( array( 'text' => '%s', 'before' => $before, 'after' => $after ) );
		$hips = ccm_get_profile_meta( get_the_ID(), 'hips' );
		$hips_formatted = ccm_get_profile_hips( array( 'text' => '%s', 'before' => $before, 'after' => $after ) );
		$shoe = ccm_get_profile_meta( get_the_ID(), 'shoe' );
		$shoe_formatted = ccm_get_profile_shoe( array( 'text' => '%s', 'before' => $before, 'after' => $after ) );
		$eyes = ccm_get_profile_meta( get_the_ID(), 'eyes' );
		$eyes_formatted = ccm_get_profile_eyes( array( 'text' => '%s', 'before' => $before, 'after' => $after ) );

		$facebook = ccm_get_profile_meta( get_the_ID(), 'facebook' );
		$facebook_formatted = ccm_get_profile_facebook( array( 'text' => '%s', 'before' => $before, 'after' => $after ) );
		$twitter = ccm_get_profile_meta( get_the_ID(), 'twitter' );
		$twitter_formatted = ccm_get_profile_twitter( array( 'text' => '%s', 'before' => $before, 'after' => $after ) );
		$googleplus = ccm_get_profile_meta( get_the_ID(), 'googleplus' );
		$googleplus_formatted = ccm_get_profile_googleplus( array( 'text' => '%s', 'before' => $before, 'after' => $after ) );
		$linkedin = ccm_get_profile_meta( get_the_ID(), 'linkedin' );
		$linkedin_formatted = ccm_get_profile_linkedin( array( 'text' => '%s', 'before' => $before, 'after' => $after ) );
		$pinterest = ccm_get_profile_meta( get_the_ID(), 'pinterest' );
		$pinterest_formatted = ccm_get_profile_pinterest( array( 'text' => '%s', 'before' => $before, 'after' => $after ) );
		$instagram = ccm_get_profile_meta( get_the_ID(), 'instagram' );
		$instagram_formatted = ccm_get_profile_instagram( array( 'text' => '%s', 'before' => $before, 'after' => $after ) );

		$output = '';

		if( !empty($address) ) {
			$output .= '<div class="model-address">';
			$output .= esc_html__( 'Address: ', 'custom-content-models' ).esc_html__($address);
			$output .= '</div>';
		}

		if( !empty($phone) ) {
			$output .= '<div class="model-phone">';
			$output .= esc_html__( 'Phone: ', 'custom-content-models' ).esc_html__($phone);
			$output .= '</div>';
		}

		if( !empty($email) ) {
			$output .= '<div class="model-email">';
			$output .= esc_html__( 'Email: ', 'custom-content-models' ).esc_html__($email);
			$output .= '</div>';
		}



		if( !empty($height) ) {
			$output .= '<div class="model-height">';
			$output .= esc_html__( 'Height: ', 'custom-content-models' ).esc_html__($height);
			$output .= '</div>';
		}

		if( !empty($bust) ) {
			$output .= '<div class="model-bust">';
			$output .= esc_html__( 'Bust: ', 'custom-content-models' ).esc_html__($bust);
			$output .= '</div>';
		}

		if( !empty($waist) ) {
			$output .= '<div class="model-waist">';
			$output .= esc_html__( 'Waist: ', 'custom-content-models' ).esc_html__($waist);
			$output .= '</div>';
		}

		if( !empty($hips) ) {
			$output .= '<div class="model-hips">';
			$output .= esc_html__( 'Hips: ', 'custom-content-models' ).esc_html__($hips);
			$output .= '</div>';
		}

		if( !empty($shoe) ) {
			$output .= '<div class="model-shoe">';
			$output .= esc_html__( 'Shoe: ', 'custom-content-models' ).esc_html__($shoe);
			$output .= '</div>';
		}

		if( !empty($eyes) ) {
			$output .= '<div class="model-eyes">';
			$output .= esc_html__( 'Eyes: ', 'custom-content-models' ).esc_html__($eyes);
			$output .= '</div>';
		}

		$images = ccm_get_multi_images_src();
		foreach ( $images as $image ){
			$output .= '<a href="'.$image[1][0].'" rel="prettyPhoto[gallery]">';
			$output .= '<img src="'.$image[0][0].'" alt="">';
			$output .= '</a>';
		}

		if( !empty($facebook) ) {
			$output .= '<div class="model-facebook">';
			$output .= esc_html__( 'Facebook: ', 'custom-content-models' ).$facebook_formatted;
			$output .= '</div>';
		}

		if( !empty($twitter) ) {
			$output .= '<div class="model-twitter">';
			$output .= esc_html__( 'Twitter: ', 'custom-content-models' ).$twitter_formatted;
			$output .= '</div>';
		}

		if( !empty($googleplus) ) {
			$output .= '<div class="model-googleplus">';
			$output .= esc_html__( 'Google+: ', 'custom-content-models' ).$googleplus_formatted;
			$output .= '</div>';
		}

		if( !empty($linkedin) ) {
			$output .= '<div class="model-linkedin">';
			$output .= esc_html__( 'LinkedIn: ', 'custom-content-models' ).$linkedin_formatted;
			$output .= '</div>';
		}

		if( !empty($pinterest) ) {
			$output .= '<div class="model-pinterest">';
			$output .= esc_html__( 'Pinterest: ', 'custom-content-models' ).$pinterest_formatted;
			$output .= '</div>';
		}

		if( !empty($instagram) ) {
			$output .= '<div class="model-instagram">';
			$output .= esc_html__( 'Instagram: ', 'custom-content-models' ).$instagram_formatted;
			$output .= '</div>';
		}



		if( !empty($categories) ) {
			$output .= '<div class="model-categories"><ul>';

			$output .= '<li>'.esc_html__( 'Categories: ', 'custom-content-models' ).'</li>';

			foreach ($categories as $category) {
				$output .= '<li><a href="'.get_term_link($category).'">'.$category->name.'</a></li>';
			}

			$output .= '</ul></div>';
		}

		if( !empty($tags) ) {
			$output .= '<div class="model-categories"><ul>';

			$output .= '<li>'.esc_html__( 'Tags: ', 'custom-content-models' ).'</li>';

			foreach ($tags as $tag) {
				$output .= '<li><a href="'.get_term_link($tag).'">'.$tag->name.'</a></li>';
			}

			$output .= '</ul></div>';
		}

		$output .= ccm_get_profile_contact_form();

		if ( $output )
			$content .= sprintf( '<span class="profile-meta">%s</span>', $output );
	}

	return $content;
}

/**
 * Filter on `the_posts` for the profile archive. Moves sticky posts to the top of
 * the profile archive list.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $posts
 * @param  object $query
 * @return array
 */
function ccm_posts_sticky_filter( $posts, $query ) {

	// Allow devs to filter when to show sticky profiles.
	$show_stickies = apply_filters( 'ccm_show_stickies', $query->is_main_query() && ! is_admin() && ccm_is_profile_archive() && ! is_paged() );

	// If we should show stickies, let's get them.
	if ( $show_stickies ) {

		remove_filter( 'the_posts', 'ccm_posts_sticky_filter' );

		$posts = ccm_add_stickies( $posts, ccm_get_sticky_profiles() );
	}

	return $posts;
}

/**
 * Adds sticky posts to the front of the line with any given set of posts and stickies.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $posts         Array of post objects.
 * @param  array  $sticky_posts  Array of post IDs.
 * @return array
 */
function ccm_add_stickies( $posts, $sticky_posts ) {

	// Only do this if on the first page and we indeed have stickies.
	if ( ! empty( $sticky_posts ) ) {

		$num_posts     = count( $posts );
		$sticky_offset = 0;

		// Loop over posts and relocate stickies to the front.
		for ( $i = 0; $i < $num_posts; $i++ ) {

			if ( in_array( $posts[ $i ]->ID, $sticky_posts ) ) {

				$sticky_post = $posts[ $i ];

				// Remove sticky from current position.
				array_splice( $posts, $i, 1);

				// Move to front, after other stickies.
				array_splice( $posts, $sticky_offset, 0, array( $sticky_post ) );

				// Increment the sticky offset. The next sticky will be placed at this offset.
				$sticky_offset++;

				// Remove post from sticky posts array.
				$offset = array_search( $sticky_post->ID, $sticky_posts );

				unset( $sticky_posts[ $offset ] );
			}
		}

		// Fetch sticky posts that weren't in the query results.
		if ( ! empty( $sticky_posts ) ) {

			$args = array(
					'post__in'    => $sticky_posts,
					'post_type'   => ccm_get_profile_post_type(),
					'post_status' => 'publish',
					'nopaging'    => true
			);

			$stickies = get_posts( $args );

			foreach ( $stickies as $sticky_post ) {
				array_splice( $posts, $sticky_offset, 0, array( $sticky_post ) );
				$sticky_offset++;
			}
		}
	}

	return $posts;
}

/**
 * Filter on `document_title_parts` (WP 4.4.0).
 *
 * @since  1.0.0
 * @access public
 * @param  array  $title
 * @return array
 */
function ccm_document_title_parts( $title ) {

	if ( ccm_is_author() )
		$title['title'] = ccm_get_single_author_title();

	return $title;
}

/**
 * Filter on 'post_type_archive_title' to allow for the use of the 'archive_title' label that isn't supported
 * by WordPress.  That's okay since we can roll our own labels.
 *
 * @since  0.1.0
 * @access public
 * @param  string  $title
 * @param  string  $post_type
 * @return string
 */
function ccm_post_type_archive_title( $title, $post_type ) {

	$profile_type = ccm_get_profile_post_type();

	return $profile_type === $post_type ? get_post_type_object( ccm_get_profile_post_type() )->labels->archive_title : $title;
}

/**
 * Filters the archive title. Note that we need this additional filter because core WP does
 * things like add "Archives:" in front of the archive title.
 *
 * @since  1.0.0
 * @access public
 * @param  string  $title
 * @return string
 */
function ccm_get_the_archive_title( $title ) {

	if ( ccm_is_author() )
		$title = ccm_get_single_author_title();

	else if ( ccm_is_profile_archive() )
		$title = post_type_archive_title( '', false );

	return $title;
}

/**
 * Filters the archive description.
 *
 * @since  1.0.0
 * @access public
 * @param  string  $desc
 * @return string
 */
function ccm_get_the_archive_description( $desc ) {

	if ( ccm_is_author() )
		$desc = get_the_author_meta( 'description', get_query_var( 'author' ) );

	else if ( ccm_is_profile_archive() && ! $desc )
		$desc = ccm_get_models_description();

	return $desc;
}

/**
 * Filter on `post_type_link` to make sure that single models profiles have the correct
 * permalink.
 *
 * @since  0.1.0
 * @access public
 * @param  string  $post_link
 * @param  object  $post
 * @return string
 */
function ccm_post_type_link( $post_link, $post ) {

	// Bail if this isn't a models profile.
	if ( ccm_get_profile_post_type() !== $post->post_type )
		return $post_link;

	$cat_taxonomy = ccm_get_category_taxonomy();
	$tag_taxonomy = ccm_get_tag_taxonomy();

	$author = $category = $tag = '';

	// Check for the category.
	if ( false !== strpos( $post_link, "%{$cat_taxonomy}%" ) ) {

		// Get the terms.
		$terms = get_the_terms( $post, $cat_taxonomy );

		// Check that terms were returned.
		if ( $terms ) {

			usort( $terms, '_usort_terms_by_ID' );

			$category = $terms[0]->slug;
		}
	}

	// Check for the tag.
	if ( false !== strpos( $post_link, "%{$tag_taxonomy}%" ) ) {

		// Get the terms.
		$terms = get_the_terms( $post, $tag_taxonomy );

		// Check that terms were returned.
		if ( $terms ) {

			usort( $terms, '_usort_terms_by_ID' );

			$tag = $terms[0]->slug;
		}
	}

	// Check for the author.
	if ( false !== strpos( $post_link, '%author%' ) ) {

		$authordata = get_userdata( $post->post_author );
		$author     = $authordata->user_nicename;
	}

	$rewrite_tags = array(
		'%models_category%',
		'%models_tag%',
		'%author%'
	);

	$map_tags = array(
		$category,
		$tag,
		$author
	);

	return str_replace( $rewrite_tags, $map_tags, $post_link );
}

/**
 * Filter on `author_link` to change the URL when viewing a models profile. The new link
 * should point to the models author archive.
 *
 * @since  1.0.0
 * @access public
 * @param  string  $url
 * @param  int     $author_id
 * @param  string  $nicename
 * @return string
 */
function ccm_author_link_filter( $url, $author_id, $nicename ) {

	return ccm_is_profile() ? ccm_get_author_url( $author_id ) : $url;
}

/**
 * If a profile has `%models_category%` or `%models_tag%` in its permalink structure,
 * it must have a term set for the taxonomy.  This function is a callback on `save_post`
 * that checks if a term is set.  If not, it forces the first term of the taxonomy to be
 * the selected term.
 *
 * @since  1.0.0
 * @access public
 * @param  int    $post_id
 * @return void
 */
function ccm_force_term_selection( $post_id ) {

	if ( ccm_is_profile( $post_id ) ) {

		$profile_base = ccm_get_profile_rewrite_base();
		$cat_tax      = ccm_get_category_taxonomy();
		$tag_tax      = ccm_get_tag_taxonomy();

		if ( false !== strpos( $profile_base, "%{$cat_tax}%" ) )
			ccm_set_term_if_none( $post_id, $cat_tax, ccm_get_default_category() );

		if ( false !== strpos( $profile_base, "%{$tag_tax}%" ) )
			ccm_set_term_if_none( $post_id, $tag_tax, ccm_get_default_tag() );
	}
}

/**
 * Checks if a post has a term of the given taxonomy.  If not, set it with the first
 * term available from the taxonomy.
 *
 * @since  1.0.0
 * @access public
 * @param  int     $post_id
 * @param  string  $taxonomy
 * @param  int     $default
 * @return void
 */
function ccm_set_term_if_none( $post_id, $taxonomy, $default = 0 ) {

	// Get the current post terms.
	$terms = wp_get_post_terms( $post_id, $taxonomy );

	// If no terms are set, let's roll.
	if ( ! $terms ) {

		$new_term = false;

		// Get the default term if set.
		if ( $default )
			$new_term = get_term( $default, $taxonomy );

		// If no default term or if there's an error, get the first term.
		if ( ! $new_term || is_wp_error( $new_term ) ) {
			$available = get_terms( $taxonomy, array( 'number' => 1 ) );

			// Get the first term.
			$new_term = $available ? array_shift( $available ) : false;
		}

		// Only run if there are taxonomy terms.
		if ( $new_term ) {
			$tax_object = get_taxonomy( $taxonomy );

			// Use the ID for hierarchical taxonomies. Use the slug for non-hierarchical.
			$slug_or_id = $tax_object->hierarchical ? $new_term->term_id : $new_term->slug;

			// Set the new post term.
			wp_set_post_terms( $post_id, $slug_or_id, $taxonomy, true );
		}
	}
}

/**
 * Filters the Breadcrumb Trail plugin arguments.  We're basically just telling it to show the
 * `models_category` taxonomy when viewing single models profiles.
 *
 * @since  1.0.0
 * @access public
 * @param  array  $args
 * @return array
 */
function ccm_breadcrumb_trail_args( $args ) {

	$profile_type = ccm_get_profile_post_type();
	$profile_base = ccm_get_profile_rewrite_base();

	if ( false === strpos( $profile_base, '%' ) && ! isset( $args['post_taxonomy'][ $profile_type ] ) )
		$args['post_taxonomy'][ $profile_type ] = ccm_get_category_taxonomy();

	return $args;
}

if ( ! function_exists( 'ccm_image_size_setup' ) ) :
	/**
	 * Sets up theme defaults and registers support for various WordPress features.
	 *
	 * Note that this function is hooked into the after_setup_theme hook, which
	 * runs before the init hook. The init hook is too late for some features, such
	 * as indicating support for post thumbnails.
	 */
	function ccm_image_size_setup() {
		if(!has_image_size('ccm-model-thumbnail')){
			add_image_size( 'ccm-model-thumbnail', 1140, 640, true );
		}
		if(!has_image_size('ccm-model-thumbnail-2')){
			add_image_size( 'ccm-model-thumbnail-2', 600, 800, true );
		}
	}
endif;