<?php
/**
 * Admin-related functions and filters.
 *
 * @package    CustomContentModels
 * @subpackage Admin
 * @author     Justin Tadlock <justin@justintadlock.com>
 * @copyright  Copyright (c) 2013-2015, Justin Tadlock
 * @license    http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 */

# Register scripts and styles.
add_action( 'admin_enqueue_scripts', 'ccm_admin_register_scripts', 0 );
add_action( 'admin_enqueue_scripts', 'ccm_admin_register_styles',  0 );

# Registers profile details box sections, controls, and settings.
add_action( 'ccm_profile_details_manager_register', 'ccm_profile_details_register', 5 );

# Filter post format support for profiles.
add_action( 'load-post.php',     'ccm_post_format_support_filter' );
add_action( 'load-post-new.php', 'ccm_post_format_support_filter' );
add_action( 'load-edit.php',     'ccm_post_format_support_filter' );

/**
 * Registers admin scripts.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function ccm_admin_register_scripts() {

	$min = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

	wp_register_script( 'ccm-edit-profile', ccm_plugin()->js_uri . "edit-profile{$min}.js", array( 'jquery' ), '', true );

	// Localize our script with some text we want to pass in.
	$i18n = array(
		'label_sticky'     => esc_html__( 'Sticky',     'custom-content-models' ),
		'label_not_sticky' => esc_html__( 'Not Sticky', 'custom-content-models' ),
	);

	wp_localize_script( 'ccm-edit-profile', 'ccm_i18n', $i18n );
}

/**
 * Registers admin styles.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function ccm_admin_register_styles() {

	$min = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

	wp_register_style( 'ccm-admin', ccm_plugin()->css_uri . "admin{$min}.css" );
}

/**
 * Registers the default cap groups.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function ccm_profile_details_register( $manager ) {

	/* === Register Sections === */

	// Contacts section.
	$manager->register_section( 'contacts',
		array(
			'label' => esc_html__( 'Contacts', 'custom-content-models' ),
			'icon'  => 'dashicons-email'
		)
	);

	// Measurements section.
	$manager->register_section( 'measurements',
		array(
			'label' => esc_html__( 'Measurements', 'custom-content-models' ),
			'icon'  => 'dashicons-leftright'
		)
	);

	// Social Networks section.
	$manager->register_section( 'social',
		array(
			'label' => esc_html__( 'Social Networks', 'custom-content-models' ),
			'icon'  => 'dashicons-share'
		)
	);

	/* === Register Controls === */

	$address_args = array(
		'section'     => 'contacts',
		'attr'        => array( 'class' => 'widefat', 'placeholder' => 'PO Box 97845 Baker 567, San Diego, California, US' ),
		'label'       => esc_html__( 'Address', 'custom-content-models' ),
		'description' => esc_html__( 'Model\'s address.' , 'custom-content-models' )
	);

	$phone_args = array(
		'section'     => 'contacts',
		'attr'        => array( 'class' => 'widefat', 'placeholder' => __( '8(800) 456-2696', 'custom-content-models' ) ),
		'label'       => esc_html__( 'Phone', 'custom-content-models' ),
		'description' => esc_html__( 'Enter model\'s phone number.', 'custom-content-models' )
	);

	$email_args = array(
		'section'     => 'contacts',
		'attr'        => array( 'class' => 'widefat', 'placeholder' => __( 'agency@email.com', 'custom-content-models' ) ),
		'label'       => esc_html__( 'Email', 'custom-content-models' ),
		'description' => esc_html__( 'Enter model\'s email address.', 'custom-content-models' )
	);

	$height_args = array(
		'section'     => 'measurements',
		'attr'        => array( 'class' => 'widefat', 'placeholder' => __( '185' ) ),
		'label'       => esc_html__( 'Height', 'custom-content-models' ),
		'description' => esc_html__( 'Model\'s height.', 'custom-content-models' )
	);

	$bust_args = array(
		'section'     => 'measurements',
		'attr'        => array( 'class' => 'widefat', 'placeholder' => __( '79' ) ),
		'label'       => esc_html__( 'Bust', 'custom-content-models' ),
		'description' => esc_html__( 'Model\'s bust size.', 'custom-content-models' )
	);

	$waist_args = array(
		'section'     => 'measurements',
		'attr'        => array( 'class' => 'widefat', 'placeholder' => __( '69' ) ),
		'label'       => esc_html__( 'Waist', 'custom-content-models' ),
		'description' => esc_html__( 'Model\'s waist size.', 'custom-content-models' )
	);

	$hips_args = array(
		'section'     => 'measurements',
		'attr'        => array( 'class' => 'widefat', 'placeholder' => __( '87' ) ),
		'label'       => esc_html__( 'Hips', 'custom-content-models' ),
		'description' => esc_html__( 'Model\'s hips size.', 'custom-content-models' )
	);

	$shoe_args = array(
		'section'     => 'measurements',
		'attr'        => array( 'class' => 'widefat', 'placeholder' => __( '39' ) ),
		'label'       => esc_html__( 'Shoe', 'custom-content-models' ),
		'description' => esc_html__( 'Model\'s shoe size.', 'custom-content-models' )
	);

	$eyes_args = array(
		'section'     => 'measurements',
		'attr'        => array( 'class' => 'widefat', 'placeholder' => __( 'Blue' ) ),
		'label'       => esc_html__( 'Eyes', 'custom-content-models' ),
		'description' => esc_html__( 'Model\'s eye color.', 'custom-content-models' )
	);

	$facebook_args = array(
		'section'     => 'social',
		'attr'        => array( 'class' => 'widefat', 'placeholder' => __( 'Jane Doe', 'custom-content-models' ) ),
		'label'       => esc_html__( 'Facebook', 'custom-content-models' ),
		'description' => esc_html__( 'Model\'s Facebook Profile Address.', 'custom-content-models' )
	);

	$twitter_args = array(
		'section'     => 'social',
		'attr'        => array( 'class' => 'widefat', 'placeholder' => __( 'Jane Doe', 'custom-content-models' ) ),
		'label'       => esc_html__( 'Twitter', 'custom-content-models' ),
		'description' => esc_html__( 'Model\'s Twitter Profile Address.', 'custom-content-models' )
	);

	$googleplus_args = array(
		'section'     => 'social',
		'attr'        => array( 'class' => 'widefat', 'placeholder' => __( 'Jane Doe', 'custom-content-models' ) ),
		'label'       => esc_html__( 'Google+', 'custom-content-models' ),
		'description' => esc_html__( 'Model\'s Google+ Profile Address.', 'custom-content-models' )
	);

	$linkedin_args = array(
		'section'     => 'social',
		'attr'        => array( 'class' => 'widefat', 'placeholder' => __( 'Jane Doe', 'custom-content-models' ) ),
		'label'       => esc_html__( 'LinkedIn', 'custom-content-models' ),
		'description' => esc_html__( 'Model\'s LinkedIn Profile Address.', 'custom-content-models' )
	);

	$pinterest_args = array(
		'section'     => 'social',
		'attr'        => array( 'class' => 'widefat', 'placeholder' => __( 'Jane Doe', 'custom-content-models' ) ),
		'label'       => esc_html__( 'Pinterest', 'custom-content-models' ),
		'description' => esc_html__( 'Model\'s Pinterest Profile Address.', 'custom-content-models' )
	);

	$instagram_args = array(
		'section'     => 'social',
		'attr'        => array( 'class' => 'widefat', 'placeholder' => __( 'Jane Doe', 'custom-content-models' ) ),
		'label'       => esc_html__( 'Instagram', 'custom-content-models' ),
		'description' => esc_html__( 'Model\'s Instagram Profile Address.', 'custom-content-models' )
	);

	$manager->register_control( new CCM_Fields_Control( $manager, 'address',    $address_args    ) );
	$manager->register_control( new CCM_Fields_Control( $manager, 'phone',      $phone_args      ) );
	$manager->register_control( new CCM_Fields_Control( $manager, 'email',      $email_args      ) );
	$manager->register_control( new CCM_Fields_Control( $manager, 'height',     $height_args     ) );
	$manager->register_control( new CCM_Fields_Control( $manager, 'bust',       $bust_args       ) );
	$manager->register_control( new CCM_Fields_Control( $manager, 'waist',      $waist_args      ) );
	$manager->register_control( new CCM_Fields_Control( $manager, 'hips',       $hips_args       ) );
	$manager->register_control( new CCM_Fields_Control( $manager, 'shoe',       $shoe_args       ) );
	$manager->register_control( new CCM_Fields_Control( $manager, 'eyes',       $eyes_args       ) );
	$manager->register_control( new CCM_Fields_Control( $manager, 'facebook',   $facebook_args   ) );
	$manager->register_control( new CCM_Fields_Control( $manager, 'twitter',    $twitter_args    ) );
	$manager->register_control( new CCM_Fields_Control( $manager, 'googleplus', $googleplus_args ) );
	$manager->register_control( new CCM_Fields_Control( $manager, 'linkedin',   $linkedin_args   ) );
	$manager->register_control( new CCM_Fields_Control( $manager, 'pinterest',  $pinterest_args  ) );
	$manager->register_control( new CCM_Fields_Control( $manager, 'instagram',  $instagram_args  ) );

	/* === Register Settings === */

	$manager->register_setting( 'address',    array( 'sanitize_callback' => 'wp_strip_all_tags'  ) ); // wp_strip_all_tags | absint | esc_url  (see wp sanitize)
	$manager->register_setting( 'phone',      array( 'sanitize_callback' => 'wp_strip_all_tags'  ) );
	$manager->register_setting( 'email',      array( 'sanitize_callback' => 'wp_strip_all_tags'  ) );
	$manager->register_setting( 'height',     array( 'sanitize_callback' => 'wp_strip_all_tags' ) );
	$manager->register_setting( 'bust',       array( 'sanitize_callback' => 'wp_strip_all_tags' ) );
	$manager->register_setting( 'waist',      array( 'sanitize_callback' => 'wp_strip_all_tags' ) );
	$manager->register_setting( 'hips',       array( 'sanitize_callback' => 'wp_strip_all_tags' ) );
	$manager->register_setting( 'shoe',       array( 'sanitize_callback' => 'wp_strip_all_tags' ) );
	$manager->register_setting( 'eyes',       array( 'sanitize_callback' => 'wp_strip_all_tags' ) );
	$manager->register_setting( 'facebook',   array( 'sanitize_callback' => 'esc_url' ) );
	$manager->register_setting( 'twitter',    array( 'sanitize_callback' => 'esc_url' ) );
	$manager->register_setting( 'googleplus', array( 'sanitize_callback' => 'esc_url' ) );
	$manager->register_setting( 'linkedin',   array( 'sanitize_callback' => 'esc_url' ) );
	$manager->register_setting( 'pinterest',  array( 'sanitize_callback' => 'esc_url' ) );
	$manager->register_setting( 'instagram',  array( 'sanitize_callback' => 'esc_url' ) );

}

function ccm_sanitize_textarea($content){
	$content = htmlentities( str_replace("\n", "{ccm_break}", $content), ENT_QUOTES);
	return $content;
}

/*
function ccm_sanitize_textarea($text){
    return nl2br(htmlentities($text, ENT_QUOTES));

}*/

/**
 * Returns an array of post formats allowed for the profile post type.
 *
 * @since  1.0.0
 * @access public
 * @return array
 */
function ccm_get_allowed_profile_formats() {

	return apply_filters( 'ccm_get_allowed_profile_formats', array( /*'audio', 'gallery', 'image', 'video'*/ ) );
}

/**
 * If a theme supports post formats, limit profile to only only the audio, image,
 * gallery, and video formats.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function ccm_post_format_support_filter() {

	$screen       = get_current_screen();
	$profile_type = ccm_get_profile_post_type();

	// Bail if not on the profiles screen.
	if ( empty( $screen->post_type ) || $profile_type !== $screen->post_type )
		return;

	// Check if the current theme supports formats.
	if ( current_theme_supports( 'post-formats' ) ) {

		$formats = get_theme_support( 'post-formats' );

		// If we have formats, add theme support for only the allowed formats.
		if ( isset( $formats[0] ) ) {
			$new_formats = array_intersect( $formats[0], ccm_get_allowed_profile_formats() );

			// Remove post formats support.
			remove_theme_support( 'post-formats' );

			// If the theme supports the allowed formats, add support for them.
			if ( $new_formats )
				add_theme_support( 'post-formats', $new_formats );
		}
	}

	// Filter the default post format.
	add_filter( 'option_default_post_format', 'ccm_default_post_format_filter', 95 );
}

/**
 * Filters the default post format to make sure that it's in our list of supported formats.
 *
 * @since  1.0.0
 * @access public
 * @param  string  $format
 * @return string
 */
function ccm_default_post_format_filter( $format ) {

	return in_array( $format, ccm_get_allowed_profile_formats() ) ? $format : 'standard';
}

/**
 * Help sidebar for all of the help tabs.
 *
 * @since  1.0.0
 * @access public
 * @return string
 */
function ccm_get_help_sidebar_text() {

	// Get docs and help links.
	$docs_link = sprintf( '<li><a href="http://themehybrid.com/docs">%s</a></li>', esc_html__( 'Documentation', 'custom-cotent-models' ) );
	$help_link = sprintf( '<li><a href="http://themehybrid.com/board/topics">%s</a></li>', esc_html__( 'Support Forums', 'custom-content-models' ) );

	// Return the text.
	return sprintf(
		'<p><strong>%s</strong></p><ul>%s%s</ul>',
		esc_html__( 'For more information:', 'custom-content-models' ),
		$docs_link,
		$help_link
	);
}

