<?php
/**
 * Plugin settings screen.
 *
 * @package    CustomContentModels
 * @subpackage Admin
 * @author     Justin Tadlock <justin@justintadlock.com>
 * @copyright  Copyright (c) 2013-2015, Justin Tadlock
 * @license    http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 */

/**
 * Sets up and handles the plugin settings screen.
 *
 * @since  1.0.0
 * @access public
 */
final class CCM_Settings_Page {

	/**
	 * Settings page name.
	 *
	 * @since  1.0.0
	 * @access public
	 * @var    string
	 */
	public $settings_page = '';

	/**
	 * Sets up the needed actions for adding and saving the meta boxes.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 */
	private function __construct() {
		add_action( 'admin_menu', array( $this, 'admin_menu' ) );
	}

	/**
	 * Sets up custom admin menus.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 */
	public function admin_menu() {

		// Create the settings page.
		$this->settings_page = add_submenu_page(
			'edit.php?post_type=' . ccm_get_profile_post_type(),
			esc_html__( 'Models Settings', 'custom-content-models' ),
			esc_html__( 'Settings',           'custom-content-models' ),
			apply_filters( 'ccm_settings_capability', 'manage_options' ),
			'ccm_settings',
			array( $this, 'ccm_settings_page' )
		);

		if ( $this->settings_page ) {

			// Register settings.
			add_action( 'admin_init', array( $this, 'register_settings' ) );

			// Add help tabs.
			add_action( "load-{$this->settings_page}", array( $this, 'add_help_tabs' ) );
		}
	}

	/**
	 * Registers the plugin settings.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 */
	function register_settings() {

		// Register the setting.
		register_setting( 'ccm_settings', 'ccm_settings', array( $this, 'validate_settings' ) );

		/* === Settings Sections === */

		add_settings_section( 'general',    esc_html__( 'General Settings', 'custom-content-models' ), array( $this, 'section_general'    ), $this->settings_page );
		add_settings_section( 'permalinks', esc_html__( 'Permalinks',       'custom-content-models' ), array( $this, 'section_permalinks' ), $this->settings_page );

		/* === Settings Fields === */

		// General section fields
		add_settings_field( 'models_title',       esc_html__( 'Title',       'custom-content-models' ), array( $this, 'field_models_title'       ), $this->settings_page, 'general' );
		add_settings_field( 'models_description', esc_html__( 'Description', 'custom-content-models' ), array( $this, 'field_models_description' ), $this->settings_page, 'general' );
		add_settings_field( 'models_per_page', esc_html__( 'Profiles per page', 'custom-content-models' ), array( $this, 'field_models_per_page' ), $this->settings_page, 'general' );
		add_settings_field( 'models_layout', esc_html__( 'Models category layout', 'custom-content-models' ), array( $this, 'field_models_layout_choose' ), $this->settings_page, 'general' );

		// Permalinks section fields.
		add_settings_field( 'models_rewrite_base', esc_html__( 'Models Base', 'custom-content-models' ), array( $this, 'field_models_rewrite_base' ), $this->settings_page, 'permalinks' );
		add_settings_field( 'profile_rewrite_base',   esc_html__( 'Profile Slug',   'custom-content-models' ), array( $this, 'field_profile_rewrite_base'   ), $this->settings_page, 'permalinks' );
		add_settings_field( 'category_rewrite_base',  esc_html__( 'Category Slug',  'custom-content-models' ), array( $this, 'field_category_rewrite_base'  ), $this->settings_page, 'permalinks' );
		add_settings_field( 'tag_rewrite_base',       esc_html__( 'Tag Slug',       'custom-content-models' ), array( $this, 'field_tag_rewrite_base'       ), $this->settings_page, 'permalinks' );
		add_settings_field( 'author_rewrite_base',    esc_html__( 'Author Slug',    'custom-content-models' ), array( $this, 'field_author_rewrite_base'    ), $this->settings_page, 'permalinks' );
	}

	/**
	 * Validates the plugin settings.
	 *
	 * @since  1.0.0
	 * @access public
	 * @param  array  $input
	 * @return array
	 */
	function validate_settings( $settings ) {

		// Text boxes.
		$settings['models_rewrite_base'] = $settings['models_rewrite_base'] ? trim( strip_tags( $settings['models_rewrite_base'] ), '/' ) : 'models';
		$settings['profile_rewrite_base']   = $settings['profile_rewrite_base']   ? trim( strip_tags( $settings['profile_rewrite_base']   ), '/' ) : '';
		$settings['category_rewrite_base']  = $settings['category_rewrite_base']  ? trim( strip_tags( $settings['category_rewrite_base']  ), '/' ) : '';
		$settings['tag_rewrite_base']       = $settings['tag_rewrite_base']       ? trim( strip_tags( $settings['tag_rewrite_base']       ), '/' ) : '';
		$settings['author_rewrite_base']    = $settings['author_rewrite_base']    ? trim( strip_tags( $settings['author_rewrite_base']    ), '/' ) : '';
		$settings['models_title']        = $settings['models_title']        ? strip_tags( $settings['models_title'] )                     : esc_html__( 'Models', 'custom-content-models' );
		$settings['models_layout'] = $settings['models_layout'] ? trim( strip_tags( $settings['models_layout'] ), '/' ) : 'style_1';

		// Kill evil scripts.
		$settings['models_description'] = stripslashes( wp_filter_post_kses( addslashes( $settings['models_description'] ) ) );

		/* === Handle Permalink Conflicts ===*/

		// No profile or category base, profiles win.
		if ( ! $settings['profile_rewrite_base'] && ! $settings['category_rewrite_base'] )
			$settings['category_rewrite_base'] = 'categories';

		// No profile or tag base, profiles win.
		if ( ! $settings['profile_rewrite_base'] && ! $settings['tag_rewrite_base'] )
			$settings['tag_rewrite_base'] = 'tags';

		// No profile or author base, profiles win.
		if ( ! $settings['profile_rewrite_base'] && ! $settings['author_rewrite_base'] )
			$settings['author_rewrite_base'] = 'authors';

		// No category or tag base, categories win.
		if ( ! $settings['category_rewrite_base'] && ! $settings['tag_rewrite_base'] )
			$settings['tag_rewrite_base'] = 'tags';

		// No category or author base, categories win.
		if ( ! $settings['category_rewrite_base'] && ! $settings['author_rewrite_base'] )
			$settings['author_rewrite_base'] = 'authors';

		// No author or tag base, authors win.
		if ( ! $settings['author_rewrite_base'] && ! $settings['tag_rewrite_base'] )
			$settings['tag_rewrite_base'] = 'tags';

		// Return the validated/sanitized settings.
		return $settings;
	}

	/**
	 * General section callback.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 */
	public function section_general() { ?>

		<p class="description">
			<?php esc_html_e( 'General models settings for your site.', 'custom-content-models' ); ?>
		</p>
	<?php }

	/**
	 * Models title field callback.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 */
	public function field_models_title() { ?>

		<label>
			<input type="text" class="regular-text" name="ccm_settings[models_title]" value="<?php echo esc_attr( ccm_get_models_title() ); ?>" />
			<br />
			<span class="description"><?php esc_html_e( 'The name of your models. May be used for the models page title and other places, depending on your theme.', 'custom-content-models' ); ?></span>
		</label>
	<?php }

	/**
	 * Models description field callback.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 */
	public function field_models_description() {

		wp_editor(
			ccm_get_models_description(),
			'ccm_models_description',
			array(
				'textarea_name'    => 'ccm_settings[models_description]',
				'drag_drop_upload' => true,
				'editor_height'    => 150
			)
		); ?>

		<p>
			<span class="description"><?php esc_html_e( 'Your models description. This may be shown by your theme on the models page.', 'custom-content-models' ); ?></span>
		</p>
	<?php }

	/**
	 * Models title field callback.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 */
	public function field_models_per_page() { ?>

		<label>
			<input type="number" class="regular-text" name="ccm_settings[models_per_page]" value="<?php echo esc_attr( ccm_get_models_per_page() ); ?>" />
			<br />
			<span class="description"><?php esc_html_e( 'Choose models profiles count.', 'custom-content-models' ); ?></span>
		</label>
	<?php }

	/**
	 * Models layout choose.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 */
	public function field_models_layout_choose() { ?>
		<label>
			<input class="radio" type="radio"  name="ccm_settings[models_layout]" value="style_1" <?php checked( ccm_get_models_layout() == 'style_1' ); ?> /><?php esc_html_e( 'Style 1', 'custom-content-models' ); ?><br/>
			<input class="radio" type="radio"  name="ccm_settings[models_layout]" value="style_2" <?php checked( ccm_get_models_layout() == 'style_2' ); ?> /><?php esc_html_e( 'Style 2', 'custom-content-models' ); ?>
			<br />
			<span class="description"><?php esc_html_e( 'Choose models category layout.', 'custom-content-models' ); ?></span>
		</label>
	<?php }

	/**
	 * Permalinks section callback.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 */
	public function section_permalinks() { ?>

		<p class="description">
			<?php esc_html_e( 'Set up custom permalinks for the models section on your site.', 'custom-content-models' ); ?>
		</p>
	<?php }

	/**
	 * Models rewrite base field callback.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 */
	public function field_models_rewrite_base() { ?>

		<label>
			<code><?php echo esc_url( home_url( '/' ) ); ?></code>
			<input type="text" class="regular-text code" name="ccm_settings[models_rewrite_base]" value="<?php echo esc_attr( ccm_get_models_rewrite_base() ); ?>" />
		</label>
	<?php }

	/**
	 * Models rewrite base field callback.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 */
	public function field_profile_rewrite_base() { ?>

		<label>
			<code><?php echo esc_url( home_url( ccm_get_models_rewrite_base() . '/' ) ); ?></code>
			<input type="text" class="regular-text code" name="ccm_settings[profile_rewrite_base]" value="<?php echo esc_attr( ccm_get_profile_rewrite_base() ); ?>" />
		</label>
	<?php }

	/**
	 * Models rewrite base field callback.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 */
	public function field_category_rewrite_base() { ?>

		<label>
			<code><?php echo esc_url( home_url( ccm_get_models_rewrite_base() . '/' ) ); ?></code>
			<input type="text" class="regular-text code" name="ccm_settings[category_rewrite_base]" value="<?php echo esc_attr( ccm_get_category_rewrite_base() ); ?>" />
		</label>
	<?php }

	/**
	 * Models rewrite base field callback.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 */
	public function field_tag_rewrite_base() { ?>

		<label>
			<code><?php echo esc_url( home_url( ccm_get_models_rewrite_base() . '/' ) ); ?></code>
			<input type="text" class="regular-text code" name="ccm_settings[tag_rewrite_base]" value="<?php echo esc_attr( ccm_get_tag_rewrite_base() ); ?>" />
		</label>
	<?php }

	/**
	 * Author rewrite base field callback.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 */
	public function field_author_rewrite_base() { ?>

		<label>
			<code><?php echo esc_url( home_url( ccm_get_models_rewrite_base() . '/' ) ); ?></code>
			<input type="text" class="regular-text code" name="ccm_settings[author_rewrite_base]" value="<?php echo esc_attr( ccm_get_author_rewrite_base() ); ?>" />
		</label>
	<?php }

	/**
	 * Renders the settings page.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 */
	public function ccm_settings_page() {

		// Flush the rewrite rules if the settings were updated.
		if ( isset( $_GET['settings-updated'] ) )
			flush_rewrite_rules(); ?>

		<div class="wrap">
			<h1><?php esc_html_e( 'Models Settings', 'custom-content-models' ); ?></h1>

			<?php settings_errors(); ?>

			<form method="post" action="options.php">
				<?php settings_fields( 'ccm_settings' ); ?>
				<?php do_settings_sections( $this->settings_page ); ?>
				<?php submit_button( esc_attr__( 'Update Settings', 'custom-content-models' ), 'primary' ); ?>
			</form>

		</div><!-- wrap -->
	<?php }

	/**
	 * Adds help tabs.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 */
	public function add_help_tabs() {

		// Get the current screen.
		$screen = get_current_screen();

		// General settings help tab.
		$screen->add_help_tab(
			array(
				'id'       => 'general',
				'title'    => esc_html__( 'General Settings', 'custom-content-models' ),
				'callback' => array( $this, 'help_tab_general' )
			)
		);

		// Permalinks settings help tab.
		$screen->add_help_tab(
			array(
				'id'       => 'permalinks',
				'title'    => esc_html__( 'Permalinks', 'custom-content-models' ),
				'callback' => array( $this, 'help_tab_permalinks' )
			)
		);

		// Set the help sidebar.
		$screen->set_help_sidebar( ccm_get_help_sidebar_text() );
	}

	/**
	 * Displays the general settings help tab.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 */
	public function help_tab_general() { ?>

		<ul>
			<li><?php _e( '<strong>Title:</strong> Allows you to set the title for the models section on your site. This is general shown on the models profiles archive, but themes and other plugins may use it in other ways.', 'custom-content-models' ); ?></li>
			<li><?php _e( '<strong>Description:</strong> This is the description for your models. Some themes may display this on the models profiles archive.', 'custom-content-models' ); ?></li>
		</ul>
	<?php }

	/**
	 * Displays the permalinks help tab.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 */
	public function help_tab_permalinks() { ?>

		<ul>
			<li><?php _e( '<strong>Models Base:</strong> The primary URL for the models section on your site. It lists your models profiles.', 'custom-content-models' ); ?></li>
			<li>
				<?php _e( '<strong>Profile Slug:</strong> The slug for single models profiles. You can use something custom, leave this field empty, or use one of the following tags:', 'custom-content-models' ); ?>
				<ul>
					<li><?php printf( esc_html__( '%s - The profile author name.', 'custom-content-models' ), '<code>%author%</code>' ); ?></li>
					<li><?php printf( esc_html__( '%s - The profile category.', 'custom-content-models' ), '<code>%' . ccm_get_category_taxonomy() . '%</code>' ); ?></li>
					<li><?php printf( esc_html__( '%s - The profile tag.', 'custom-content-models' ), '<code>%' . ccm_get_tag_taxonomy() . '%</code>' ); ?></li>
				</ul>
			</li>
			<li><?php _e( '<strong>Category Slug:</strong> The base slug used for models category archives.', 'custom-content-models' ); ?></li>
			<li><?php _e( '<strong>Tag Slug:</strong> The base slug used for models tag archives.', 'custom-content-models' ); ?></li>
			<li><?php _e( '<strong>Author Slug:</strong> The base slug used for models author archives.', 'custom-content-models' ); ?></li>
		</ul>
	<?php }

	/**
	 * Returns the instance.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return object
	 */
	public static function get_instance() {

		static $instance = null;

		if ( is_null( $instance ) )
			$instance = new self;

		return $instance;
	}
}

CCM_Settings_Page::get_instance();
