<?php
/**
 * Custom functions for entry.
 *
 * @package Grd
 */

/**
 * Prints HTML with meta information for the current post-date/time and author.
 *
 * @since 1.0.0
 */
function grd_posted_on() {
	$time_string = '<time class="entry-date published updated">%s</time>';
	$time_string = sprintf( $time_string,
		esc_html( get_the_modified_date() )
	);
	$posted_on   = sprintf(
		_x( '%s', 'post date', 'grd' ),
		'<a href="' . esc_url( get_permalink() ) . '" rel="bookmark">' . $time_string . '</a>'
	);
	$byline      = sprintf(
		_x( 'by %s', 'post author', 'grd' ),
		'<span class="author vcard"><a class="url fn n" href="' . esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ) . '">' . esc_html( get_the_author() ) . '</a></span>'
	);
	$category    = get_the_category();
	$catName     = $category[0]->cat_name;
	$catUrl      = get_category_link( $category[0]->term_id );
	$byCat       = sprintf(
		_x( 'in %s', 'post cat', 'grd' ),
		'<a href="' . $catUrl . '">' . $catName . '</a>'
	);
	echo '<span class="posted-on">' . $posted_on . '</span><span class="sep"> /</span><span class="byline"> ' . $byline . '</span><span class="sep"> /</span><span class="bycat"> ' . $byCat . '</span>';
}

/**
 * Prints HTML with meta information for the categories, tags and comments.
 *
 * @since 1.0.0
 */
function grd_entry_footer() {
	// Hide category and tag text for pages.
	if ( 'post' == get_post_type() ) {
		/* translators: used between list items, there is a space after the comma */
		$tags_list = get_the_tag_list( '', '<span>,</span> ' );
		if ( $tags_list ) {
			printf( '<span class="tags-links">' . esc_html__( 'Tags: %1$s', 'grd' ) . '</span>', $tags_list );
		}
	}

	if ( grd_get_option( 'post_share_box' ) ):
		printf( '<div class="footer-socials">' );
		printf( '<span><span class="svg-icon icon-share"><svg><use xlink:href="#share"></use></svg></span> %s: </span>', esc_html__( 'Share', 'grd' ) );
		$image = get_the_post_thumbnail( 'full' );
		grd_share_link_socials( get_the_title(), get_permalink(), $image );
		printf( '</div>' );
	endif;

	if ( ! is_single() && ! post_password_required() && ( comments_open() || get_comments_number() ) ) {
		echo '<span class="comments-link">';
		comments_popup_link( esc_html__( 'Leave a comment', 'grd' ), esc_html__( '1 Comment', 'grd' ), esc_html__( '% Comments', 'grd' ) );
		echo '</span>';
	}
	edit_post_link( esc_html__( 'Edit', 'grd' ), '<span class="edit-link">', '</span>' );
}

/**
 * Get or display limited words from given string.
 * Strips all tags and shortcodes from string.
 *
 * @since 1.0.0
 *
 * @param integer $num_words The maximum number of words
 * @param string $more More link.
 * @param bool $echo Echo or return output
 *
 * @return string|void Limited content.
 */
function grd_content_limit( $content, $num_words, $more = "&hellip;" ) {

	// Strip tags and shortcodes so the content truncation count is done correctly
	$content = strip_tags( strip_shortcodes( $content ), apply_filters( 'grd_content_limit_allowed_tags', '<script>,<style>' ) );

	// Remove inline styles / scripts
	$content = trim( preg_replace( '#<(s(cript|tyle)).*?</\1>#si', '', $content ) );

	if ( grd_is_service() ) {
		$num_words = grd_get_option( 'service_limit' );
	} else {
		$num_words = grd_get_option( 'blog_classic_excerpt_length' );
	}

	// Truncate $content to $max_char
	$content = wp_trim_words( $content, $num_words );

	if ( $more ) {
		$output = sprintf(
			'<p>%s</p>',
			$content
		);
	} else {
		$output = sprintf( '<p>%s</p>', $content );
	}

	return $output;
}


/**
 * Show entry thumbnail base on its format
 *
 * @since  1.0
 */
function grd_entry_thumbnail( $size = 'thumbnail' ) {
	$html      = '';
	$css_class = 'format-' . get_post_format();

	switch ( get_post_format() ) {
		case 'gallery':
			$images = grd_get_meta( 'images', "type=image&size=$size" );

			if ( empty( $images ) ) {
				break;
			}

			$gallery = array();
			foreach ( $images as $image ) {
				$gallery[] = '<li>' . '<img src="' . esc_url( $image['url'] ) . '" alt="' . the_title_attribute( 'echo=0' ) . '">' . '</li>';
			}
			$html .= '<div class="format-gallery-slider entry-image"><ul class="slides">' . implode( '', $gallery ) . '</ul></div>';
			break;

		case 'video':
			$video = grd_get_meta( 'video' );
			if ( ! $video ) {
				break;
			}

			// If URL: show oEmbed HTML
			if ( filter_var( $video, FILTER_VALIDATE_URL ) ) {
				if ( $oembed = @wp_oembed_get( $video ) ) {
					$html .= $oembed;
				} else {
					$atts = array(
						'src'   => $video,
						'width' => 848,
					);
					if ( has_post_thumbnail() ) {
						$atts['poster'] = claudio_get_image( 'format=src&echo=0&size=full' );
					}
					$html .= wp_video_shortcode( $atts );
				}
			} // If embed code: just display
			else {
				$html .= $video;
			}
			break;

		default:
			$thumb = get_the_post_thumbnail( get_the_ID(), $size );
			if ( empty( $thumb ) ) {
				break;
			}

			$html .= '<a class="entry-image" href="' . get_permalink() . '">' . $thumb . '</a>';
			break;
	}

	if ( $html = apply_filters( __FUNCTION__, $html, get_post_format() ) ) {
		$css_class = esc_attr( $css_class );
		echo "<div class='entry-format $css_class'>$html</div>";
	}
}

/**
 * Share link socials
 *
 * @since  1.0
 */

if ( ! function_exists( 'grd_share_link_socials' ) ) :
	function grd_share_link_socials( $title, $link, $media ) {
		$socials = grd_get_option( 'product_social_icons' );
		if ( is_singular( 'post' ) ) {
			$socials = grd_get_option( 'post_social_icons' );
		} elseif ( is_singular( 'portfolio_project' ) ) {
			$socials = grd_get_option( 'portfolio_social_icons' );
		}

		$socials_html = '';
		if ( $socials ) {
			if ( in_array( 'facebook', $socials ) ) {
				$socials_html .= sprintf(
					'<a class="share-facebook grd-facebook" title="%s" href="http://www.facebook.com/sharer.php?u=%s&t=%s" target="_blank"><i class="fa fa-facebook-f"></i></a>',
					esc_attr( $title ),
					urlencode( $link ),
					urlencode( $title )
				);
			}

			if ( in_array( 'twitter', $socials ) ) {
				$socials_html .= sprintf(
					'<a class="share-twitter grd-twitter" href="http://twitter.com/share?text=%s&url=%s" title="%s" target="_blank"><i class="fa fa-twitter"></i></a>',
					esc_attr( $title ),
					urlencode( $link ),
					urlencode( $title )
				);
			}

			if ( in_array( 'pinterest', $socials ) ) {
				$socials_html .= sprintf(
					'<a class="share-pinterest grd-pinterest" href="http://pinterest.com/pin/create/button?media=%s&url=%s&description=%s" title="%s" target="_blank"><i class="fa fa-pinterest-p"></i></a>',
					urlencode( $media ),
					urlencode( $link ),
					esc_attr( $title ),
					urlencode( $title )
				);
			}

			if ( in_array( 'google', $socials ) ) {
				$socials_html .= sprintf(
					'<a class="share-google-plus grd-google-plus" href="https://plus.google.com/share?url=%s&text=%s" title="%s" target="_blank"><i class="fa fa-google-plus"></i></a>',
					urlencode( $link ),
					esc_attr( $title ),
					urlencode( $title )
				);
			}

			if ( in_array( 'linkedin', $socials ) ) {
				$socials_html .= sprintf(
					'<a class="share-linkedin grd-linkedin" href="http://www.linkedin.com/shareArticle?url=%s&title=%s" title="%s" target="_blank"><i class="fa fa-linkedin"></i></a>',
					urlencode( $link ),
					esc_attr( $title ),
					urlencode( $title )
				);
			}

			if ( in_array( 'vkontakte', $socials ) ) {
				$socials_html .= sprintf(
					'<a class="share-vkontakte grd-vkontakte" href="http://vk.com/share.php?url=%s&title=%s&image=%s" title="%s" target="_blank"><i class="fa fa-vk"></i></a>',
					urlencode( $link ),
					esc_attr( $title ),
					urlencode( $media ),
					urlencode( $title )
				);
			}

		}

		if ( $socials_html ) {
			printf( '<div class="social-links">%s</div>', $socials_html );
		}
		?>
		<?php
	}

endif;

/**
 * Get single entry meta
 *
 * @since  1.0
 *
 */
function grd_entry_meta() {
	$fields = (array) grd_get_option( 'blog_entry_meta' );


	if ( empty ( $fields ) ) {
		return;
	}

	foreach ( $fields as $field ) {
		switch ( $field ) {
			case 'date':
				$month = esc_html( get_the_date() );

				echo '<div class="meta-date published updated">' . $month . '</div>';

				break;

			case 'author':
				echo '<span class="meta-author"><span>' . esc_html__( 'By ', 'grd' ) . '</span><a class="url fn n" href="' . esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) )
				     . '">' . esc_html( get_the_author() ) . '</a></span>';
				break;

			case 'cat':
				echo '<span class="meta-cat">';
				the_category( ', ' );
				echo '</span>';
				break;

			case 'comment':
				if ( ! post_password_required() && ( comments_open() || get_comments_number() ) ) {
					echo '<span class="meta-comment">';
					if ( get_comments_number() ) {
						echo sprintf( '%s %s', number_format_i18n( get_comments_number() ), esc_html__( 'Comments', 'grd' ) );
					} else {
						esc_html_e( '0 Comment', 'grd' );
					}

					echo '</span>';
				}

				break;
		}
	}
}


/**
 * Check is service
 *
 * @since  1.0
 */

if ( ! function_exists( 'grd_service_title' ) ) {
	function grd_service_title() { ?>
        <a href="<?php echo the_permalink(); ?>" class="entry-title text-center">
            <p class="col-xs-12 no-padding title"><?php echo get_the_title(); ?></p>
        </a>
		<?php
	}
}

/**
 * Check is service
 *
 * @since  1.0
 */
if ( ! function_exists( 'grd_service_descr' ) ) {
	function grd_service_descr( $number_ex ) {
		$desc = get_the_excerpt();
		if ( ! empty( $desc ) ) {
			?>
            <div class="descr">
				<?php
				echo grd_content_limit( $desc, $number_ex );
				?>
            </div>
			<?php
		}
	}
}

/**
 * Check is service
 *
 * @since  1.0
 */
if ( ! function_exists( 'grd_content_descr' ) ) {
	function grd_content_descr( $number_ex ) {
		$desc = get_the_excerpt();
		if ( ! empty( $desc ) ) {
			?>
            <div class="descr">
				<?php
				echo grd_content_limit( $desc, $number_ex );
				?>
            </div>
			<?php
		}
	}
}


/**
 * Get socials
 *
 * @since  1.0.0
 *
 *
 * @return string
 */
function grd_get_socials() {
	$socials = array(
		'facebook'   => esc_html__( 'Facebook', 'grd' ),
		'twitter'    => esc_html__( 'Twitter', 'grd' ),
		'google'     => esc_html__( 'Google', 'grd' ),
		'tumblr'     => esc_html__( 'Tumblr', 'grd' ),
		'flickr'     => esc_html__( 'Flickr', 'grd' ),
		'vimeo'      => esc_html__( 'Vimeo', 'grd' ),
		'youtube'    => esc_html__( 'Youtube', 'grd' ),
		'linkedin'   => esc_html__( 'LinkedIn', 'grd' ),
		'pinterest'  => esc_html__( 'Pinterest', 'grd' ),
		'dribbble'   => esc_html__( 'Dribbble', 'grd' ),
		'spotify'    => esc_html__( 'Spotify', 'grd' ),
		'instagram'  => esc_html__( 'Instagram', 'grd' ),
		'tumbleupon' => esc_html__( 'Tumbleupon', 'grd' ),
		'wordpress'  => esc_html__( 'WordPress', 'grd' ),
		'rss'        => esc_html__( 'Rss', 'grd' ),
		'deviantart' => esc_html__( 'Deviantart', 'grd' ),
		'share'      => esc_html__( 'Share', 'grd' ),
		'skype'      => esc_html__( 'Skype', 'grd' ),
		'behance'    => esc_html__( 'Behance', 'grd' ),
		'apple'      => esc_html__( 'Apple', 'grd' ),
		'yelp'       => esc_html__( 'Yelp', 'grd' ),
	);

	return apply_filters( 'grd_header_socials', $socials );
}

if ( ! function_exists( 'grd_get_page_header_image' ) ) :

	/**
	 * Get page header image URL
	 *
	 * @return string
	 */
	function grd_get_page_header_image() {
		if ( grd_get_post_meta( 'hide_page_header' ) ) {
			return false;
		}

		if ( is_404() || is_page_template( 'template-coming-soon.php' ) ) {
			return false;
		}

		if ( is_page_template( array( 'template-homepage.php' ) ) ) {
			return false;
		}
		$elements = array( 'title', 'breadcrumb' );

		$page_header = array(
			'bg_image'      => '',
			'br_color'      => '',
			'br_link_color' => '',
			'color_image'   => '',
			'title_color'   => '',
			'parallax'      => false,
			'elements'      => $elements
		);


		if ( grd_is_portfolio() ) {
			if ( ! intval( grd_get_option( 'portfolio_page_header' ) ) ) {
				return false;
			}

			$elements                     = grd_get_option( 'portfolio_page_header_els' );
			$page_header['bg_image']      = grd_get_option( 'portfolio_page_header_bg' );
			$page_header['title_color']   = grd_get_option( 'portfolio_page_header_title_color' );
			$page_header['br_color']      = grd_get_option( 'portfolio_page_header_breadcrumb_color' );
			$page_header['br_link_color'] = grd_get_option( 'portfolio_page_header_breadcrumb_link_color' );
			$page_header['color_image']   = get_theme_mod( 'portfolio_page_header_color', '' );
			$page_header['parallax']      = intval( grd_get_option( 'portfolio_page_header_parallax' ) );

		} elseif ( grd_is_service() ) {
			if ( ! intval( grd_get_option( 'service_page_header' ) ) ) {
				return false;
			}

			$elements                     = grd_get_option( 'service_page_header_els' );
			$page_header['bg_image']      = grd_get_option( 'service_page_header_bg' );
			$page_header['title_color']   = grd_get_option( 'service_page_header_title_color' );
			$page_header['br_color']      = grd_get_option( 'service_page_header_breadcrumb_color' );
			$page_header['br_link_color'] = grd_get_option( 'service_page_header_breadcrumb_link_color' );
			$page_header['color_image']   = get_theme_mod( 'service_page_header_color', '' );
			$page_header['parallax']      = intval( grd_get_option( 'service_page_header_parallax' ) );

		} elseif ( grd_is_catalog() ) {
			if ( ! intval( grd_get_option( 'page_header_shop' ) ) ) {
				return false;
			}

			$elements                     = grd_get_option( 'page_header_shop_els' );
			$page_header['bg_image']      = grd_get_option( 'page_header_shop_bg' );
			$page_header['title_color']   = grd_get_option( 'page_header_shop_title_color' );
			$page_header['br_color']      = grd_get_option( 'page_header_shop_breadcrumb_color' );
			$page_header['br_link_color'] = grd_get_option( 'page_header_shop_breadcrumb_link_color' );
			$page_header['color_image']   = get_theme_mod( 'page_header_shop_color', '' );
			$page_header['parallax']      = intval( grd_get_option( 'page_header_shop_parallax' ) );

		} elseif ( function_exists( 'is_product' ) && is_product() ) {
			if ( ! intval( grd_get_option( 'single_product_page_header' ) ) ) {
				return false;
			}

			$elements                     = grd_get_option( 'single_product_page_header_els' );
			$page_header['bg_image']      = grd_get_option( 'single_product_page_header_bg' );
			$page_header['title_color']   = grd_get_option( 'single_product_page_header_title_color' );
			$page_header['br_color']      = grd_get_option( 'single_product_page_header_breadcrumb_color' );
			$page_header['br_link_color'] = grd_get_option( 'single_product_page_header_breadcrumb_link_color' );
			$page_header['color_image']   = get_theme_mod( 'single_product_page_header_color', '' );
			$page_header['parallax']      = intval( grd_get_option( 'single_product_page_header_parallax' ) );

		} elseif ( is_singular( 'post' )
		           || is_singular( 'portfolio' )
		           || is_singular( 'service' )
		           || is_page()
		           || is_page_template( 'template-fullwidth.php' )
		) {
			if ( is_singular( 'post' ) ) {
				if ( ! intval( grd_get_option( 'single_page_header' ) ) ) {
					return false;
				}

				$elements      = grd_get_option( 'single_page_header_els' );
				$bg_image      = grd_get_option( 'single_page_header_bg' );
				$title_color   = grd_get_option( 'single_page_header_title_color' );
				$br_color      = grd_get_option( 'single_page_header_breadcrumb_color' );
				$br_link_color = grd_get_option( 'single_page_header_breadcrumb_link_color' );
				$color_image   = get_theme_mod( 'single_page_header_color', '' );
				$parallax      = intval( grd_get_option( 'single_page_header_parallax' ) );

				if ( get_post_meta( get_the_ID(), 'hide_page_header', true ) ) {
					return false;
				}
				if ( get_post_meta( get_the_ID(), 'hide_title', true ) ) {
					$key = array_search( 'title', $elements );
					if ( $key !== false ) {
						unset( $elements[ $key ] );
					}
				}
				if ( get_post_meta( get_the_ID(), 'hide_breadcrumb', true ) ) {
					$key = array_search( 'breadcrumb', $elements );
					if ( $key !== false ) {
						unset( $elements[ $key ] );
					}
				}
				if ( get_post_meta( get_the_ID(), 'hide_parallax', true ) ) {
					unset( $parallax );
				}
				if ( get_post_meta( get_the_ID(), 'page_bg', true ) ) {
					$meta_page_bg = get_post_meta( get_the_ID(), 'page_bg', true );
					$bg_image     = wp_get_attachment_url( $meta_page_bg );
				}
			} elseif ( is_singular( 'portfolio' ) ) {
				if ( ! intval( grd_get_option( 'single_portfolio_page_header' ) ) ) {
					return false;
				}

				$elements      = grd_get_option( 'single_portfolio_page_header_els' );
				$bg_image      = grd_get_option( 'single_portfolio_page_header_bg' );
				$title_color   = grd_get_option( 'single_portfolio_page_header_title_color' );
				$br_color      = grd_get_option( 'single_portfolio_page_header_breadcrumb_color' );
				$br_link_color = grd_get_option( 'single_portfolio_page_header_breadcrumb_link_color' );
				$color_image   = get_theme_mod( 'single_portfolio_page_header_color', '' );
				$parallax      = intval( grd_get_option( 'single_portfolio_page_header_parallax' ) );

				if ( get_post_meta( get_the_ID(), 'hide_page_header', true ) ) {
					return false;
				}
				if ( get_post_meta( get_the_ID(), 'hide_title', true ) ) {
					$key = array_search( 'title', $elements );
					if ( $key !== false ) {
						unset( $elements[ $key ] );
					}
				}
				if ( get_post_meta( get_the_ID(), 'hide_breadcrumb', true ) ) {
					$key = array_search( 'breadcrumb', $elements );
					if ( $key !== false ) {
						unset( $elements[ $key ] );
					}
				}
				if ( get_post_meta( get_the_ID(), 'hide_parallax', true ) ) {
					unset( $parallax );
				}
				if ( get_post_meta( get_the_ID(), 'page_bg', true ) ) {
					$meta_page_bg = get_post_meta( get_the_ID(), 'page_bg', true );
					$bg_image     = wp_get_attachment_url( $meta_page_bg );
				}

			} elseif ( is_singular( 'service' ) ) {
				if ( ! intval( grd_get_option( 'single_service_page_header' ) ) ) {
					return false;
				}

				$elements      = grd_get_option( 'single_service_page_header_els' );
				$bg_image      = grd_get_option( 'single_service_page_header_bg' );
				$title_color   = grd_get_option( 'single_service_page_header_title_color' );
				$br_color      = grd_get_option( 'single_service_page_header_breadcrumb_color' );
				$br_link_color = grd_get_option( 'single_service_page_header_breadcrumb_link_color' );
				$color_image   = get_theme_mod( 'single_service_page_header_color', '' );
				$parallax      = intval( grd_get_option( 'single_service_page_header_parallax' ) );

				if ( get_post_meta( get_the_ID(), 'hide_page_header', true ) ) {
					return false;
				}
				if ( get_post_meta( get_the_ID(), 'hide_title', true ) ) {
					$key = array_search( 'title', $elements );
					if ( $key !== false ) {
						unset( $elements[ $key ] );
					}
				}
				if ( get_post_meta( get_the_ID(), 'hide_breadcrumb', true ) ) {
					$key = array_search( 'breadcrumb', $elements );
					if ( $key !== false ) {
						unset( $elements[ $key ] );
					}
				}
				if ( get_post_meta( get_the_ID(), 'hide_parallax', true ) ) {
					unset( $parallax );
				}
				if ( get_post_meta( get_the_ID(), 'page_bg', true ) ) {
					$meta_page_bg = get_post_meta( get_the_ID(), 'page_bg', true );
					$bg_image     = wp_get_attachment_url( $meta_page_bg );
				}

			} elseif ( is_page() || is_page_template( 'template-fullwidth.php' ) ) {
				if ( ! intval( grd_get_option( 'page_page_header' ) ) ) {
					return false;
				}

				$elements      = grd_get_option( 'page_page_header_els' );
				$bg_image      = grd_get_option( 'page_page_header_bg' );
				$title_color   = grd_get_option( 'page_page_header_title_color' );
				$br_color      = grd_get_option( 'page_page_header_breadcrumb_color' );
				$br_link_color = grd_get_option( 'page_page_header_breadcrumb_link_color' );
				$color_image   = get_theme_mod( 'page_page_header_color', '' );
				$parallax      = intval( grd_get_option( 'page_page_header_parallax' ) );

				if ( get_post_meta( get_the_ID(), 'hide_page_header', true ) ) {
					return false;
				}
				if ( get_post_meta( get_the_ID(), 'hide_title', true ) ) {
					$key = array_search( 'title', $elements );
					if ( $key !== false ) {
						unset( $elements[ $key ] );
					}
				}
				if ( get_post_meta( get_the_ID(), 'hide_breadcrumb', true ) ) {
					$key = array_search( 'breadcrumb', $elements );
					if ( $key !== false ) {
						unset( $elements[ $key ] );
					}
				}
				if ( get_post_meta( get_the_ID(), 'hide_parallax', true ) ) {
					unset( $parallax );
				}
				if ( get_post_meta( get_the_ID(), 'page_bg', true ) ) {
					$meta_page_bg = get_post_meta( get_the_ID(), 'page_bg', true );
					$bg_image     = wp_get_attachment_url( $meta_page_bg );
				}
			}


			$page_header['bg_image']      = $bg_image;
			$page_header['title_color']   = $title_color;
			$page_header['br_color']      = $br_color;
			$page_header['br_link_color'] = $br_link_color;
			$page_header['color_image']   = $color_image;
			$page_header['parallax']      = $parallax;
		} else {
			if ( ! intval( grd_get_option( 'blog_page_header' ) ) ) {
				return false;
			}

			$elements                     = grd_get_option( 'blog_page_header_els' );
			$page_header['bg_image']      = grd_get_option( 'blog_page_header_bg' );
			$page_header['title_color']   = grd_get_option( 'blog_page_header_title_color' );
			$page_header['br_color']      = grd_get_option( 'blog_page_header_breadcrumb_color' );
			$page_header['br_link_color'] = grd_get_option( 'blog_page_header_breadcrumb_link_color' );
			$page_header['color_image']   = get_theme_mod( 'blog_page_header_color', '' );
			$page_header['parallax']      = intval( grd_get_option( 'blog_page_header_parallax' ) );
		}

		if ( grd_get_post_meta( 'hide_title' ) ) {
			$key = array_search( 'title', $elements );
			if ( $key !== false ) {
				unset( $elements[ $key ] );
			}
		}
		if ( grd_get_post_meta( 'hide_breadcrumb' ) ) {
			$key = array_search( 'breadcrumb', $elements );
			if ( $key !== false ) {
				unset( $elements[ $key ] );
			}
		}
		if ( grd_get_post_meta( 'hide_parallax' ) ) {
			unset( $page_header['parallax'] );
		}
		if ( grd_get_post_meta( 'page_bg' ) ) {
			$meta_page_bg            = grd_get_post_meta( 'page_bg' );
			$page_header['bg_image'] = wp_get_attachment_url( $meta_page_bg );
		}

		$page_header['elements'] = $elements;


		return $page_header;
	}
endif;

/**
 * Get blog meta
 *
 * @since  1.0
 *
 * @return string
 */
if ( ! function_exists( 'grd_get_post_meta' ) ) :
	function grd_get_post_meta( $meta ) {

		$post_id = get_the_id();
		if ( is_home() && ! is_front_page() ) {
			$post_id = get_queried_object_id();
		} elseif ( function_exists( 'is_shop' ) && is_shop() ) {
			$post_id = intval( get_option( 'woocommerce_shop_page_id' ) );
		} elseif ( is_post_type_archive( 'portfolio' ) ) {
			$post_id = intval( get_option( 'grd_portfolio_page_id' ) );
		} elseif ( is_post_type_archive( 'service' ) ) {
			$post_id = intval( get_option( 'grd_service_page_id' ) );
		}

		return $post_id ? get_post_meta( $post_id, $meta, true ) : false;

	}
endif;


/**
 * Check is services
 *
 * @since  1.0
 */

if ( ! function_exists( 'grd_content_service' ) ) {
	function grd_content_service() {
		if ( grd_get_option( 'service_layout' ) == 'grid' ) {
			if ( grd_get_option( 'service_columns' ) == '2' ) {
				echo 'col-xs-6 col-sm-6 col-6 type-service';
			} else if ( grd_get_option( 'service_columns' ) == '3' ) {
				echo 'col-xs-6 col-sm-6 col-md-4 col-4 type-service';
			} else if ( grd_get_option( 'service_columns' ) == '4' ) {
				echo 'col-xs-6 col-sm-6 col-md-3 col-3 type-service';
			}
		} else if ( grd_get_option( 'service_layout' ) == 'full-width' ) {
			echo 'col-sm-12 col-12 type-service full-width';
		}
	}
}
if ( ! function_exists( 'grd_service_pagination' ) ) {
	function grd_service_pagination() {
		if ( grd_get_option( 'service_pagination' ) == '1' ) {
			echo '<div class="col-xs-12 pag-1 text-center">';
			the_posts_pagination( array(
				'prev_text' => '<span class="fa fa fa-angle-left"></span>',
				'next_text' => '<span class="fa fa fa-angle-right"></span>',
			) );
		}
	}
}
if ( ! function_exists( 'grd_service_contact' ) ) {
	function grd_service_footer() {
		if ( grd_get_option( 'service_show' ) == 1 ) {
			if ( grd_get_option( 'service_template' ) == 1 ) {
				?>
                <div class="col-xs-12 col-sm-6">
                    <h3 class="col-xs-12 title"><?php echo grd_get_option( 'service_ft_title' ); ?></h3>
                    <div class="col-xs-12 descr"><?php echo grd_get_option( 'service_ft_descr' ); ?></div>
                </div>

				<?php
			}
		}
	}
}

/**
 * Check is blog
 *
 * @since  1.0
 */

if ( ! function_exists( 'grd_is_blog' ) ) :
	function grd_is_blog() {
		if ( ( is_archive() || is_author() || is_category() || is_home() || is_tag() ) && 'post' == get_post_type() ) {
			return true;
		}

		return false;
	}

endif;


/**
 * Check is catalog
 *
 * @return bool
 */
if ( ! function_exists( 'grd_is_catalog' ) ) :
	function grd_is_catalog() {

		if ( function_exists( 'is_shop' ) && ( is_shop() || is_product_category() || is_product_tag() ) ) {
			return true;
		}

		return false;
	}
endif;

/**
 * Check is portfolio
 *
 * @since  1.0
 */

if ( ! function_exists( 'grd_is_portfolio' ) ) :
	function grd_is_portfolio() {
		if ( is_post_type_archive( 'portfolio' ) || is_tax( 'portfolio_category' ) ) {
			return true;
		}

		return false;
	}
endif;

/**
 * Check is services
 *
 * @since  1.0
 */

if ( ! function_exists( 'grd_is_service' ) ) :
	function grd_is_service() {
		if ( is_post_type_archive( 'service' ) || is_tax( 'service_category' ) ) {
			return true;
		}

		return false;
	}
endif;


if ( ! function_exists( 'grd_portfolio_thumbnail' ) ) :
	function grd_portfolio_thumbnail() {

		global $grd_portfolio_loop, $wp_query;
		$size                                  = "grd-portfolio-grid";
		$grd_portfolio_loop['size']            = $size;
		$grd_portfolio_loop['portfolio_class'] = 'col-xs-6 col-sm-6 col-md-3 portfolio-grid';
		$layout                                = grd_portfolio_layout();
		$grid                                  = 9;

		if ( $layout == 'modern' ) {
			$mod = $wp_query->current_post % $grid;
			if ( in_array( $mod, array( 0, 8 ) ) ) {
				$grd_portfolio_loop['size']            = 'grd-portfolio-grid-dual-horizontal';
				$grd_portfolio_loop['portfolio_class'] = 'col-xs-6 col-sm-6 portfolio-horizontal item-' . $layout;
			} elseif ( $mod == 2 ) {
				$grd_portfolio_loop['size']            = 'grd-portfolio-grid-dual-vertical';
				$grd_portfolio_loop['portfolio_class'] = 'col-xs-6 col-sm-6 col-md-3 portfolio-vertical item-' . $layout;
			} else {
				$grd_portfolio_loop['size']            = 'grd-portfolio-grid-single';
				$grd_portfolio_loop['portfolio_class'] = 'col-xs-12 col-sm-6 col-md-3 portfolio-nomal item-' . $layout;
			}
		} else if ( $layout == 'full-width' ) {
			$grd_portfolio_loop['size']            = 'grd-portfolio-full-width';
			$grd_portfolio_loop['portfolio_class'] = 'col-xs-6 col-sm-6 col-md-3 item-' . $layout;
		}

		return $grd_portfolio_loop;
	}
endif;


/**
 * Check is portfolio
 *
 * @since  1.0
 */
if ( ! function_exists( 'grd_portfolio_pagination' ) ) {
	function grd_portfolio_pagination() {
		if ( grd_get_option( 'portfolio_pagination' ) == '1' ) {
			if ( grd_get_option( 'portfolio_pagination_style' ) == '2' ) {
				$show_more = esc_html__( 'Show more', 'grd' );
				echo '<div class="col-xs-12 pag-2 numeric-navigation text-center">';
				$next = "<span>$show_more</span>";
				$prev = "";
			} else {
				echo '<div class="col-xs-12 pag-1 numeric-navigation text-center">';
				$next = "<span class='fa fa-angle-right'></span>";
				$prev = "<span class='fa fa-angle-left'></span>";
			}
			echo get_the_posts_pagination( array(
				'prev_text' => $prev,
				'next_text' => $next,
			) );
		}
	}
}


if ( ! function_exists( 'grd_portfolio_layout' ) ) {
	function grd_portfolio_layout() {
		$layout = grd_get_option( 'portfolio_layout' );

		return $layout;
	}
}


/**
 * Load images
 *
 * @since  1.0
 */
function grd_load_image( $img_id, $size ) {
	if ( function_exists( 'wpb_getImageBySize' ) ) {
		$image       = wpb_getImageBySize(
			array(
				'attach_id'  => $img_id,
				'thumb_size' => $size,
			)
		);
		$image_thumb = $image['thumbnail'];
	} else {
		$image = wp_get_attachment_image_src( $img_id, $size );
		if ( $image ) {
			$img_url     = $image[0];
			$img_size    = $size;
			$image_thumb = "<img alt='$img_size'  src='$img_url'/>";
		}
	}

	return $image_thumb;
}

if ( ! function_exists( 'grd_menu_icon' ) ) :
	/**
	 * Get menu icon
	 */
	function grd_menu_icon() {
		printf(
			'<span id="mf-navbar-toggle" class="navbar-icon">
					<span class="navbars-line"></span>
				</span>'
		);
	}
endif;