<?php
/**
 * Hooks for template header
 *
 * @package Grd
 */

/**
 * Enqueue scripts and styles.
 *
 * @since 1.0
 */
function grd_enqueue_scripts() {
	/**
	 * Register and enqueue styles
	 */

	wp_register_style( 'grd-fonts', grd_fonts_url(), array(), '' );
	wp_register_style( 'bootstrap', get_template_directory_uri() . '/css/bootstrap.min.css', array(), '3.3.7' );
	wp_register_style( 'fontawesome', get_template_directory_uri() . '/css/font-awesome.min.css', array(), '4.6.3' );
	wp_register_style( 'flaticon', get_template_directory_uri() . '/css/flaticon.css', array(), '4.6.3' );
	wp_register_style( 'slick', get_template_directory_uri() . '/css/slick.css', array(), '1.8.1' );
	wp_register_style( 'photoswipe', get_template_directory_uri() . '/css/photoswipe.css', array(), '4.1.1' );

	wp_enqueue_style( 'grd-style', get_template_directory_uri() . '/style.css', array(
		'grd-fonts',
		'bootstrap',
		'fontawesome',
		'flaticon',
		'slick',
		'photoswipe',

	), '20181007'
	);

	wp_add_inline_style( 'grd-style', grd_customize_css() );

	/**
	 * Register and enqueue scripts
	 */


	$min = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

	wp_register_script( 'sticky', get_template_directory_uri() . '/js/plugins/jquery.sticky.js', array(), '1.0', true );
	wp_register_script( 'fitvids', get_template_directory_uri() . '/js/plugins/jquery.fitvids.js', array(), '1.0', true );

	wp_register_script( 'isotope', get_template_directory_uri() . '/js/plugins/isotope.pkgd.min.js', array(), '2.2.2', true );
	wp_register_script( 'slick', get_template_directory_uri() . '/js/plugins/slick.min.js', array(), '1.0', true );
	wp_register_script( 'counterup', get_template_directory_uri() . '/js/frontend/jquery.counterup.min.js', array(), '1.0', true );
	wp_register_script( 'waypoints', get_template_directory_uri() . '/js/frontend/waypoints.min.js', array(), '1.0', true );
	wp_register_script( 'photoswipe', get_template_directory_uri() . '/js/plugins/photoswipe.min.js', array(), '4.1.1', true );
	wp_register_script( 'photoswipe-ui', get_template_directory_uri() . '/js/plugins/photoswipe-ui.min.js', array( 'photoswipe' ), '4.1.1', true );

	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}

	wp_localize_script(
		'grd', 'grdData', array(
			'ajax_url' => admin_url( 'admin-ajax.php' ),
		)
	);
	wp_enqueue_script( 'grd-script', get_template_directory_uri() . "/js/scripts.js", array(
		'jquery',
		'imagesloaded',
		'sticky',
		'isotope',
		'slick',
		'counterup',
		'waypoints',
		'imagesloaded',
		'fitvids'
	), '20171013', true );

	if ( is_singular() ) {

		wp_enqueue_style( 'photoswipe' );
		wp_enqueue_script( 'photoswipe-ui' );

		$photoswipe_skin = 'photoswipe-default-skin';
		if ( wp_style_is( $photoswipe_skin, 'registered' ) && ! wp_style_is( $photoswipe_skin, 'enqueued' ) ) {
			wp_enqueue_style( $photoswipe_skin );
		}
	}
}

add_action( 'wp_enqueue_scripts', 'grd_enqueue_scripts' );


/**
 * Custom scripts and styles on header
 *
 * @since  1.0.0
 */
function grd_header_scripts() {
	/**
	 * All Custom CSS rules
	 */

}

add_action( 'wp_head', 'grd_header_scripts' );

/**
 * Enqueues front-end CSS for theme customization
 */
function grd_customize_css() {
	$css = '';

	// Logo
	$logo_size_width = intval( grd_get_option( 'logo_width' ) );
	$logo_css        = $logo_size_width ? 'width:' . $logo_size_width . 'px; ' : '';

	$logo_size_height = intval( grd_get_option( 'logo_height' ) );
	$logo_css         .= $logo_size_height ? 'height:' . $logo_size_height . 'px; ' : '';

	$logo_margin = grd_get_option( 'logo_position' );
	$logo_css    .= $logo_margin['top'] ? 'margin-top:' . $logo_margin['top'] . '; ' : '';
	$logo_css    .= $logo_margin['right'] ? 'margin-right:' . $logo_margin['right'] . '; ' : '';
	$logo_css    .= $logo_margin['bottom'] ? 'margin-bottom:' . $logo_margin['bottom'] . '; ' : '';
	$logo_css    .= $logo_margin['left'] ? 'margin-left:' . $logo_margin['left'] . '; ' : '';

	if ( ! empty( $logo_css ) ) {
		$css .= '.site-header .logo img ' . ' {' . $logo_css . '}';
	}

	//Header Transparent of Header v3
	$color_header     = grd_get_option( 'header_transparent_header_color' );
	$color_header_css = $color_header ? 'color:' . $color_header : '';

	$color_nav     = grd_get_option( 'header_transparent_nav_color' );
	$color_nav_css = $color_nav ? 'color:' . $color_nav : '';

	if ( ! empty( $color_header ) ) {
		$css .= '.page-template-template-homepage.header-transparent .header-content ' . ' {' . $color_header_css . '}';
		$css .= '.page-template-template-homepage.header-transparent .socials a' . ' {' . $color_header_css . '}';
		$css .= '.page-template-template-homepage.header-transparent .main-nav ul.menu > li > a' . ' {' . $color_nav_css . '}';
	}

	/* Color Scheme */
	$color_scheme_option = grd_get_option( 'color_scheme' );
	if ( intval( grd_get_option( 'custom_color_scheme' ) ) ) {
		$color_scheme_option = grd_get_option( 'custom_color' );
	}
	// Don't do anything if the default color scheme is selected.
	if ( $color_scheme_option ) {
		$css .= grd_get_color_scheme_css( $color_scheme_option );
	}

	$css .= grd_typography_css();

	$css .= grd_get_heading_typography_css();

	return $css;
}

/**
 * Display topbar on top of site v1
 *
 * @since 1.0.0
 */
function grd_show_topbar() {
	if ( ! intval( grd_get_option( 'topbar_enable' ) ) ) {
		return;
	}

	if ( grd_get_option( 'header_layout' ) !== 'v1' ) {
		return;
	}

	if ( is_active_sidebar( 'header-bar' ) == false ) {
		return '';
	}

	$style = $class = '';

	?>
    <div id="topbar" class="topbar <?php echo esc_attr( $class ) ?>">


        <div class="topbar-widgets clearfix">

			<?php
			ob_start();
			dynamic_sidebar( 'header-bar' );
			$output = ob_get_clean();

			echo apply_filters( 'grd_header_bar', $output );
			?>

            <div class="menu-block-right"></div>
        </div>

    </div>
	<?php
}

add_action( 'grd_before_header', 'grd_show_topbar' );

/**
 * Display topbar on top of site v2
 *
 * @since 1.0.0
 */
function grd_show_topbar_v2() {
	if ( ! intval( grd_get_option( 'topbar_enable' ) ) ) {
		return;
	}

	if ( grd_get_option( 'header_layout' ) !== 'v2' ) {
		return;
	}

	if ( is_active_sidebar( 'topbar-left' ) == false &&
	     is_active_sidebar( 'topbar-right' ) == false
	) {
		return '';
	}

	$style = $class = '';

	$topbar_bg = grd_get_option( 'topbar_background' );

	if ( $topbar_bg ) {
		$style = 'style=background-color:' . esc_attr( $topbar_bg ) . '';
	}

	?>
    <div id="topbar" class="topbar <?php echo esc_attr( $class ) ?>" <?php echo esc_attr( $style ); ?>>
        <div class="topbar-widgets clearfix">
            <div class="container">
                <div class="row">
                    <div class="col-md-12 col-lg-7 sidebar-left">
						<?php if ( is_active_sidebar( 'topbar-left' ) ) : ?>
							<?php dynamic_sidebar( 'topbar-left' ); ?>
						<?php endif; ?>
                    </div>
                    <div class="col-md-12 col-lg-5 sidebar-right">
						<?php if ( is_active_sidebar( 'topbar-right' ) ) : ?>
							<?php dynamic_sidebar( 'topbar-right' ); ?>
						<?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
	<?php
}

add_action( 'grd_before_header_v2', 'grd_show_topbar_v2' );

/**
 * Display topbar on top of site v3
 *
 * @since 1.0.0
 */

function grd_show_topbar_v3() {
	if ( ! intval( grd_get_option( 'topbar_enable' ) ) ) {
		return;
	}

	if ( grd_get_option( 'header_layout' ) !== 'v3' ) {
		return;
	}

	if ( is_active_sidebar( 'header-bar' ) == false ) {
		return '';
	}

	$style = $class = '';

	$topbar_bg = grd_get_option( 'topbar_background' );

	if ( $topbar_bg ) {
		$style = 'style=background-color:' . esc_attr( $topbar_bg ) . '';
	}

	?>
    <div id="topbar" class="topbar <?php echo esc_attr( $class ) ?>" <?php echo esc_attr( $style ); ?>>
        <div class="topbar-widgets clearfix">
			<?php
			ob_start();
			dynamic_sidebar( 'header-bar' );
			$output = ob_get_clean();

			echo apply_filters( 'grd_header_bar', $output );
			?>
			<?php
			if ( grd_get_option( 'header_layout' ) == 'v3' ) {
				echo '<div class="menu-block-right-v3"></div>';
			} else {
				echo '<div class="menu-block-right"></div>';
			}
			?>
        </div>
    </div>
	<?php
}

add_action( 'grd_before_header_v3', 'grd_show_topbar_v3' );

/**
 * Display topbar on mobile
 *
 * @since 1.0.0
 */
function grd_show_topbar_mobile() {
	if ( ! intval( grd_get_option( 'topbar_mobile' ) ) ) {
		return;
	}

	if ( is_active_sidebar( 'topbar-mobile' ) == false ) {
		return '';
	}

	$style = $class = '';

	$topbar_bg = grd_get_option( 'topbar_background' );

	if ( $topbar_bg ) {
		$style = 'style=background-color:' . esc_attr( $topbar_bg ) . '';
	}

	?>
    <div id="topbar_mobile"
         class="topbar topbar-mobile <?php echo esc_attr( $class ) ?>" <?php echo esc_attr( $style ); ?>>

        <div class="container">
            <div class="topbar-mobile-content">
				<?php dynamic_sidebar( 'topbar-mobile' ); ?>
            </div>
        </div>

    </div>
	<?php
}

add_action( 'grd_before_header_mobile', 'grd_show_topbar_mobile' );

/**
 * Display header
 */
function grd_show_header() {
	do_action( 'grd_before_header_mobile' );
	get_template_part( 'parts/headers/header', grd_get_option( 'header_layout' ) );
}

add_action( 'grd_header', 'grd_show_header' );

/**
 * Display page header
 */
function grd_page_header() {
	if ( is_page_template( 'template-homepage.php' ) ) {
		return;
	}

	get_template_part( 'parts/page-headers/page-header' );
}

add_action( 'grd_after_header', 'grd_page_header' );

/**
 * Get breadcrumbs
 *
 * @since  1.0.0
 *
 * @return string
 */

if ( ! function_exists( 'grd_get_breadcrumbs' ) ) :
	function grd_get_breadcrumbs() {

		ob_start();
		?>
        <nav class="breadcrumbs">
			<?php
			grd_breadcrumbs(
				array(
					'before'   => '',
					'taxonomy' => function_exists( 'is_woocommerce' ) && is_woocommerce() ? 'product_cat' : 'category',
				)
			);
			?>
        </nav>
		<?php
		echo ob_get_clean();
	}

endif;

/**
 * Filter to archive title and add page title for singular pages
 *
 * @param string $title
 *
 * @return string
 */
function grd_the_archive_title( $title ) {
	if ( is_search() ) {
		$title = sprintf( esc_html__( 'Search Results', 'grd' ) );
	} elseif ( is_404() ) {
		$title = sprintf( esc_html__( 'Page Not Found', 'grd' ) );
	} elseif ( is_page() ) {
		$title = get_the_title();
	} elseif ( is_home() && is_front_page() ) {
		$title = esc_html__( 'The Latest Posts', 'grd' );
	} elseif ( is_home() && ! is_front_page() ) {
		$title = get_the_title( get_option( 'page_for_posts' ) );
	} elseif ( function_exists( 'is_shop' ) && is_shop() ) {
		$title = get_the_title( get_option( 'woocommerce_shop_page_id' ) );
	} elseif ( function_exists( 'is_product' ) && is_product() ) {
		$cats = get_the_terms( get_the_ID(), 'product_cat' );
		if ( ! is_wp_error( $cats ) && $cats ) {
			$title = $cats[0]->name;
		} else {
			$title = get_the_title( get_option( 'woocommerce_shop_page_id' ) );
		}
	} elseif ( is_post_type_archive( 'portfolio' ) ) {

		if ( get_option( 'grd_portfolio_page_id' ) ) {
			$title = get_the_title( get_option( 'grd_portfolio_page_id' ) );
		} else {
			$title = esc_html__( 'Portfolio', 'grd' );
		}

	} elseif ( is_post_type_archive( 'service' ) ) {

		if ( get_option( 'grd_service_page_id' ) ) {
			$title = get_the_title( get_option( 'grd_service_page_id' ) );
		} else {
			$title = esc_html__( 'Services', 'grd' );
		}

	} elseif ( is_singular( 'portfolio' ) ) {
		$title = get_the_title( get_the_ID() );
	} elseif ( is_singular( 'service' ) ) {
		$title = get_the_title( get_the_ID() );
	} elseif ( is_tax() || is_category() ) {
		$title = single_term_title( '', false );
	} elseif ( is_singular( 'post' ) ) {
		$title = get_the_title( get_the_ID() );
	}

	return $title;
}

add_filter( 'get_the_archive_title', 'grd_the_archive_title' );

function grd_header_social() {

	$footer_socials = (array) grd_get_option( 'header_social' );

	if ( empty( $footer_socials ) ) {
		return;
	}

	?>
    <li class="extra-menu-item menu-item-socials socials">
		<?php

		if ( $footer_socials ) {
			$socials = (array) grd_get_socials();

			foreach ( $footer_socials as $social ) {
				foreach ( $socials as $name => $label ) {
					$link_url = $social['link_url'];

					if ( preg_match( '/' . $name . '/', $link_url ) ) {

						if ( $name == 'google' ) {
							$name = 'google-plus';
						}

						printf( '<a href="%s" target="_blank"><i class="fa fa-%s"></i></a>', esc_url( $link_url ), esc_attr( $name ) );
						break;
					}
				}
			}
		}
		?>
    </li>
	<?php
}

/**
 * Returns CSS for the color schemes.
 *
 *
 * @param array $colors Color scheme colors.
 *
 * @return string Color scheme CSS.
 */
function grd_get_color_scheme_css( $colors ) {
	return <<<CSS
	/* Color Scheme */
	/* Background Color */
	.grd_title-type-1 .title:before,
	.grd_title-type-1 .title:after,
    .grd_title-type-2 .title:before,
    .grd_title-type-2 .title:after,
    .grd-icon-box-3 .box_btn a:hover,
    .grd-icon-box-7:hover .icon .icon-content,
    .grd-icon-box-8 .icon-header,
    .grd-icon-box-9:hover .icon-top .svg-icon,
    .icon-box-list .icon-box-list-items .icon-box-wrapter .icon .icon-content:before,
    .icon-box-list .grd-icon-box-list-2,
    .grd-portfolio-shortcode .portfolio-cats-filters #filters .active,
    .grd-portfolio-shortcode .pag-2 .pagination .next,
    .grd-banner-style-1 .content .button:hover,
    .grd-banner-style-2 .content,
    .grd-image-box-1 .entry-icon,
    .grd-image-box-2 .show,
    .grd-image-box-3 .top .entry-icon,
    .grd-image-box-4 .entry-icon,
    .slick-dots li button:hover:before,
    .dl_members.style-2 .box-member .overlay-link a:hover,
    .dl_prices_3 .sale,
    .dl_prices_3 .dl-button a:hover,
    .dl_team_carousel .box-title h2:before,
    .dl_team_carousel .item-team .box-member .overlay-link a:hover,
    .dl-box-title h2:before,
    .dl_testimorial_carousel .style-1.slick-slide:hover .box-avatar:before,
    .woocommerce .widget_shopping_cart_content .buttons .checkout:hover,.widget_shopping_cart_content .buttons .checkout:hover,
    .numeric-navigation .page-numbers:hover,.numeric-navigation .page-numbers.current,
    .menu-extra .schedule .schedule-content,
    .header-v2 .primary-nav > ul.menu > li.mf-active-menu,
    .header-v4 .primary-nav > ul.menu > li.mf-active-menu,
    .header-v4 .menu-extra .schedule .schedule-box,
    .blog-wrapper .entry-thumbnail .meta-cat,
    .grd-portfolio .pag-2 .pagination .next,
    .woocommerce button.button.alt,
    .woocommerce .widget_price_filter .ui-slider .ui-slider-handle,
    .woocommerce .widget_price_filter .ui-slider .ui-slider-range,
    .woocommerce table.shop_table thead,
    .woocommerce .widget_product_tag_cloud a:hover
	{background-color: $colors}

	/* Color */
	button,html input[type="button"],input[type="reset"],input[type="submit"], 
    a, 
    .grd_portfolio_atrrs ul .rating span i, 
    .grd-contact-box ul li .value .text-green, 
    .grd-icon-box:hover .icon .icon-content .svg-icon, 
    .grd-icon-box-1:hover .icon .icon-content .svg-icon, 
    .grd-icon-box-7 .icon .icon-content svg, 
    .grd-icon-box-8 .content .readmore a:hover, 
    .grd-icon-box-8:hover .emtry-title .title, 
    .grd-icon-box-8:hover .icon .icon-content .svg-icon, 
    .grd-icon-box-9 .icon-top .svg-icon, 
    .grd-icon-box-10 .icon-box-wrapter .content .readmore a:hover, 
    .grd-icon-box-11 .icon-box-wrapter:hover .emtry-title .title, 
    .icon-box-list .icon-box-list-items .icon-box-wrapter .emtry-title:hover, 
    .icon-box-list .icon-box-list-items:hover .icon .icon-content .svg-icon, 
    .icon-box-list .icon-box-list-items:hover .emtry-title .title, 
    .grd-portfolio-shortcode .portfolio-item .content-item .entry-content .entry-meta .category, 
    .grd-portfolio-shortcode .portfolio-item .content-item .entry-content .entry-read-more:hover .read-more, 
    .grd-portfolio-shortcode .pag-2 .pagination .next:hover, 
    .grd-list ul li a:hover, 
    .grd-list ul li a:hover span:before, 
    .grd-banner-style-1 .content .button, 
    .grd-image-box-3 .bottom .entry-read-more a:hover, 
    .grd-image-box-4:hover .entry-title .title, 
    .grd-image-box-6 .entry-icon .svg-icon, 
    .grd-image-box-6:hover .entry-read-more a:hover, 
    .grd-award-carousel .item .svg-icon, 
    .dl_faq .style-1:hover h3, 
    .dl_faq .style-1:hover .box-icon svg, 
    .dl_faq .style-2 .asked h2, 
    .dl_members .job, 
    .dl_members .box-member .overlay-link a:hover, 
    .dl_members.box-2 a:hover, 
    .dl_prices_3 .price, 
    .slick-arrow:hover, 
    .dl_testimorial_carousel .address, 
    .dl_testimorial_carousel .box-star .fa-star, 
    .dl_testimorial_carousel .style-1.slick-slide:hover svg, 
    .dl_testimorial_carousel .style-2 svg, 
    .map-box a:hover, 
    .vc_tta-style-classic .vc_tta-panel.vc_active i, 
    .search-modal .product-cats label span:hover, 
    .search-modal .product-cats input:checked + span, 
    .search-modal .search-results ul li .search-item:hover .title, 
    .header-v2 .header-bar svg, 
    .header-v4 .header-bar .highlights .widget .pull-left svg, 
    .woocommerce .widget_shopping_cart_content ul.cart_list .mini_cart_item a:hover,.widget_shopping_cart_content ul.cart_list .mini_cart_item a:hover, 
    .main-nav li li:hover > a, 
    .main-nav ul.menu > li.current-menu-item > a,.main-nav ul.menu > li.current-menu-parent > a,.main-nav ul.menu > li.current-menu-ancestor > a,.main-nav ul.menu > li:hover > a, 
    .main-nav div.menu > ul > li.current_page_item > a,.main-nav div.menu > ul > li:hover > a, 
    .menu-extra a:hover, 
    .header-v2 .menu-extra svg:hover, 
    .header-v3 .menu-item-search a:hover, 
    .primary-mobile-nav ul.menu li > a:hover, 
    .primary-mobile-nav ul.menu li.active > a, 
    .primary-mobile-nav ul.menu li.current_page_parent > a,.primary-mobile-nav ul.menu li.current-menu-item > a,.primary-mobile-nav ul.menu li.current-menu-ancestor > a,.primary-mobile-nav ul.menu li.current-menu-parent > a, 
    .entry-meta i,.entry-meta svg, 
    .entry-meta a:hover, 
    .entry-footer .tags-links span, 
    .entry-footer .footer-socials span i,.entry-footer .footer-socials span svg, 
    .post-navigation .nav-links a:hover .meta-nav, 
    .blog article.sticky .entry-title:hover:before, 
    .blog-wrapper .entry-title a:hover, 
    .no-images .blog-wrapper .entry-footer a:hover, 
    .no-images .blog-wrapper:hover .entry-title a, 
    .comment-respond .form-submit input[type="submit"], 
    .wpcf7 .wpcf7-form input[type="date"]:before, 
    .wpcf7 .wpcf7-form .time:before, 
    .wpcf7 .contact-home input[type="submit"]:hover, 
    .widget ul li a:hover, 
    .widget_tag_cloud a:hover, 
    .blog-sidebar .widget ul li a:hover, 
    .topbar a:hover, 
    .topbar ul li a:hover, 
    .header-v1 .topbar .widget .pull-left svg, 
    .header-v1 .topbar .socials li a:hover, 
    .header-v2 .topbar .sidebar-left .widget svg, 
    .header-v2 .topbar .sidebar-right li a:hover, 
    .header-v2 .topbar .socials a:hover, 
    .header-v2 .topbar #lang_sel > ul > li > a:after,.header-v2 .topbar .lang_sel > ul > li > a:after, 
    .header-v3 .topbar .widget svg, 
    .header-v3 .socials a:hover, 
    .socials a:hover, 
    .page-header .breadcrumbs a:hover, 
    .single-service .widget_categories ul li a:hover, 
    .brochures > div:hover i,.brochures > div:hover a, 
    .grd-portfolio .archive-portfolio .portfolio-item .content-item .entry-content .entry-meta .category, 
    .grd-portfolio .archive-portfolio .portfolio-item .content-item .entry-content .entry-read-more:hover .read-more, 
    .grd-portfolio .pag-2 .pagination .next:hover, 
    .grd-portfolio .pag-2 .pagination .loader:before, 
    .single-portfolio .portfolio-navigation a:hover .meta-nav,.single-portfolio .portfolio-navigation a:hover span, 
    .single-portfolio .portfolio-navigation .nav-project a:hover, 
    .woocommerce .woocommerce-result-count, 
    .woocommerce ul.products li.product .price, 
    .woocommerce ul.products li.product a:hover, 
    .woocommerce .star-rating, 
    .woocommerce .star-rating::before, 
    .woocommerce a.button,.woocommerce button.button,.woocommerce input.button,.woocommerce #respond input#submit, 
    .woocommerce div.product form.cart .button, 
    .woocommerce div.product .woocommerce-tabs ul.tabs li.active a, 
    .woocommerce div.product div.grd-images:hover .slick-arrow, 
    .woocommerce .quantity .increase:hover,.woocommerce .quantity .decrease:hover, 
    .woocommerce .comment-form-rating .stars a, 
    .woocommerce .comment-form-rating .stars a:hover, 
    .woocommerce table.shop_table td.product-subtotal, 
    .woocommerce table.shop_table td .coupon .button, 
    .woocommerce .wc-proceed-to-checkout a.checkout-button, 
    .woocommerce .woocommerce-message a,.woocommerce .woocommerce-error a,.woocommerce .woocommerce-info a, 
    .woocommerce .woocommerce-MyAccount-navigation ul li:hover a,.woocommerce .woocommerce-MyAccount-navigation ul li.is-active a, 
    .woocommerce .widget_product_categories .product-categories li:hover, 
    .woocommerce .widget_product_categories .product-categories:hover a, 
    .woocommerce .widget_product_categories .product-categories .current-cat a, 
    .footer-widgets .map-link a, 
    .footer-widgets .map-link:hover, 
    .footer-widgets .working .green, 
    .footer-widgets.footer-middle .phone div, 
    .footer-widgets.footer-middle a:hover
	{color: $colors}

	.woocommerce table.shop_table a.remove:hover
	{color: $colors !important;}

	/* Border */
	button,html input[type="button"],input[type="reset"],input[type="submit"], 
    .grd-icon-box .content .readmore, 
    .grd-icon-box-1:hover:before, 
    .grd-icon-box-1:hover:after, 
    .grd-icon-box-3 .box_btn a, 
    .grd-icon-box-5:hover:before, 
    .grd-icon-box-5:hover:after, 
    .grd-icon-box-8:hover:before, 
    .grd-icon-box-8:hover:after, 
    .grd-icon-box-9:hover .icon-top .svg-icon, 
    .icon-box-8 .grd-button:hover, 
    .grd-portfolio-shortcode .pag-2 .pagination .next, 
    .grd-portfolio-shortcode .pag-2 .pagination .next:hover, 
    .grd-banner-style-1 .content .button, 
    .grd-award-carousel .slick-arrow:hover, 
    .slick-dots li.slick-active button:before, 
    .dl_team_carousel .box-team .box-banner .text-banner a, 
    .dl_testimorial_carousel .style-1.slick-slide:hover, 
    .dl_testimorial_carousel .style-1.slick-slide:hover .box-avatar:before, 
    .dl_testimorial_carousel .style-1.slick-current:hover, 
    .dl_testimorial_carousel .style-3:hover .box-avatar, 
    .woocommerce .widget_shopping_cart_content .buttons .checkout:hover,.widget_shopping_cart_content .buttons .checkout:hover, 
    .numeric-navigation .page-numbers:hover,.numeric-navigation .page-numbers.current, 
    .header-v3 .menu-item-search a:hover, 
    .header-transparent.header-v3 .menu-item-search a:hover, 
    .comment-respond .form-submit input[type="submit"], 
    .grd-portfolio .pag-2 .pagination .next, 
    .grd-portfolio .pag-2 .pagination .next:hover, 
    .woocommerce nav.woocommerce-pagination ul li span.current,.woocommerce nav.woocommerce-pagination ul li a:hover,.woocommerce nav.woocommerce-pagination ul li a:focus, 
    .woocommerce a.button,.woocommerce button.button,.woocommerce input.button,.woocommerce #respond input#submit, 
    .woocommerce div.product form.cart .button, 
    .woocommerce table.shop_table td .coupon .button, 
    .woocommerce .wc-proceed-to-checkout a.checkout-button, 
    .woocommerce .widget_product_tag_cloud a:hover
	{border-color: $colors}

	/* Border Top */
	.header-v1 .site-header .header-main,
	.header-v4 .header-bar,
	.single-service .widget_categories ul,
	.brochures
	{border-top-color: $colors}

	/* Border Bottom */
	.header-v1 .topbar,
	.header-v1 .topbar .menu-block-right,
	.menu-extra .schedule .schedule-content:after,
	.single-service .widget_categories ul,
	.brochures
	{border-bottom-color: $colors}

	/* Background */
	button:hover,html input[type="button"]:hover,input[type="reset"]:hover,input[type="submit"]:hover, 
    blockquote, 
    .dl_team_carousel .box-team .box-banner .text-banner a:hover, 
    .grd_quotes:after, 
    .header-v2 .menu-extra .schedule .schedule-box, 
    .entry-footer .footer-socials .social-links a, 
    .blog-wrapper .entry-meta .meta-date:after, 
    .no-images .blog-wrapper .entry-header:after, 
    .comment-respond .form-submit input[type="submit"]:hover, 
    .wpcf7 .contact-home input[type="submit"], 
    .header-v1 .topbar:after, 
    .grd-service .type-service .entry-header .service-icon, 
    .grd-portfolio .portfolio-cats-filter #filters .active, 
    .woocommerce button.button.alt:hover, 
    .woocommerce ul.products li.product .box-images .button:hover, 
    .woocommerce nav.woocommerce-pagination ul li span.current,.woocommerce nav.woocommerce-pagination ul li a:hover,.woocommerce nav.woocommerce-pagination ul li a:focus, 
    .woocommerce a.button:hover,.woocommerce button.button:hover,.woocommerce input.button:hover,.woocommerce #respond input#submit:hover, 
    .woocommerce div.product form.cart .button:hover, 
    .woocommerce #review_form #respond .form-submit input:hover, 
    .woocommerce table.shop_table td .coupon .button:hover, 
    .woocommerce table.shop_table td .update-cart:hover, 
    .woocommerce .wc-proceed-to-checkout a.checkout-button:hover, 
    .footer-widgets .widget-title:before, 
    .footer-widgets .widget-title:after, 
    .footer-widgets input[type="submit"]:hover, 
    .backtotop:hover,
    .header-v2 .menu-extra .schedule
	{background: $colors}

	/* Color Important */
	.grd-icon-box-5:hover .text-footer,
	.icon-box-8 .grd-button:hover,
	.grd-button-group .hover-2:hover 
	{color: $colors !important}

	/* Background Important */
	.grd-button-group .grd-button:hover
	{background-color: $colors !important}

	/* Border Important */
	.grd-button-group .grd-button:hover,
	.icon-box-8 .grd-button:hover
	{border-color: $colors !important}

CSS;
}

/**
 * Get typography CSS base on settings
 *
 * @since 1.1.6
 */
if ( ! function_exists( 'grd_typography_css' ) ) :
	function grd_typography_css() {
		$css        = '';
		$properties = array(
			'font-family'    => 'font-family',
			'font-size'      => 'font-size',
			'variant'        => 'font-weight',
			'line-height'    => 'line-height',
			'letter-spacing' => 'letter-spacing',
			'color'          => 'color',
			'text-transform' => 'text-transform',
			'text-align'     => 'text-align',
		);

		$settings = array(
			'body_typo'              => 'body, .grd-award-carousel .item .title',
			'heading1_typo'          => 'h1',
			'heading2_typo'          => 'h2, .entry-footer, .blog-wrapper .entry-title',
			'heading3_typo'          => 'h3, .grd-icon-box .emtry-title, 
			                                .grd-icon-box-8 .icon-header, 
			                                .grd-icon-box-8 .title, 
			                                .dl_prices_1 .title-price, 
			                                .header-bar .highlights .widget .pull-left div:first-child, 
			                                .header-v4 .header-bar .highlights .widget .pull-right > div:first-child, 
			                                .post-navigation .nav-links .nav-title, 
			                                .header-v1 .topbar .widget .pull-right > div:first-child, 
			                                .brochures > div .title, 
			                                .woocommerce a.button,
                                            .woocommerce button.button,
                                            .woocommerce input.button,
                                            .woocommerce #respond input#submit,
                                            .woocommerce div.product .woocommerce-tabs ul.tabs li a,
                                            .woocommerce #review_form #respond .comment-reply-title,
                                            .woocommerce table.shop_table tr th,
                                            .woocommerce table.shop_table tr td,
                                            .woocommerce .woocommerce-message,
                                            .woocommerce .woocommerce-error,
                                            .woocommerce .woocommerce-info,
                                            .woocommerce .widget_product_categories .product-categories li a,
                                            .footer-widgets.footer-middle .phone div',
			'heading4_typo'          => 'h4, .grd-banner-style-3 .content .descr .content-title .title, .widget .widget-title',
			'heading5_typo'          => 'h5',
			'heading6_typo'          => 'h6',
			'menu_typo'              => '.main-nav a, .main-nav ul.menu > li > a, .primary-mobile-nav ul.menu li > a',
			'sub_menu_typo'          => '.main-nav li li a, .primary-mobile-nav ul.menu ul li a',
			'footer_text_typo'       => '.footer-widgets',
		);

		foreach ( $settings as $setting => $selector ) {
			$typography = grd_get_option( $setting );
			$default    = (array) grd_get_option_default( $setting );
			$style      = '';

			foreach ( $properties as $key => $property ) {
				if ( isset( $typography[$key] ) && ! empty( $typography[$key] ) ) {
					if ( isset( $default[$key] ) && strtoupper( $default[$key] ) == strtoupper( $typography[$key] ) ) {
						continue;
					}

					$value = 'font-family' == $key ? '"' . rtrim( trim( $typography[ $key ] ), ',' ) . '"' : $typography[$key];
					$value = 'variant' == $key ? str_replace( 'regular', '400', $value ) : $value;

					if ( $value ) {
						$style .= $property . ': ' . $value . ';';
					}
				}
			}

			if ( ! empty( $style ) ) {
				$css .= $selector . '{' . $style . '}';
			}
		}

		return $css;
	}
endif;
/**
 * Returns CSS for the typography.
 *
 * @return string typography CSS.
 */
function grd_get_heading_typography_css() {

	$headings   = array(
		'h1' => 'heading1_typo',
		'h2' => 'heading2_typo',
		'h3' => 'heading3_typo',
		'h4' => 'heading4_typo',
		'h5' => 'heading5_typo',
		'h6' => 'heading6_typo'
	);
	$inline_css = '';
	foreach ( $headings as $heading ) {
		$keys = array_keys( $headings, $heading );
		if ( $keys ) {
			$inline_css .= grd_get_heading_font( $keys[0], $heading );
		}
	}

	return $inline_css;

}

/**
 * Returns CSS for the typography.
 *
 *
 * @param array $body_typo Color scheme body typography.
 *
 * @return string typography CSS.
 */
function grd_get_heading_font( $key, $heading ) {

	$inline_css   = '';
	$heading_typo = grd_get_option( $heading );

	if ( $heading_typo ) {
		if ( isset( $heading_typo['font-family'] ) && strtolower( $heading_typo['font-family'] ) !== 'hind' ) {
			$inline_css .= $key . '{font-family:' . rtrim( trim( $heading_typo['font-family'] ), ',' ) . ', Arial, sans-serif}';
		}
	}

	if ( empty( $inline_css ) ) {
		return;
	}

	return <<<CSS
	{$inline_css}
CSS;
}


/**
 * Display the header minimized
 *
 * @since 1.0.0
 */
function grd_header_minimized() {
	if ( ! intval( grd_get_option( 'header_sticky' ) ) ) {
		return;
	}

	$css_class = 'grd-header-' . grd_get_option( 'header_layout' );

	printf( '<div id="grd-header-minimized" class="grd-header-minimized %s"></div>', esc_attr( $css_class ) );

}

add_action( 'grd_before_header_v2', 'grd_header_minimized' );