<?php
/**
 * Lead model.
 *
 * @package SocialContactForm
 * @since 1.0.0
 */

// Namespace .
namespace WPPOOL\SCF;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;


if ( ! class_exists('\WPPOOL\SCF\Lead') ) :
	/**
	 * Lead model.
	 *
	 * @package SocialContactForm
	 * @since 1.0.0
	 */
	final class Lead {

		/**
		 * Create table.
		 *
		 * @return void
		 */
		public static function create_table() {
			global $wpdb;
			$scf_table_name      = $wpdb->prefix . 'scf_leads';
			$cf7_table_name      = $wpdb->prefix . 'cf7_leads';
			$charset_collate = $wpdb->get_charset_collate();

			// create table if not exists for SCF.
			$scf_sql = "CREATE TABLE IF NOT EXISTS $scf_table_name(
				id mediumint(11) NOT NULL AUTO_INCREMENT,
				field text NULL DEFAULT NULL,
				meta text NULL DEFAULT NULL,
				created_at datetime DEFAULT '0000-00-00 00:00:00' NOT NULL,
				deleted_at datetime DEFAULT NULL,
				note text DEFAULT NULL,
				PRIMARY KEY  (id)
			) $charset_collate;";

			$cf7_sql = "CREATE TABLE IF NOT EXISTS $cf7_table_name(
				id mediumint(11) NOT NULL AUTO_INCREMENT,
				cf7_id text NULL DEFAULT NULL,
				field text NULL DEFAULT NULL,
				meta text NULL DEFAULT NULL,
				created_at datetime DEFAULT '0000-00-00 00:00:00' NOT NULL,
				deleted_at datetime DEFAULT NULL,
				note text DEFAULT NULL,
				PRIMARY KEY  (id)
			) $charset_collate;";

			require_once ABSPATH . '/wp-admin/includes/upgrade.php';
			dbDelta($scf_sql);
			dbDelta($cf7_sql);
		}

		/**
		 * Create lead for SCF.
		 *
		 * @param mixed $field Field.
		 * @param mixed $meta Meta.
		 * @return int
		 */
		public static function create( $field, $meta ) {
			global $wpdb;
			$table_name = $wpdb->prefix . 'scf_leads';

			$wpdb->insert(
				$table_name,
				[
					'field'      => wp_json_encode($field),
					'meta'       => wp_json_encode($meta),
					'created_at' => current_time('mysql'),
				]
			); // db call ok; no-cache ok.

			return $wpdb->insert_id;
		}

		/**
		 * Create lead for CF7.
		 *
		 * @param mixed $contact_forms_id Contact Forms id.
		 * @param mixed $field Field.
		 * @param mixed $meta Meta.
		 * @return int
		 */
		public static function cf7create( $contact_forms_id, $field, $meta ) {
			global $wpdb;
			$table_name = $wpdb->prefix . 'cf7_leads';

			$wpdb->insert(
				$table_name,
				[
					'cf7_id'      => wp_json_encode($contact_forms_id),
					'field'      => wp_json_encode($field),
					'meta'       => wp_json_encode($meta),
					'created_at' => current_time('mysql'),
				]
			); // db call ok; no-cache ok.

			return $wpdb->insert_id;
		}

		/**
		 * Summary of insert for SCF
		 *
		 * @param mixed $field Field.
		 * @param mixed $meta Meta.
		 * @return int
		 */
		public static function insert( $field, $meta ) {
			return self::create($field, $meta);
		}

		/**
		 * Get lead.
		 *
		 * @param mixed $id ID.
		 * @return array|null|object Database query result (see WPDB::get_row() for description).
		 */
		public static function get( $id ) {
			global $wpdb;

			$lead = $wpdb->get_row(
				$wpdb->prepare(
					"SELECT * FROM {$wpdb->prefix}scf_leads WHERE id = %d",
					$id
				)
			); // db call ok; no-cache ok.

			if ( $lead ) {
				$lead->field = json_decode($lead->field);
				$lead->meta  = json_decode($lead->meta);
			}

			return $lead;
		}

		/**
		 * Get all leads.
		 *
		 * @return array|null|object
		 */
		public static function get_all() {
			global $wpdb;

			$leads = $wpdb->get_results(
				"SELECT * FROM {$wpdb->prefix}scf_leads WHERE `deleted_at` IS NULL ORDER BY id DESC"
			); // db call ok; no-cache ok.

			if ( $leads ) {
				foreach ( $leads as $lead ) {
					$lead->field = json_decode($lead->field);
					$lead->meta  = json_decode($lead->meta);
				}
			}

			return $leads;
		}

		/**
		 * Get lead count.
		 *
		 * @return mixed
		 */
		public static function get_count() {
			global $wpdb;

			return $wpdb->get_var( "SELECT COUNT(*) FROM {$wpdb->prefix}scf_leads" ); // db call ok; no-cache ok.
		}

		/**
		 * Updates lead.
		 *
		 * @param mixed $id ID.
		 * @param mixed $field Field.
		 * @param mixed $meta Meta.
		 * @return mixed
		 */
		public static function update( $id, $field, $meta ) {
			global $wpdb;

			$wpdb->update(
				$wpdb->prefix . 'scf_leads',
				[
					'field'      => wp_json_encode($field),
					'meta'       => wp_json_encode($meta),
					'created_at' => current_time('mysql'),
				],
				[
					'id' => $id,
				]
			); // db call ok; no-cache ok.
		}

		/**
		 * Delete lead.
		 *
		 * @param mixed $id ID.
		 * @return mixed
		 */
		public static function delete( $id ) {
			global $wpdb;
			$table_name = $wpdb->prefix . 'scf_leads';

			$wpdb->update(
				$table_name,
				[
					'deleted_at' => current_time('mysql'),
				],
				[
					'id' => $id,
				]
			); // db call ok; no-cache ok.
		}

		/**
		 * Delete multiple leads SCF.
		 *
		 * @param mixed $id_array IDs.
		 * @return mixed
		 */
		public static function delete_multiple( $id_array ) {
			global $wpdb;

			$id_list = implode( ',', $id_array );
			$wpdb->query( $wpdb->prepare(
				// phpcs:ignore
				"UPDATE {$wpdb->prefix}scf_leads SET deleted_at = %s WHERE id IN ($id_list)",
				current_time('mysql')
			) ); // db call ok; no-cache ok.

			return $wpdb->rows_affected;
		}
		/**
		 * Delete multiple leads CF7.
		 *
		 * @param mixed $id_array IDs.
		 * @return mixed
		 */
		public static function delete_multiple_cf7( $id_array ) {
			global $wpdb;

			$id_list = implode( ',', $id_array );
			$wpdb->query( $wpdb->prepare(
				// phpcs:ignore
				"UPDATE {$wpdb->prefix}cf7_leads SET deleted_at = %s WHERE id IN ($id_list)",
				current_time('mysql')
			) ); // db call ok; no-cache ok.

			return $wpdb->rows_affected;
		}


		/**
		 * Delete all leads.
		 *
		 * @return mixed
		 */
		public static function delete_all() {
			global $wpdb;

			$wpdb->query(
				$wpdb->prepare( "UPDATE {$wpdb->prefix}scf_leads SET deleted_at = `%s`", current_time('mysql'))
			);

			return $wpdb->rows_affected;
		}

		/**
		 * Erase lead.
		 *
		 * @return mixed
		 */
		public static function erase_all() {
			global $wpdb;

			$wpdb->query( "DELETE FROM {$wpdb->prefix}scf_leads WHERE deleted_at IS NOT NULL" ); // db call ok; no-cache ok.

			return $wpdb->rows_affected;
		}

		/**
		 * Get lead field.
		 *
		 * @param mixed $id ID.
		 * @param mixed $field Field.
		 * @return mixed
		 */
		public static function get_field( $id, $field ) {
			$lead = self::get($id);

			if ( $lead ) {
				return $lead->field->$field;
			}

			return null;
		}

		/**
		 * Get lead meta.
		 *
		 * @param mixed $id ID.
		 * @param mixed $meta Meta.
		 * @return mixed
		 */
		public static function get_meta( $id, $meta ) {
			$lead = self::get($id);

			if ( $lead ) {
				return $lead->meta->$meta;
			}

			return null;
		}
	}

endif;
