<?php
/**
 * Message class.
 * Handles all Message requests.
 *
 * @package SocialContactForm
 * @since 1.0.0
 */

// Namespace.
namespace WPPOOL\SCF;

use WPCF7_ContactForm;
use WPCF7_Submission;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;


if ( ! class_exists( '\WPPOOL\SCF\Message' ) ) :
	/**
	 * Message class.
	 * Handles all Message requests.
	 *
	 * @package SocialContactForm
	 * @since 1.0.0
	 */
	final class Message {

		/**
		 * Constructor.
		 *
		 * @since 1.0.0
		 */
		public function __construct() {
			add_action( 'wpcf7_before_send_mail', [ $this, 'social_contact_form_send_message' ] );
			add_action( 'wpcf7_skip_mail', [ $this, 'social_contact_form_disable_cf7_mail_sending' ] );
		}

		/**
		 * Get tab configuration data.
		 *
		 * @param array $contact_form CF7 Id and get configuration.
		 *
		 * @return mixed
		 */
		public function social_contact_form_send_message( $contact_form ) {

			$scf        = scf_get_options();
			$selectedmode = $scf['contact_form_7']['selectedmode'];
			$activemail = $scf['contact_form_7']['activemail'];
			$custommail = $scf['contact_form_7']['cf7mail'];

			// Check if the social contact form is enabled for this form.
			$social_contact_form_status = get_post_meta( $contact_form->id(), '_social_contact_form_configuration', true )['social_contact_form_status'];
			$social_contact_form_mail_status = get_post_meta( $contact_form->id(), '_social_contact_form_configuration', true )['social_contact_form_mail_status'];
			$social_contact_form_number = get_post_meta( $contact_form->id(), '_social_contact_form_configuration', true )['social_contact_form_number'];
			$social_contact_form_tabs_status = get_post_meta( $contact_form->id(), '_social_contact_form_configuration', true )['social_contact_form_tabs_status'];
			$social_contact_form_message_fields = get_post_meta( $contact_form->id(), '_social_contact_form_configuration', true )['social_contact_form_message_fields'];
			$form_title = get_post_meta( $contact_form->id(), '_social_contact_form_configuration', true )['form_title'];
			$form_id = get_post_meta( $contact_form->id(), '_social_contact_form_configuration', true )['form_id'];

			if ( 'on' != $social_contact_form_status ) {
				return;
			}

			// Sanitize and validate the phone number.
			$social_contact_form_number = preg_replace( '/[^0-9\+]/', '', $social_contact_form_number );
			if ( substr( $social_contact_form_number, 0, 1 ) != '+' ) {
				$social_contact_form_number = '+' . $social_contact_form_number;
			}
			if ( ! preg_match( '/^\+[0-9]{1,15}$/', $social_contact_form_number ) ) {
				return;
			}

			// Get the form data.
			$submission = WPCF7_Submission::get_instance();
			if ( $submission ) {
				$posted_data = $submission->get_posted_data();
				$contact_forms = WPCF7_ContactForm::get_instance($contact_form->id());
				$contact_forms_id = $contact_form->id();

				$form_fields = $contact_forms->scan_form_tags();

				$posted_data['site_title'] = get_bloginfo('name');
				$posted_data['site_description'] = get_bloginfo('description');
				$posted_data['site_url'] = get_bloginfo('url');
				$posted_data['admin_email'] = get_bloginfo('admin_email');
				$posted_data['form_title'] = $form_title;
				$posted_data['form_id'] = $form_id;

				// IP and Meta data calculations.
				if ( ! empty( $_SERVER['HTTP_CLIENT_IP'] ) ) {
					$ip = sanitize_text_field( wp_unslash( $_SERVER['HTTP_CLIENT_IP'] ) );
				} elseif ( ! empty( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) {
					$ip = sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_FORWARDED_FOR'] ) );
				} else {
					if ( isset( $_SERVER['REMOTE_ADDR'] ) ) {
						$ip = sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) );
					}
				}

				$location_data = json_decode( wp_remote_retrieve_body( wp_remote_get( 'http://ip-api.com/json/' . $ip ) ) );

				if ( empty( $location_data ) || ! isset( $location_data->status ) || 'fail' === $location_data->status ) {
					$ipdata = wp_remote_get( 'http://ipecho.net/plain' );
					if ( ! is_wp_error( $ipdata ) ) {
						$ip = $ipdata['body'];
						$location_data = json_decode( wp_remote_retrieve_body( wp_remote_get( 'http://ip-api.com/json/' . $ip ) ) );
					} else {
						$location_data = new \stdClass();
						$location_data->country = null;
						$location_data->city = null;
					}
				}

				$country = $location_data->country;
				$city = $location_data->city;
				// End IP.

				$scf_user_agent = $submission->get_meta( 'user_agent' ) ? $submission->get_meta( 'user_agent' ) : '';
				// phpcs:ignore
				if ( $timestamp = $submission->get_meta( 'timestamp' ) ) {
					$date = wp_date( get_option( 'date_format' ), $timestamp );
					$time = wp_date( get_option( 'time_format' ), $timestamp );
				}

				$single_meta_data = array_merge([ $country ], [ $city ], [ $ip ], [ $scf_user_agent ], [ $date ], [ $time ]);

				// Meta Data.
				$meta_data = [
					'country' => $single_meta_data[0],
					'city' => $single_meta_data[1],
					'ip_address' => $single_meta_data[2],
					'useragent' => $single_meta_data[3],
					'date' => $single_meta_data[4],
					'time' => $single_meta_data[5],
				];

				$current_url = '';
				if ( isset( $_SERVER['HTTP_REFERER'] ) ) {
					$current_url = esc_url_raw( wp_unslash( $_SERVER['HTTP_REFERER'] ) );
				}
				$posted_data['page_url'] = $current_url;

				if ( ! empty($posted_data) ) {
					foreach ( $posted_data as $key => $value ) {
						if ( is_array($value) ) {
							$value = implode(', ', $value);
						}
						if ( strpos($social_contact_form_message_fields, '_format_' . $key) ) {
							if ( strpos($social_contact_form_message_fields, $key) ) {
								$new_fields[ '[' . $key . ']' ] = $value;
							}
							$format = substr($social_contact_form_message_fields, strpos($social_contact_form_message_fields, '[_format_' . $key));
							$format = substr($format, 0, strpos($format, ']') + 1);
							$format = str_replace([ '[_format_' . $key . ' "', '"' ], '', $format);
							$timestamp = strtotime($value);
							$date = gmdate($format, $timestamp);
							$new_fields[ '[_format_' . $key . ' "' . $format . '"]' ] = ( false !== $date ) ? $date : '';
						} elseif ( strpos($social_contact_form_message_fields, '_raw_' . $key) ) {
							if ( strpos($social_contact_form_message_fields, $key) ) {
								$new_fields[ '[' . $key . ']' ] = $value;
							}
							if ( $form_fields ) {
								foreach ( $form_fields as $form_field ) {
									if ( isset($form_field->basetype) && 'select' === $form_field->basetype ) {
										$raw_index = array_search($value, $form_field->raw_values, true);
										if ( false !== $raw_index ) {
											$new_fields[ '[_raw_' . $key . ']' ] = $form_field->values[ $raw_index ];
										}
									}
								}
							}
						} else {
							$new_fields[ '[' . $key . ']' ] = $value;
						}
					}
				}

				$message = [];
				$message['data'] = strtr($social_contact_form_message_fields, $new_fields);

				/**
				 * Mail activation.
				 */

				if ( 'true' === $activemail && 'cf7' === $selectedmode ) {
					$data = explode("\n", $message['data']);
					$new_message = '';

					foreach ( $data as $field ) {
						$field = trim($field);
						if ( ! empty($field) ) {
							if ( strpos($field, '*') === 0 ) {
								// Extract the field name.
								$field_name = str_replace('*', '', $field);
								$new_message .= "{$field_name}<br>";
							} else {
								// Append the field value to the previous field name.
								$field_value = trim($field);
								$new_message .= "{$field_value}<br>";
							}
						}
					}

					// Send the email.
					$admin_user = get_userdata(1);
					$admin_email = $custommail ? $custommail : $admin_user->user_email;

					$headers[] = 'Content-Type: text/html; charset=UTF-8';
					wp_mail($admin_email, $posted_data['form_title'], $new_message, $headers);
				}
				// END .

				// Store all leads in DB.
				$message_id = Lead::cf7create( $contact_forms_id, $posted_data, $meta_data );
				do_action( 'scf_after_submit', $message_id, $posted_data );

				// Send to WhatsApp now it has no used as url set from JS with new update code.
				if ( 'on' != $social_contact_form_tabs_status ) {
					$wh_url = 'https://wa.me/' . $social_contact_form_number . '?text=' . urlencode(html_entity_decode( $message['data'] ));
				} else {
					$wh_url = 'https://web.whatsapp.com/send?phone=' . $social_contact_form_number . '&text=' . urlencode(html_entity_decode( $message['data'] ));
				}

				$scf_new_opt = [];
				$scf_new_opt['scf_whatsapp_url'] = $wh_url; // Send to WhatsApp now it has no used as url set from JS with new update code.
				$scf_new_opt['scf_whatsapp_number'] = $social_contact_form_number;
				$scf_new_opt['scf_whatsapp_data'] = $message['data'];
				$scf_new_opt['scf_new_tab'] = $social_contact_form_tabs_status;

				// Add nonce.
				$nonce = wp_create_nonce( 'social_contact_form_submission' );
				$scf_new_opt['nonce'] = $nonce;

				$cookie_name = 'social_contact_cf7_options';
				setcookie($cookie_name, json_encode($scf_new_opt), time() + ( 86400 * 30 ), '/');

			}
		}

		/**
		 * Get tab form id and disable mail.
		 *
		 * @param array $skip_mail CF7 Id and disable mail.
		 *
		 * @return mixed
		 */
		public function social_contact_form_disable_cf7_mail_sending( $skip_mail ) {

			$contact_form = WPCF7_ContactForm::get_current();
			$form_id = $contact_form->id();
			$social_contact_form_mail_status = get_post_meta( $form_id, '_social_contact_form_configuration', true )['social_contact_form_mail_status'];
			$social_contact_form_wpnonce = get_post_meta( $form_id, '_social_contact_form_configuration', true )['social_contact_form_wpnonce'];
			if ( $social_contact_form_wpnonce ) {
				// phpcs:ignore
				if ( isset( $_POST['_wpcf7'] ) && $_POST['_wpcf7'] == $form_id && 'on' == $social_contact_form_mail_status ) {
					return true;
				}
			}
			return $skip_mail;
		}

	}


	// Initialize Message class. Only if doing Message.
	new Message();
endif;
