<?php
/**
 * Form class.
 * Handles all Form requests.
 *
 * @package SocialContactForm
 * @since 1.0.0
 */

// Namespace.
namespace WPPOOL\SCF;

use WPCF7_ContactForm;
use WPCF7_Submission;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;


if ( ! class_exists( '\WPPOOL\SCF\Form' ) ) :
	/**
	 * Form class.
	 * Handles all Form requests.
	 *
	 * @package SocialContactForm
	 * @since 1.0.0
	 */
	final class Form {

		/**
		 * Constructor.
		 *
		 * @since 1.0.0
		 */
		public function __construct() {
			add_filter( 'wpcf7_editor_panels', [ $this, 'scf_custom_tab' ], 10, 1 );
			add_action( 'wpcf7_save_contact_form', [ $this, 'save_social_contact_form_meta_data' ] );
		}

		/**
		 * Create tab inside the CF7.
		 *
		 * @param array $panels Plugin Tabs and callback.
		 *
		 * @return mixed
		 */
		public function scf_custom_tab( $panels ) {
			$panels['new-tab'] = [
				'title' => __('Social Contact Form', 'social-contact-form'),
				'callback' => [ $this, 'wpcf7_editor_panel_new_tab' ],
			];
			return $panels;
		}

		/**
		 * Create tab and all its functionalities in CF7.
		 *
		 * @param array $post Plugin Id and post  meta.
		 *
		 * @return mixed
		 */
		public function wpcf7_editor_panel_new_tab( $post ) {
			$form = WPCF7_ContactForm::get_current();
			$ultimate = is_scf_ultimate() ? 'ultimate-mail' : 'scf-ultimate';
			$badge = is_scf_ultimate() ? '' : 'Ultimate';
			$social_contact_form_configuration = get_post_meta( $form->id(), '_social_contact_form_configuration', true );
			$social_contact_form_status = isset($social_contact_form_configuration['social_contact_form_status']) ? $social_contact_form_configuration['social_contact_form_status'] : '';
			$social_contact_form_message_fields = isset($social_contact_form_configuration['social_contact_form_message_fields']) ? $social_contact_form_configuration['social_contact_form_message_fields'] : '*Name*: [your-name] ' . "\n" . '*Email*: [your-email]' . "\n" . '*Subject*: [your-subject]' . "\n" . '*Message*:' . "\n" . '[your-message]' . "\n";
			$social_contact_form_country_code = isset($social_contact_form_configuration['social_contact_form_country_code']) ? $social_contact_form_configuration['social_contact_form_country_code'] : '';
			$social_contact_cnumber = isset($social_contact_form_configuration['social_contact_form_number']) ? $social_contact_form_configuration['social_contact_form_number'] : '';
			$social_contact_form_number = isset($social_contact_form_configuration['social_contact_form_number']) ? $social_contact_form_configuration['social_contact_form_number'] : '';
			$social_contact_form_tabs_status = isset($social_contact_form_configuration['social_contact_form_tabs_status']) ? $social_contact_form_configuration['social_contact_form_tabs_status'] : '';
			$social_contact_form_country_store = isset($social_contact_form_configuration['social_contact_form_country_store']) ? $social_contact_form_configuration['social_contact_form_country_store'] : '';
			$social_contact_form_mail_status = isset($social_contact_form_configuration['social_contact_form_mail_status']) ? $social_contact_form_configuration['social_contact_form_mail_status'] : '';
			$social_contact_form_cf7_nonce = wp_create_nonce('social_contact_form_cf7_nonce');

			$icon = plugin_dir_url(__FILE__) . '../../public/images/question.png';

			$phone_err = false;
			if ( isset($social_contact_form_configuration['social_contact_form_number']) && empty($social_contact_form_configuration['social_contact_form_number']) && 'on' == $social_contact_form_status ) {
				$phone_err = true;
			}

			?>
<h2><?php esc_html_e('Social Contact Form', 'social-contact-form'); ?></h2>
<fieldset>
	<!-- <legend><?php esc_html_e('Send messages through this form directly to your WhatsApp account while submitting', 'social-contact-form'); ?></legend> -->
	<legend><?php esc_html_e('Send messages through this form while submitting', 'social-contact-form'); ?>
		<span class='scf-desc'><?php esc_html_e('directly to your WhatsApp account', 'social-contact-form'); ?></span>
			<?php esc_html_e('account while submitting', 'social-contact-form'); ?>
	</legend>

	<table class="form-table">
		<tbody>
			<tr>
				<th scope="row">
					<div class="scf-left-content">
						<span class="tooltip"
							data-tooltip="Enabling  this feature will allow you to send the CF7 leads to your given WhatsApp number."
							data-tooltip-pos="right" data-tooltip-length="medium"><img
								src="<?php echo esc_url( $icon, 'social-contact-form'); ?>" alt="⊛"></span>
						<label for="social_contact_form_status">
							<?php esc_html_e('Connect WhatsApp', 'social-contact-form'); ?>
						</label>
					</div>
				</th>
				<td>
					<label class="social-contact-form-switch">
						<input type="checkbox" class="social-contact-form-checkbox" id="social_contact_form_status"
							name="social_contact_form_status" value="on" 
							<?php
							if ( 'on' == $social_contact_form_status ) {
								esc_attr_e('checked', 'social-contact-form');}
							?>
							>
						<span class="social-contact-form-slider social-contact-form-round"></span>
					</label>
					<p class="scf_toogle_desc">
						<?php esc_html_e('Enable to connect WhatsApp for this contact form.', 'social-contact-form'); ?>
					</p>
				</td>
			</tr>

			<tr>
				<th scope="row">
					<div class="scf-left-content">
						<span class="tooltip" data-tooltip="Provide your WhatsApp number to get the leads."
							data-tooltip-pos="right" data-tooltip-length="medium"><img
								src="<?php echo esc_url( $icon, 'social-contact-form'); ?>" alt="⊛"></span>
						<label
							for="social_contact_form_number"><?php esc_html_e('WhatsApp Number', 'social-contact-form'); ?></label>
					</div>
				</th>

				<td class="scf_country_list">
					<?php
								$countries = wp_json_file_decode( SCF_DIR . '/public/json/countries.json' );
								$selected = $social_contact_form_country_store ? $social_contact_form_country_store : '🏳️‍🌈 1-649';

								$allowed_html = [
									'span' => [
										'id' => [],
										'name' => [],
										'class' => [],
										'data-value' => [],
									],
									'img' => [
										'role' => [],
										'class' => [],
										'alt' => [],
										'src' => [],
									],
								];

								?>
					<div class="custom-dropdown">
						<input type="hidden" id="social_contact_form_country_code"
							name="social_contact_form_country_code"
							value="<?php echo esc_attr( $social_contact_form_country_code ); ?>">
						<input type="hidden" id="social_contact_form_country_store"
							name="social_contact_form_country_store"
							value="<?php echo esc_attr( $social_contact_form_country_store ); ?>">

						<span class="dropdown-label"
							data-value=""><?php echo wp_kses($selected,$allowed_html ); ?></span>

						<div class="dropdown-options">

							<div class="search">
								<input class="search-input" type="text" placeholder="Search country">
							</div>
							<div class="options">
								<?php foreach ( $countries as $country ) : ?>
									<?php

											$flag_icon = str_replace('\\', '', $country->emoji);
									?>
								<div class="option" data-value="<?php echo esc_html($country->phone_code); ?>">
									<span
										class="flag-icon"><?php echo esc_html($flag_icon); ?></span>(<?php echo esc_html($country->phone_code); ?><span>)-<?php echo esc_html($country->name); ?></span>
								</div>
								<?php endforeach; ?>
							</div>
						</div>
						<div class="carrot"></div>
						<input type="number" id="social_contact_form_number" name="social_contact_form_number"
							placeholder="<?php esc_html_e( 'Phone number', 'social-contact-form' ); ?>" class="large-text scf_whats_number 
								<?php
								if ( true == $phone_err ) {
									esc_attr_e('scf-input-error', 'social-contact-form');}
								?>
									" value="<?php echo esc_attr( $social_contact_cnumber ); ?>">
					</div>
				</td>

			<tr>
				<th scope="row">
					<div class="scf-left-content">
						<span class="tooltip"
							data-tooltip="Decorate this message field how you want to get your user's leads."
							data-tooltip-pos="right" data-tooltip-length="medium"><img
								src="<?php echo esc_url( $icon, 'social-contact-form'); ?>" alt="⊛"></span>

						<label><?php esc_html_e('Message', 'social-contact-form'); ?></label>
					</div>
				</th>
				<td>
					<p class="message_body_title">
						<?php esc_html_e('Use these tags in the message body:', 'social-contact-form'); ?></p>
					<span class="tags dynamic-tags"><?php $post->suggest_mail_tags(); ?></span>

					<span class="tags s-mailtag"><?php esc_html_e('[site_title]', 'social-contact-form'); ?></span>
					<span
						class="tags s-mailtag"><?php esc_html_e('[site_description]', 'social-contact-form'); ?></span>
					<span class="tags s-mailtag"><?php esc_html_e('[site_url]', 'social-contact-form'); ?></span>
					<span class="tags s-mailtag"><?php esc_html_e('[page_url]', 'social-contact-form'); ?></span>
					<span class="tags s-mailtag"><?php esc_html_e('[admin_email]', 'social-contact-form'); ?></span>
					<span class="tags s-mailtag"><?php esc_html_e('[form_title]', 'social-contact-form'); ?></span>
					<span class="tags s-mailtag"><?php esc_html_e('[form_id]', 'social-contact-form'); ?></span>
					<textarea id="social-contact-form_message_body" id="social_contact_form_message_fields"
						name="social_contact_form_message_fields" cols="100" rows="18"
						class="large-text code"><?php echo esc_attr( $social_contact_form_message_fields ); ?></textarea>
					<a target="_blank" style="text-decoration:underline"
						href="https://faq.whatsapp.com/general/chats/how-to-format-your-messages/?lang=en"><?php esc_html_e('Format your whatsapp message', 'social-contact-form'); ?></a>
					<p><?php esc_html_e('Note:', 'social-contact-form'); ?>
						<i><?php esc_html_e('File Upload field will not support on WhatsApp message body.', 'social-contact-form'); ?></i>
					</p>
				</td>
			</tr>

			<tr>
				<th scope="row">
					<div class="scf-left-content">
						<span class="tooltip"
							data-tooltip="Enabling this feature means, WhatsApp will be opened automatically on the web."
							data-tooltip-pos="right" data-tooltip-length="medium"><img
								src="<?php echo esc_url( $icon, 'social-contact-form'); ?>" alt="⊛"></span>
						<label><?php esc_html_e('Open in new tab', 'social-contact-form'); ?></label>
					</div>
				</th>
				<td>
					<label class="social-contact-form-switch">
						<input type="checkbox" class="social-contact-form-checkbox" id="social_contact_form_tabs_status"
							name="social_contact_form_tabs_status" value="on" 
							<?php
							if ( 'on' == $social_contact_form_tabs_status ) {
								esc_attr_e('checked', 'social-contact-form');}
							?>
							>
						<span class="social-contact-form-slider social-contact-form-round"></span>
					</label>
					<p class="scf_toogle_desc">
						<?php esc_html_e('Enable to open whatsapp in new tab.', 'social-contact-form'); ?></p>
					<p><i><?php esc_html_e('Note:', 'social-contact-form'); ?>
							<?php esc_html_e('This option is for only desktop devices, It will be useful for WhatsApp web on desktop devices.', 'social-contact-form'); ?></i>
					</p>
				</td>
			</tr>

			<tr>
				<th scope="row">
					<div class="scf-left-content">
						<span class="tooltip"
							data-tooltip="Enabling this feature will disable the email functionality of this form."
							data-tooltip-pos="right" data-tooltip-length="medium"><img
								src="<?php echo esc_url( $icon, 'social-contact-form'); ?>" alt="⊛"></span>
						<label><?php esc_html_e('Disable email notifications', 'social-contact-form'); ?></label>
					</div>
				</th>
				<td class="social-contact-form-switch scf-ultiamte-feature" id="<?php echo esc_html( $ultimate ); ?>">
					<label class="social-contact-form-switch <?php echo esc_html( $ultimate ); ?>">
						<input type="checkbox" class="social-contact-form-checkbox" id="social_contact_form_mail_status"
							name="social_contact_form_mail_status" value="on" 
							<?php
							if ( 'on' == $social_contact_form_mail_status && is_scf_ultimate() ) {
								esc_attr_e('checked', 'social-contact-form');}
							?>
							>
						<span class="social-contact-form-slider social-contact-form-round"></span>
					</label>
					<h4 class='ultimate_badge <?php echo esc_html( $badge ); ?>'><?php echo esc_html( $badge ); ?></h4>
					<p class="scf_toogle_desc <?php echo esc_html( $ultimate ); ?>">
						<?php esc_html_e('Enable this toggle to stop receiving email notifications from this form.', 'social-contact-form'); ?>
					</p>

					<p><?php esc_html_e('Note:', 'social-contact-form'); ?>
						<i><?php esc_html_e('With this function enabled, you can use this Contact Form 7 form without using any SMTP plugin.', 'social-contact-form'); ?></i>
					</p>

				</td>
			</tr>

			<input type="hidden" name="social-contact-form-cf7-nonce" id="social-contact-form-cf7-nonce" value="
						<?php
						//phpcs:ignore
						esc_attr_e($social_contact_form_cf7_nonce, 'social-contact-form');
						?>
						">
		</tbody>
	</table>
</fieldset>
			<?php
		}

		/**
		 * Create tab and all its functionalities in CF7.
		 *
		 * @param array $cf7 CF7 Id and get form submitted data.
		 *
		 * @return mixed
		 */
		public function save_social_contact_form_meta_data( $cf7 ) {
			$social_contact_form_wpnonce = isset( $_POST['social-contact-form-cf7-nonce'] ) ? sanitize_text_field( wp_unslash($_POST['social-contact-form-cf7-nonce']) ) : null;
			$scf_plugin_status = isset( $_POST['social_contact_form_status'] ) ? sanitize_text_field( wp_unslash($_POST['social_contact_form_status']) ) : '';
			$social_contact_form_country_code = isset( $_POST['social_contact_form_country_code'] ) ? sanitize_text_field( wp_unslash($_POST['social_contact_form_country_code'] )) : '';
			$social_contact_form_number = isset( $_POST['social_contact_form_number'] ) ? sanitize_text_field( wp_unslash($_POST['social_contact_form_number'] )) : '';
			$social_contact_form_message_fields = isset( $_POST['social_contact_form_message_fields'] ) ? sanitize_textarea_field( wp_unslash($_POST['social_contact_form_message_fields']) ) : '';
			$social_contact_form_tabs_status = isset( $_POST['social_contact_form_tabs_status'] ) ? sanitize_text_field( wp_unslash($_POST['social_contact_form_tabs_status']) ) : '';
			$social_contact_form_mail_status = isset( $_POST['social_contact_form_mail_status'] ) ? sanitize_text_field( wp_unslash($_POST['social_contact_form_mail_status']) ) : '';
			$social_contact_form_country_store = isset( $_POST['social_contact_form_country_store'] ) ? wp_kses_post( wp_unslash($_POST['social_contact_form_country_store'] ) ) : '';
			$post_id = $cf7->id();
			$form_title = $cf7->title();

			// Add the country code to the phone number.
			$full_phone_number = $social_contact_form_country_code . $social_contact_form_number;

			$meta_values = [
				'form_id' => $post_id,
				'form_title' => $form_title,
				'social_contact_form_status' => $scf_plugin_status,
				'social_contact_form_country_code' => $social_contact_form_country_code,
				'social_contact_form_number' => $social_contact_form_number,
				'full_phone_number' => $full_phone_number,
				'social_contact_form_message_fields' => $social_contact_form_message_fields,
				'social_contact_form_tabs_status' => $social_contact_form_tabs_status,
				'social_contact_form_mail_status' => $social_contact_form_mail_status,
				'social_contact_form_country_store' => $social_contact_form_country_store,
				'social_contact_form_wpnonce' => $social_contact_form_wpnonce,
			];
			if ( wp_verify_nonce($social_contact_form_wpnonce, 'social_contact_form_cf7_nonce') ) {
				update_post_meta( $post_id, '_social_contact_form_configuration', $meta_values );
			}
		}


	}

	// Initialize Form class. Only if doing Form.
	new Form();
endif;
