<?php
/**
 * Hooks class.
 * Register all the hooks.
 *
 * @package SocialContactForm
 * @since 1.0.0
 */

// Namespace .
namespace WPPOOL\SCF;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;


if ( ! class_exists( '\WPPOOL\SCF\Hooks' ) ) :
	/**
	 * Hooks class.
	 * Register all the hooks.
	 *
	 * @package SocialContactForm
	 * @since 1.0.0
	 */
	class Hooks {

		// Use Common trait.
		use \WPPOOL\SCF\Common;

		/**
		 * Constructor.
		 *
		 * @since 1.0.0
		 */
		public function __construct() {
			add_action( 'wp_enqueue_scripts', [ $this, 'wp_enqueue_scripts' ] );
			add_action( 'wp_footer', [ $this, 'add_scf_contact_form_to_footer' ], 99 );
			add_action( 'wp_ajax_scf_submit_form', [ $this, 'scf_submit_form' ] );
			add_action( 'wp_ajax_nopriv_scf_submit_form', [ $this, 'scf_submit_form' ] );

			add_action('admin_init', [ $this, 'add_scf_form_new_option_values' ]);
		}


		/**
		 * Enqueue scripts and styles.
		 *
		 * @return void
		 */
		public function wp_enqueue_scripts() {
			$scf        = scf_get_options();
			$is_enabled = isset( $scf['enable'] ) ? $scf['enable'] : 'no';

			if ( scf_is_true( $is_enabled ) ) {
				wp_enqueue_style( 'scf-frontend', SCF_PUBLIC . '/css/frontend.min.css', [], time() );
				wp_enqueue_script( 'scf-frontend', SCF_PUBLIC . '/js/frontend.min.js', [ 'jquery' ], time(), true );
				wp_localize_script(
					'scf-frontend',
					'_scf',
					apply_filters( 'scf_localize_script', [
						'ajax_url'    => admin_url( 'admin-ajax.php' ),
						'nonce'       => wp_create_nonce( 'scf_form_security_nonce' ),
						'is_pro'      => is_scf_ultimate(),
						'options'     => $scf,
						'development' => scf_is_dev(),
						'tags'        => apply_filters( 'scf_tags', [
							'title' => get_the_title(),
							'url'   => get_permalink(),
							'id'    => get_the_ID(),
						] ),
					] )
				);

				// Add inline style.
				$inline_style = $this->get_inline_style( $scf );
				if ( ! empty( $inline_style ) ) {
					wp_add_inline_style( 'scf-frontend', $inline_style );
				}
			}
		}


		/**
		 * Get inline style.
		 *
		 * @param mixed $scf scf options.
		 * @return string
		 */
		public function get_inline_style( $scf = null ) {
			if ( ! $scf ) {
				$scf = scf_get_options();
			}
			$css = '';

			if ( isset( $scf['custom_css'] ) ) {
				$css .= $scf['custom_css'];
			}

			// customizable.
			$font          = isset( $scf['form'] ) ? $scf['form']['font'] : 'Open Sans';
			$font_url      = SCF_PUBLIC . '/fonts/' . ( str_replace( ' ', '+', $font ) ) . '.ttf';
			$font_family   = $this->format_font( $font );
			$premium_fonts = App::premium_fonts();

			if ( in_array( $font, $premium_fonts ) ) {
				$font_url = SCF_PUBLIC . '/fonts/' . ( str_replace( ' ', '+', $font ) ) . '-Regular.ttf';

				$css .= "
				@font-face {
					font-family: {$font_family} !important; 
					src: url({$font_url}) format('ttf') !important;	
					font-style: normal !important;
					font-weight: 400 !important; 
				  }";

			}

			$css .= "
			:root {
				--form_color: {$scf['form']['color']};
				--form_background: {$scf['form']['background']};
				--cta_color:  {$scf['call_to_action']['color']};
				--cta_background: {$scf['call_to_action']['background']};
				--font_family: {$font_family};
			}";

			return esc_html( $css );
		}


		/**
		 * Add contact form to footer.
		 *
		 * @return void
		 */
		public function add_scf_contact_form_to_footer() {
			$is_enabled = scf_get_option( 'enabled' );

			if ( scf_is_true( $is_enabled ) ) {
				$scf_activeform = scf_get_option( 'activeform' );
				$showform = isset( $scf_activeform['showform'] ) ? $scf_activeform['showform'] : false;

				if ( scf_is_true( $showform ) ) {
					global $scf;
					$scf = scf_get_options();
					include_once SCF_TEMPLATE . '/frontend/base.php';
				}
			}
		}


		/**
		 * Sanitize form data.
		 *
		 * @param mixed $data data.
		 * @return array
		 */
		public function sanitize( $data ) {
			if ( is_array( $data ) ) {
				foreach ( $data as $key => $value ) {
					$data[ $key ] = $this->sanitize( $value );
				}
			} else {
				$data = sanitize_text_field( $data );
			}

			return $data;
		}


		/**
		 * Submit form.
		 *
		 * @return void
		 */
		public function scf_submit_form() {
			$scf        = scf_get_options();
			$is_enabled = scf_is_true( scf_get_option( 'enabled' ) );
			$selectedmode = $scf['contact_form_7']['selectedmode'];
			$activemail = $scf['contact_form_7']['activemail'];
			$custommail = $scf['contact_form_7']['cf7mail'];

			if ( ! $is_enabled ) {
				wp_send_json_error( [ 'message' => __( 'Contact form is disabled.', 'social-contact-form' ) ] );
				wp_die();
			}

			// Check nonce.
			check_ajax_referer( 'scf_form_security_nonce', 'nonce' );

			// Sanitized in sanitize().
			$input_fields = isset( $_POST['fields'] ) ? $this->sanitize( $_POST['fields'] ) : []; // phpcs:ignore
			// phpcs:ignore
			$input_meta = isset($_POST['meta']) ? $this->sanitize(wp_unslash($_POST['meta'])) : '';

			$fields = [];
			if ( App::fields() ) {
				foreach ( App::fields() as $field ) {
					$fields[ $field['name'] ] = $input_fields[ $field['name'] ];
				}
			}
			$meta   = array_merge( $input_meta, $input_meta );
			$fields = apply_filters( 'scf_form_data', $fields );

			/**
			 * Mail activation.
			 */
			if ( 'true' === $activemail && 'scf' === $selectedmode ) {
				$new_message = '';

				foreach ( $fields as $field_name => $field_value ) {
					// Exclude 'code' field from being added to the $new_message.
					if ( 'code' !== $field_name ) {
						$field_name = str_replace('[', '', $field_name);
						$new_message .= "{$field_name}: {$field_value}<br>";
					}
				}

				// Send the email.
				$admin_user = get_userdata(1);
				$admin_email = $custommail ? $custommail : $admin_user->user_email;
				$headers[] = 'Content-Type: text/html; charset=UTF-8';
				// $headers[] = "From: '.$admin_email."; // remove.
				$title = 'Social Contact Form';
				wp_mail($admin_email, $title, $new_message, $headers);
			}
			// END .

			$meta   = apply_filters( 'scf_form_meta', $meta );
			do_action( 'scf_before_submit', $fields );
			$message_id = Lead::create( $fields, $meta );
			do_action( 'scf_after_submit', $message_id, $fields );
			wp_send_json_success(
				[
					'message'    => __( 'Message sent successfully', 'social-contact-form' ),
					'message_id' => $message_id,
				]
			);
			wp_die();
		}

		/**
		 * Regenerate default value for 'allow_country_selection' & 'modepress' if not present.
		 */
		public function add_scf_form_new_option_values() {

			$scf        = scf_get_options();

			$admin_user = get_userdata(1);
			$admin_email = $admin_user->user_email;

			$name  = 'scf_form';
			// Form.
			if ( ! isset($scf['form']['allow_country_selection']) ) {
				$options = [
					'title'           => 'Contact via WhatsApp',
					'header'          => '',
					'footer'          => '',
					'submit'          => 'Send on WhatsApp',
					'open_by_default' => false,
					'allow_country_selection' => 'true',
					'phone_code' => '44',
					'phone_emoji' => '&#x1F1EC;&#x1F1E7;',
					'close_on_submit' => true,
					'open_in_new_tab' => true,
					'size'            => 'medium',
					'size_custom'     => '',
					'font'            => '',
					'color'           => '#ffffff',
					'background'      => '#09816D',
				];

				update_option( $name, $options );

			}

			// Contact Form 7.
			$contact_seven_form  = 'scf_contact_form_7';
			$modepress = isset($scf['contact_form_7']['selectedmode']) ? $scf['contact_form_7']['selectedmode'] : 'scf';
			$cf7_not_activated = isset($scf['contact_form_7']['cf7NotActivated']) ? $scf['contact_form_7']['cf7NotActivated'] : 'true';
			$cf7_installed = isset($scf['contact_form_7']['cf7Installed']) ? $scf['contact_form_7']['cf7Installed'] : 'false';
			$selectedid = isset($scf['contact_form_7']['selectedid']) ? $scf['contact_form_7']['selectedid'] : '';
			$confirmationstateof_c_f7 = isset($scf['contact_form_7']['confirmationstateofCF7']) ? $scf['contact_form_7']['confirmationstateofCF7'] : 'false';
			$confirmationstateof_custommail = isset($scf['contact_form_7']['confirmationstateofCustommail']) ? $scf['contact_form_7']['confirmationstateofCustommail'] : 'true';
			$formrender = isset($scf['contact_form_7']['formrender']) ? $scf['contact_form_7']['formrender'] : '';

			if ( ! isset($scf['contact_form_7']['modepress']) || ! isset($scf['contact_form_7']['defaultmail']) ) {
				$cf7options = [
					'selectedmode'        => $modepress,
					'cf7NotActivated'        => $cf7_not_activated,
					'cf7Installed'        => $cf7_installed,
					'modepress'        => $modepress,
					'selectedid'          => $selectedid,
					'configurationformname'          => '',
					'confirmationstateofCF7'          => $confirmationstateof_c_f7,
					'confirmationstateofCustommail'          => $confirmationstateof_custommail,
					'formrender'          => $formrender,
					'activemail'        => 'false',
					'cf7mail'        => $admin_email,
					'defaultmail'        => $admin_email,
				];

				update_option( $contact_seven_form, $cf7options );

			}

		}

	}

	/**
	 * Initiate Hooks.
	 */
	new Hooks();
endif;
