<?php
/**
 * Field class.
 *
 * @package SocialContactForm
 * @since 1.0.0
 */

// Namespace .
namespace WPPOOL\SCF;

use WPPOOL\SCF\App;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;


if ( ! class_exists( '\WPPOOL\SCF\Field' ) ) :
	/**
	 * Field class.
	 *
	 * @package SocialContactForm
	 * @since 1.0.0
	 */
	final class Field {
		/**
		 * Field.
		 *
		 * @var array
		 */
		public $field;

		/**
		 * Constructor.
		 *
		 * @param array $field Field.
		 */
		public function __construct( $field ) {
			$this->field = $field;

			$this->init();
			$this->render_field();
		}

		/**
		 * Init field.
		 */
		public function init() {
			$this->field['type']           = isset( $this->field['type'] ) ? $this->field['type'] : 'text';
			$this->field['name']           = isset( $this->field['name'] ) ? $this->field['name'] : '';
			$this->field['description']    = isset( $this->field['description'] ) ? $this->field['description'] : '';
			$this->field['default']        = isset( $this->field['default'] ) ? $this->field['default'] : '';
			$this->field['options']        = isset( $this->field['options'] ) ? $this->field['options'] : [];
			$this->field['placeholder']    = isset( $this->field['placeholder'] ) ? $this->field['placeholder'] : '';
			$this->field['class']          = isset( $this->field['class'] ) ? $this->field['class'] : '';
			$this->field['style']          = isset( $this->field['style'] ) ? $this->field['style'] : '';
			$this->field['required']       = isset( $this->field['required'] ) ? $this->field['required'] : false;
			$this->field['disabled']       = isset( $this->field['disabled'] ) ? $this->field['disabled'] : false;
			$this->field['readonly']       = isset( $this->field['readonly'] ) ? $this->field['readonly'] : false;
			$this->field['multiple']       = isset( $this->field['multiple'] ) ? $this->field['multiple'] : false;
			$this->field['autocomplete']   = isset( $this->field['autocomplete'] ) ? $this->field['autocomplete'] : false;
			$this->field['autofocus']      = isset( $this->field['autofocus'] ) ? $this->field['autofocus'] : false;
			$this->field['autocorrect']    = isset( $this->field['autocorrect'] ) ? $this->field['autocorrect'] : false;
			$this->field['autocapitalize'] = isset( $this->field['autocapitalize'] ) ? $this->field['autocapitalize'] : false;
			$this->field['spellcheck']     = isset( $this->field['spellcheck'] ) ? $this->field['spellcheck'] : false;
			$this->field['min']            = isset( $this->field['min'] ) ? $this->field['min'] : '';
			$this->field['max']            = isset( $this->field['max'] ) ? $this->field['max'] : '';
			$this->field['step']           = isset( $this->field['step'] ) ? $this->field['step'] : '';
		}

		/**
		 * Render field.
		 *
		 * @return void
		 */
		public function render_field() {
			do_action( 'scf_before_field', $this->field );

			$class_name = 'scf-form-group-' . $this->field['name'];

			echo '<div class="scf-form-group ' . esc_attr( $class_name ) . '">';

			switch ( $this->field['type'] ) {
				case 'text':
					$this->render_text();
					break;
				case 'textarea':
					$this->render_textarea();
					break;
				case 'select':
					$this->render_select();
					break;
				case 'checkbox':
					$this->render_checkbox();
					break;
				case 'radio':
					$this->render_radio();
					break;
				case 'hidden':
					$this->render_hidden();
					break;
				default:
					$this->render_text();
					break;
			}

			echo '</div>';

			do_action( 'scf_after_field', $this->field );
		}

		/**
		 * Render text field.
		 *
		 * @return void
		 */
		public function render_text() {

			$scf        = scf_get_options();
			$activated = $scf['form']['allow_country_selection'];
			$ultimate = is_scf_ultimate() ? 'true' : 'false';
			$phone_code = $scf['form']['phone_code'];
			$phone_emoji = $scf['form']['phone_emoji'];
			$selected_country = $phone_code ? $phone_code : 'Choose one';
			// phpcs:ignore
			$selected = $phone_code ? $phone_emoji . ' ' . '(' . $phone_code . ')' : '&#x1F1EC;&#x1F1E7;
			(44)';
			$field    = (object) $this->field;
			$required = true === $field->required ? 'required' : '';
			$pattern  = 'tel' === $field->type ? 'pattern="\+(?:[0-9] ?){6,14}[0-9]$" ' : '';

			if ( 'true' === $ultimate ) {

				if ( 'true' === $activated && 'tel' === $field->type ) {

					$countries = wp_json_file_decode( SCF_DIR . '/public/json/countries.json' );

					$allowed_html = [
						'span' => [
							'id' => [],
							'name' => [],
							'class' => [],
							'data-value' => [],
						],
						'img' => [
							'role' => [],
							'class' => [],
							'alt' => [],
							'src' => [],
						],
					];

					echo '<div class="custom-dropdown">';
					echo '<span id="scf_phone_code" class="dropdown-label" data-flag="' . esc_attr($phone_emoji) . '" data-value="' . ( isset($phone_code) && '' !== $phone_code ? esc_attr($phone_code) : '44' ) . '">' . wp_kses($selected, $allowed_html) . '</span>';

					echo '<div class="dropdown-options">';
						echo '<div class="search">';
							echo '<input class="search-input" type="text" placeholder="Search country">';
						echo '</div>';
						echo '<div class="options">';
					foreach ( $countries as $country ) {
						$flag_icon = str_replace('\\', '', $country->emoji);
						echo '<div class="option" data-value="' . esc_html($country->phone_code) . '">';
						echo '<span class="flag-icon">' . esc_html($flag_icon) . '</span>(' . esc_html($country->phone_code) . '<span>)-' . esc_html($country->name) . '</span>';
						echo '</div>';
					}
						echo '</div>';
						echo '</div>';
						echo '<svg class="w-4 h-4 text-gray-500 scf-carrot" viewBox="0 0 20 20" fill="currentColor">
							<path fill-rule="evenodd" d="M10 12l-6-6 1.5-1.5L10 9l4.5-4.5L16 6z" clip-rule="evenodd" />
							</svg>';

						echo '<input type="' . esc_attr($field->type) . '" id="' . esc_attr($field->name) . '" name="' . esc_attr($field->name) . '" ' . esc_attr($required) . ' ' . esc_attr($pattern) . ' placeholder="' . esc_attr($field->placeholder) . '" class="scf-form-phone-field" />';
						echo '</div>';
				} else {
					echo '<input type="' . esc_attr( $field->type ) . '" id="' . esc_attr( $field->name ) . '" name="' . esc_attr( $field->name ) . '" ' . esc_attr( $required ) . ' ' . esc_attr($pattern) . ' placeholder="' . esc_attr( $field->placeholder ) . '" />';
				}
			} else {
				echo '<input type="' . esc_attr( $field->type ) . '" id="' . esc_attr( $field->name ) . '" name="' . esc_attr( $field->name ) . '" ' . esc_attr( $required ) . ' ' . esc_attr($pattern) . ' placeholder="' . esc_attr( $field->placeholder ) . '" />';
			}

		}

		/**
		 * Render textarea field.
		 *
		 * @return void
		 */
		public function render_textarea() {
			$field    = (object) $this->field;
			$required = true === $field->required ? 'required' : '';
			echo '<textarea id="' . esc_attr( $field->name ) . '" name="' . esc_attr( $field->name ) . '" ' . esc_attr( $required ) . ' placeholder="' . esc_attr( $field->placeholder ) . '"></textarea>';
		}

		/**
		 * Render checkbox field.
		 *
		 * @return void
		 */
		public function render_select() {
			$field    = (object) $this->field;
			$required = true === $field->required ? 'required' : '';
			$options  = isset( $field->options ) ? $field->options : [];
			$scf        = scf_get_options();
			$ultimate = is_scf_ultimate() ? 'true' : 'false';
			$activated = $scf['form']['allow_country_selection'];
			$phone_code = $scf['form']['phone_code'];
			$phone_emoji = $scf['form']['phone_emoji'];
			$selected_country = $phone_code ? $phone_code : 'Choose one';

			$selected = $phone_code ? $phone_code : '44';

				echo '<select id="' . esc_attr( $field->name ) . '" name="' . esc_attr( $field->name ) . '" ' . esc_attr( $required ) . ' style="display: none;">';

			foreach ( $options as $key => $value ) {
				$selected_option = $selected === $key ? 'selected' : '';
				echo '<option value="' . esc_attr( $key ) . '" ' . esc_attr( $selected_option ) . '>' . esc_html( $value ) . '</option>';
			}

				echo '</select>';
		}


		/**
		 * Render checkbox field.
		 *
		 * @return void
		 */
		public function render_checkbox() {
			$field    = (object) $this->field;
			$required = true === $field->required ? 'required' : '';
			$checked  = true === $field->default ? 'checked' : '';
			$label    = isset( $field->label ) ? $field->label : '';
			echo '<div class="scf-checkbox-wrap">';
			echo '<input type="checkbox" id="' . esc_attr( $field->name ) . '" name="' . esc_attr( $field->name ) . '" ' . esc_attr( $required ) . ' ' . esc_attr( $checked ) . ' />';
			echo '<label for="' . esc_attr( $field->name ) . '">' . esc_html( $label ) . '</label>';
			echo '</div>';
		}

		/**
		 * Render radio field.
		 *
		 * @return void
		 */
		public function render_radio() {
			$field    = (object) $this->field;
			$required = true === $field->required ? 'required' : '';
			$default  = isset( $field->default ) ? $field->default : '';
			$options  = isset( $field->options ) ? $field->options : [];

			echo '<div class="scf-radio-wrap">';
			foreach ( $options as $key => $value ) {
				$checked = $default === $key ? 'checked' : '';
				echo '<div class="scf-radio-item">';
				echo '<input type="radio" id="' . esc_attr( $field->name . '_' . $key ) . '" name="' . esc_attr( $field->name ) . '" ' . esc_attr( $required ) . ' ' . esc_attr( $checked ) . ' value="' . esc_attr( $key ) . '" />';
				echo '<label for="' . esc_attr( $field->name . '_' . $key ) . '">' . esc_html( $value ) . '</label>';
				echo '</div>';
			}
			echo '</div>';
		}

		/**
		 * Render hidden field.
		 *
		 * @return void
		 */
		public function render_hidden() {
			$field = (object) $this->field;
			$value = isset( $field->default ) ? $field->default : '';
			echo '<input type="hidden" id="' . esc_attr( $field->name ) . '" name="' . esc_attr( $field->name ) . '" value="' . esc_attr( $value ) . '" />';
		}
	}
endif;
