<?php
/**
 * App class.
 *
 * @package SocialContactForm
 * @since 1.0.0
 */

// Namespace .
namespace WPPOOL\SCF;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;


if ( ! class_exists( '\WPPOOL\SCF\App' ) ) :
	/**
	 * App class.
	 *
	 * @package SocialContactForm
	 * @since 1.0.0
	 */
	final class App {

		/**
		 * Returns the default options.
		 *
		 * @return array
		 */
		public static function options() {
			$admin_user = get_userdata(1);
			$admin_email = $admin_user->user_email;

			$options = [
				'enabled'        => true,
				'whatsapp'       => [
					'direct_web'     => true,
					'phone_code'     => '44',
					'phone_number'   => '',
					'defined_preset' => apply_filters( 'scf_whatsapp_preset', 'Name: {name} {break}Email: {email} {break}Phone: {phone} {break}Message: {message}' ),
					'preset'         => '',
				],
				'icon'           => [
					'delay'           => 0,
					'image'           => '',
					'size'            => 'medium',
					'size_custom'     => '',
					'position'        => 'right',
					'position_top'    => '',
					'position_left'   => '',
					'position_right'  => '',
					'position_bottom' => '',
				],
				'call_to_action' => [
					'text'        => 'Contact us',
					'color'       => '#555555',
					'background'  => '#FFFFFF',
					'size'        => 'medium',
					'size_custom' => '',
				],
				'form'           => [
					'title'           => 'Contact via WhatsApp',
					'header'          => '',
					'footer'          => '',
					'submit'          => 'Send on WhatsApp',
					'open_by_default' => false,
					'allow_country_selection' => 'true',
					'phone_code' => '44',
					'phone_emoji' => '&#x1F1EC;&#x1F1E7;',
					'close_on_submit' => true,
					'open_in_new_tab' => true,
					'size'            => 'medium',
					'size_custom'     => '',
					'font'            => '',
					'color'           => '#ffffff',
					'background'      => '#09816D',
				],
				'contact_form_7'         => [
					'selectedmode'        => 'scf',
					'cf7NotActivated'        => 'true',
					'cf7Installed'        => 'false',
					'modepress'        => 'scf',
					'selectedid'          => '',
					'configurationformname'          => '',
					'confirmationstateofCF7'          => 'false',
					'confirmationstateofCustommail'          => 'true',
					'formrender'          => '',
					'activemail'        => 'false',
					'cf7mail'        => $admin_email,
					'defaultmail'        => $admin_email,
				],
				'activeform'        => [
					'showform'        => 'false',
				],
			];

			return apply_filters( 'scf_options', $options );
		}

		/**
		 * Returns the default fields.
		 *
		 * @return array
		 */
		public static function fields() {
			$countries = self::countries();
			$scf = scf_get_options();
			$allow_country_selection = $scf['form']['allow_country_selection'];

			if ( 'true' == $allow_country_selection ) {

				$phone_code = isset($scf['form']['phone_code']) ? $scf['form']['phone_code'] : '44';

				$country_code_options = [];
				foreach ( $countries as $country ) {
					$country_code_options[ $country['phone_code'] ] = $country['phone_code'];
				}

				// Add $phone_code value at the beginning and $country_code_options and make it the key.
				$country_code_options = array_combine(array_merge([ $phone_code ], $country_code_options), array_merge([ $phone_code ], $country_code_options));

				// Add the 'code' field only when $allow_country_selection is true.
				$country_fields[] = [
					'name' => 'code',
					'type' => 'select',
					'label' => 'Country Code',
					'placeholder' => __('Enter your phone', 'social-contact-form'),
					'required' => true,
					'options' => $country_code_options,
				];
			}

			// Common fields that are always present.
			$common_fields = [
				[
					'name' => 'name',
					'type' => 'text',
					'label' => 'Name',
					'placeholder' => __('Enter your name', 'social-contact-form'),
					'required' => true,
				],
				[
					'name' => 'email',
					'type' => 'email',
					'label' => 'Email',
					'placeholder' => __('Enter your email', 'social-contact-form'),
					'required' => true,
				],
				[
					'name' => 'phone',
					'type' => 'tel',
					'label' => 'Phone',
					'placeholder' => __('Enter your phone', 'social-contact-form'),
					'required' => true,
				],
				[
					'name' => 'message',
					'type' => 'textarea',
					'label' => 'Message',
					'placeholder' => __('Enter your message', 'social-contact-form'),
					'required' => true,
				],
			];

			// Combine the common fields with the 'code' field (if present).
			$fields = isset($country_fields) ? array_merge($common_fields, $country_fields) : $common_fields;

			return apply_filters('scf_fields', $fields);
		}


		/**
		 * Returns the default fonts.
		 *
		 * @return array
		 */
		public static function fonts() {
			$fonts = [
				'sans-serif'       => __( 'Default', 'social-contact-form' ),
				'Arial'            => __( 'Arial', 'social-contact-form' ),
				'Arial Black'      => __( 'Arial Black', 'social-contact-form' ),
				'Comic Sans'       => __( 'Comic Sans MS', 'social-contact-form' ),
				'Courier New'      => __( 'Courier New', 'social-contact-form' ),
				'Georgia'          => __( 'Georgia', 'social-contact-form' ),
				'Lucida Console'   => __( 'Lucida Console', 'social-contact-form' ),
				'Lucida Sans'      => __( 'Lucida Sans', 'social-contact-form' ),
				'Tahoma'           => __( 'Tahoma', 'social-contact-form' ),
				'Times New Roman'  => __( 'Times New Roman', 'social-contact-form' ),
				'Trebuchet'        => __( 'Trebuchet', 'social-contact-form' ),
				'Verdana'          => __( 'Verdana', 'social-contact-form' ),
				'Ubuntu'           => __( 'Ubuntu', 'social-contact-form' ),

				'Roboto'           => __( 'Roboto', 'social-contact-form' ),
				'Roboto Condensed' => __( 'Roboto Condensed', 'social-contact-form' ),
				'Open Sans'        => __( 'Open Sans', 'social-contact-form' ),
				'Lato'             => __( 'Lato', 'social-contact-form' ),
				'Montserrat'       => __( 'Montserrat', 'social-contact-form' ),
				'Raleway'          => __( 'Raleway', 'social-contact-form' ),
				'PT Sans'          => __( 'PT Sans', 'social-contact-form' ),
				'Roboto Slab'      => __( 'Roboto Slab', 'social-contact-form' ),
				'Merriweather'     => __( 'Merriweather', 'social-contact-form' ),
				'Playfair Display' => __( 'Playfair Display', 'social-contact-form' ),
				'Source Sans Pro'  => __( 'Source Sans Pro', 'social-contact-form' ),
				'Noto Sans'        => __( 'Noto Sans', 'social-contact-form' ),
				'Noto Serif'       => __( 'Noto Serif', 'social-contact-form' ),
				'Roboto Mono'      => __( 'Roboto Mono', 'social-contact-form' ),
				'Nunito'           => __( 'Nunito', 'social-contact-form' ),
				'Poppins'          => __( 'Poppins', 'social-contact-form' ),
				'Rubik'            => __( 'Rubik', 'social-contact-form' ),

			];

			return apply_filters( 'scf_fonts', $fonts );
		}

		/**
		 * Returns the default countries.
		 *
		 * @return array
		 */
		public static function countries() {
			$countries = wp_json_file_decode( SCF_DIR . '/public/json/countries.json' );

			// Check if the file exists.
			if ( ! $countries ) {
				return [];
			}

			// map only name, phone_code, emoji.
			$countries = array_map( function ( $country ) {
				return [
					'name'       => isset( $country->name ) ? $country->name : '',
					'phone_code' => isset( $country->phone_code ) ? $country->phone_code : '',
					'emoji'      => isset( $country->emoji ) ? $country->emoji : '',
					'tags'       => isset( $country->tags ) ? $country->tags : '',
				];
			}, $countries );

			return apply_filters( 'scf_countries', $countries );
		}

		/**
		 * Returns only the supported tags.
		 *
		 * @return mixed
		 */
		public static function supported_tags() {
			$tags = [
				'break' => '',
			];

			return apply_filters( 'scf_preset_tags', $tags );
		}

		/**
		 * Returns premium fonts.
		 *
		 * @return array
		 */
		public static function premium_fonts() {
			$fonts = [
				'Ubuntu',
				'Roboto',
				'Roboto Condensed',
				'Open Sans',
				'Lato',
				'Montserrat',
				'Raleway',
				'PT Sans',
				'Roboto Slab',
				'Merriweather',
				'Playfair Display',
				'Source Sans Pro',
				'Noto Sans',
				'Noto Serif',
				'Roboto Mono',
				'Nunito',
				'Poppins',
				'Rubik',
			];

			return apply_filters( 'scf_premium_fonts', $fonts );
		}
	}

	// Initialize the plugin.
	new App();
endif;