<?php
/**
 * Admin class.
 * Handles all admin related functionality.
 *
 * @package SocialContactForm
 * @since 1.0.0
 */

// Namespace .
namespace WPPOOL\SCF;

// Exit if accessed directly.
defined('ABSPATH') || exit;


if ( ! class_exists('\WPPOOL\SCF\Admin') ) :
	/**
	 * Admin class.
	 * Handles all admin related functionality.
	 *
	 * @package SocialContactForm
	 * @since 1.0.0
	 */
	final class Admin {

		/**
		 * Appsero client.
		 *
		 * @var object
		 */

		public $appsero = null;

		// Use Common trait.
		use \WPPOOL\SCF\Common;

		/**
		 * Constructor.
		 *
		 * @since 1.0.0
		 */
		public function __construct() {
			register_activation_hook(SCF_FILE, [ $this, 'activate_scf' ]);

			// appsero.
			$this->init_appsero();

			add_action('init', [ $this, 'register_options' ]);
			add_action('admin_init', [ $this, 'handle_safe_redirection' ]);
			add_action('admin_init', [ $this, 'cf7_leads_table' ]);
			add_action('admin_menu', [ $this, 'register_admin_menu' ], 0);
			add_action('admin_enqueue_scripts', [ $this, 'enqueue_scripts' ]);
			add_action('wp_enqueue_scripts', [ $this, 'cf7_enqueue_scripts' ]);
			add_filter('plugin_action_links_' . plugin_basename(SCF_FILE), [ $this, 'plugin_action_links' ]);
		}

		/**
		 * Activate plugin.
		 *
		 * @return void
		 */
		public function activate_scf() {
			update_option('scf-setup-run', false);

			$this->initialize_scf_table();
		}

		/**
		 * Initialize database tables.
		 *
		 * @return void
		 */
		public function initialize_scf_table() {
			Lead::create_table();
		}


		/**
		 * Redirect to setup page on activation.
		 *
		 * @return void
		 */
		public function handle_safe_redirection() {
			if ( ! wp_validate_boolean( get_option( 'scf-setup-run' ) ) ) {
				update_option( 'scf-setup-run', true );
				wp_safe_redirect( admin_url( 'admin.php?page=social-contact-form' ) );
				exit;
			}
		}
		/**
		 * Contact Form Leads table.
		 *
		 * @return void
		 */
		public function cf7_leads_table() {
			global $wpdb;
			$cf7_table_name      = $wpdb->prefix . 'cf7_leads';
			$charset_collate = $wpdb->get_charset_collate();

			$cf7_sql = "CREATE TABLE IF NOT EXISTS $cf7_table_name(
				id mediumint(11) NOT NULL AUTO_INCREMENT,
				cf7_id text NULL DEFAULT NULL,
				field text NULL DEFAULT NULL,
				meta text NULL DEFAULT NULL,
				created_at datetime DEFAULT '0000-00-00 00:00:00' NOT NULL,
				deleted_at datetime DEFAULT NULL,
				note text DEFAULT NULL,
				PRIMARY KEY  (id)
			) $charset_collate;";

			require_once ABSPATH . '/wp-admin/includes/upgrade.php';
			dbDelta($cf7_sql);
		}

		/**
		 * Register plugin settings.
		 *
		 * @return void
		 */
		public function register_options() {
			$options = \WPPOOL\SCF\App::options();
			foreach ( $options as $option_name => $option_default ) {
				$option_name = 'scf_' . $option_name;

				register_setting('scf_settings', $option_name);

				add_option($option_name, $option_default);
			}
		}

		/**
		 * Register admin menu.
		 *
		 * @return void
		 */
		public function register_admin_menu() {
			add_menu_page(
				__('Social Contact Form', 'social-contact-form'),
				__('Social Contact Form', 'social-contact-form'),
				'manage_options',
				'social-contact-form',
				[ $this, 'render_settings_page' ],
				'dashicons-social-contact-form'
			);

			// Submenu with same slug as parent.
			add_submenu_page(
				'social-contact-form',
				__('Social Contact Form Settings', 'social-contact-form'),
				__('Settings', 'social-contact-form'),
				'manage_options',
				'social-contact-form',
				[ $this, 'render_settings_page' ]
			);

			add_submenu_page(
				'social-contact-form',
				__('Social Contact Form Leads', 'social-contact-form'),
				__('Leads', 'social-contact-form'),
				'manage_options',
				'social-contact-form-leads',
				[ $this, 'render_leads_page' ]
			);

			do_action('scf_admin_menu');
		}

		/**
		 * Render settings page.
		 *
		 * @return void
		 */
		public function render_settings_page() {
			scf_render('admin/settings');
		}

		/**
		 * Render leads page.
		 *
		 * @return void
		 */
		public function render_leads_page() {
			scf_render('admin/leads');
		}

		/**
		 * Custom icon.
		 *
		 * @return void
		 */
		public function scf_custom_icon() {
			$image = SCF_PUBLIC . '/images/icon-white.svg';
			$css   = "#adminmenu div.wp-menu-image.dashicons-social-contact-form {
				background: url({$image}) no-repeat center center !important; 
				background-size: 40% !important;
				width: 37px !important;
				height: 37px !important;
			}";

			wp_enqueue_style( 'scf_inline_style', esc_url( SCF_PUBLIC ) . '/css/blank.css', [], microtime() );
			wp_add_inline_style('scf_inline_style', $css);
		}

		/**
		 * Enqueue admin scripts.
		 *
		 * @param string $hook Hook name.
		 * @return mixed
		 */
		public function enqueue_scripts( $hook = '' ) {

			$this->scf_custom_icon();
			wp_enqueue_style( 'social-contact-form-cf7', SCF_PUBLIC . '/css/admin.cf7.css', [], time());
			wp_enqueue_script('social-contact-form-cf7-backend-fields', SCF_PUBLIC . '/js/admin.cf7.js', [ 'jquery' ], time(), true);

			if ( ! in_array($hook, [ 'toplevel_page_social-contact-form', 'social-contact-form_page_social-contact-form-leads' ]) ) {
				return false;
			}

			wp_enqueue_script('jquery');
			wp_enqueue_media();
			wp_enqueue_style('social-contact-form-admin', SCF_PUBLIC . '/css/admin.min.css', [], time());
			wp_enqueue_style('social-contact-form-frontend', SCF_PUBLIC . '/css/frontend.min.css', [], time());
			wp_enqueue_style( 'social-contact-form-cf7-design-overwrite', SCF_PUBLIC . '/css/admin.scf-cf7.css', [], time());
			wp_enqueue_script('social-contact-form-admin', SCF_PUBLIC . '/js/admin.min.js', [ 'jquery' ], time(), true);

			wp_localize_script(
				'social-contact-form-admin',
				'scf_admin',
				[
					'ajax_url'    => admin_url('admin-ajax.php'),
					'public'      => SCF_PUBLIC . '/',
					'nonce'       => wp_create_nonce('scf_admin_nonce'),
					'is_pro'      => is_scf_ultimate(),
					'options'     => scf_get_options(),
					'filters'     => [
						'per_page' => apply_filters('scf_leads_per_page', 100),
					],
					'countries'   => App::countries(),
					'development' => scf_is_dev(),
					'fields'      => App::fields(),
				]
			);

			$premium_fonts = App::premium_fonts();

			$css = '';

			if ( $premium_fonts ) {
				foreach ( $premium_fonts as $font ) {

					$font_url    = SCF_PUBLIC . '/fonts/' . ( str_replace(' ', '+', $font) ) . '.ttf';
					$font_family = $this->format_font($font);

					$css .= "@font-face {
					font-family: {$font_family} !important; 
					src: url({$font_url}) format('ttf') !important;	
					font-style: normal !important;
					font-weight: 400 !important; 
				  }";
				}
			}

			wp_add_inline_style('social-contact-form-frontend', $css);

		}

		/**
		 * Enqueue contact form scripts.
		 *
		 * @param string $hook Hook name.
		 *
		 * @return void
		 */
		public function cf7_enqueue_scripts( $hook = '' ) {
			wp_enqueue_script('jquery');
			wp_enqueue_style('social-contact-form-cf7-frontend', SCF_PUBLIC . '/css/cf7.frontend.css', [], time());
			wp_enqueue_script('social-contact-form-admin', SCF_PUBLIC . '/js/social-cf7-front-script.js', [ 'jquery' ], time(), true);
		}

		/**
		 * Print admin notices.
		 *
		 * @return void
		 */
		public function scf_admin_notices() {
			global $pagenow;
			// Checks if post type is scf_messages using $pagenow and current screen.
			if ( 'edit.php' === $pagenow && 'scf_messages' === get_current_screen()->post_type ) {
				$scf = get_option('scf_options');
				if ( ! isset($scf['whatsapp_number']) || empty($scf['whatsapp_number']) && wp_validate_boolean( $scf['whatsapp'] ) ) {
					echo wp_sprintf(
						'<div class="notice notice-error is-dismissible">
							<p>%s</p>
						</div>',
						wp_sprintf('Please enter WhatsApp number in the %s to use Social Contact Form', '<a href="' . esc_url( admin_url('edit.php?post_type=scf_messages&page=social-contact-form') ) . '">Settings page</a>')
					);
				}
			}
		}

		/**
		 * Add plugin action links.
		 *
		 * @param array $links Plugin action links.
		 * @return array
		 */
		public function plugin_action_links( $links ) {
			if ( is_scf_ultimate() ) {
				$links[] = '<a href="' . admin_url('admin.php?page=social-contact-form') . '">' . __('Settings', 'social-contact-form') . '</a>';
			} else {
				$links[] = '<a href="https://go.wppool.dev/2rc7/?ref=' . esc_url(home_url()) . '" target="_blank" style="color: #b32d2e;">' . __('Upgrade', 'social-contact-form') . '</a>';
			}
			return $links;
		}

		/**
		 * Initialize Appsero SDK.
		 *
		 * @return void
		 */
		public function init_appsero() {
			if ( ! class_exists('\Appsero\Client') ) {
				require_once SCF_INC . '/appsero/src/Client.php';
			}

			add_filter('appsero_is_local', '__return_false');

			$this->appsero = new \Appsero\Client('9b39bac1-3b27-41d1-aeec-18fbfd4a9977', 'Social Contact Form', SCF_FILE);
			$this->appsero->set_textdomain('social-contact-form');

			// Active insights.
			$this->appsero->insights()->init();

			if ( function_exists( 'wppool_plugin_init' ) ) {
				wppool_plugin_init('social_contact_form', plugin_dir_url( SCF_FILE ) . '/includes/wppool/background-image.png' );
			}

		}

		/**
		 * Format font name.
		 *
		 * @param string $font Font name.
		 * @return string
		 */
		public function format_font( $font = '' ) {
			$fonts = explode(',', $font);
			$fonts = array_map(function ( $font ) {
				// if space found in font name.
				if ( strpos($font, ' ') !== false ) {
					$font = "'" . $font . "'";
				}
				return $font;
			}, $fonts);

			return implode(', ', $fonts);
		}

	}


	// Initialize the plugin.
	new Admin();
endif;
