<?php
/**
 * Ajax class.
 * Handles all ajax requests.
 *
 * @package SocialContactForm
 * @since 1.0.0
 */

// Namespace .
namespace WPPOOL\SCF;

use WPCF7_ContactForm;
use WP_Upgrader;
use Plugin_Upgrader;
use WP_Upgrader_Skin;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;


if ( ! class_exists( '\WPPOOL\SCF\Ajax' ) ) :
	/**
	 * Ajax class.
	 * Handles all ajax requests.
	 *
	 * @package SocialContactForm
	 * @since 1.0.0
	 */
	final class Ajax {

		/**
		 * Constructor.
		 *
		 * @since 1.0.0
		 */
		public function __construct() {
			add_action( 'wp_ajax_scf_save_settings', [ $this, 'scf_save_settings' ] );

			add_action( 'wp_ajax_scf_get_leads', [ $this, 'scf_get_leads' ] );
			add_action( 'wp_ajax_cf7_form_leads_select', [ $this, 'cf7_form_leads_select' ] );

			add_action( 'wp_ajax_scf_delete_leads', [ $this, 'scf_delete_leads' ] );
			add_action( 'wp_ajax_scf_delete_cf7_leads', [ $this, 'scf_delete_cf7_leads' ] );

			add_action( 'wp_ajax_scf_is_cf7_installed', [ $this, 'scf_is_cf7_installed' ] );
			add_action( 'wp_ajax_cf7_plugin_installed', [ $this, 'cf7_plugin_installed' ] );
			add_action( 'wp_ajax_scf_form_select', [ $this, 'scf_form_select' ] );
		}

		/**
		 * Sanitize form data.
		 *
		 * @param array $data Form data.
		 * @return array
		 */
		public function sanitize_form_data( $data ) {
			$data = array_map('sanitize_text_field', $data);
			$data = array_map('stripslashes', $data);
			return $data;
		}


		/**
		 * Sanitize inputs.
		 *
		 * @param mixed $array_or_string Array or string.
		 * @return mixed
		 */
		public function sanitize_inputs( $array_or_string ) {
			if ( is_string( $array_or_string ) ) {
				$allowed_protocols = wp_allowed_protocols();
				$allowed_protocols[] = 'data';
				$allowed_tags = [
					'form' => [
						'action' => [],
						'method' => [],
						'class' => [],
						'aria-label' => [],
						'novalidate' => [],
						'data-status' => [],
						'style' => [],
					],
					'div' => [
						'class' => [],
						'id' => [],
						'lang' => [],
						'dir' => [],
						'style' => [],
					],
					'span' => [
						'class' => [],
						'data-name' => [],
						'style' => [],
					],
					'input' => [
						'size' => [],
						'class' => [],
						'autocomplete' => [],
						'aria-required' => [],
						'aria-invalid' => [],
						'placeholder' => [],
						'value' => [],
						'type' => [],
						'name' => [],
						'style' => [],
						'min' => [],
						'max' => [],
					],
					'textarea' => [
						'cols' => [],
						'rows' => [],
						'class' => [],
						'aria-invalid' => [],
						'placeholder' => [],
						'name' => [],
						'style' => [],
					],
					'select' => [
						'name' => [],
						'class' => [],
						'id' => [],
						'style' => [],
					],
					'option' => [
						'value' => [],
						'selected' => [],
						'disabled' => [],
						'label' => [],
						'class' => [],
						'style' => [],
					],
					'label' => [
						'for' => [],
						'style' => [],
					],
					'p' => [
						'role' => [],
						'aria-live' => [],
						'aria-atomic' => [],
						'style' => [],
					],
					'ul' => [],
					'li' => [],
					'br' => [],
				];
				$array_or_string = wp_kses( $array_or_string, $allowed_tags, $allowed_protocols );
			} elseif ( is_array( $array_or_string ) ) {
				foreach ( $array_or_string as $key => &$value ) {
					if ( is_array( $value ) ) {
						$value = $this->sanitize_inputs( $value );
					} else {
						$allowed_protocols = wp_allowed_protocols();
						$allowed_protocols[] = 'data';
						$allowed_tags = [
							'form' => [
								'action' => [],
								'method' => [],
								'class' => [],
								'aria-label' => [],
								'novalidate' => [],
								'data-status' => [],
								'style' => [],
							],
							'div' => [
								'class' => [],
								'id' => [],
								'lang' => [],
								'dir' => [],
								'style' => [],
							],
							'span' => [
								'class' => [],
								'data-name' => [],
								'style' => [],
							],
							'input' => [
								'size' => [],
								'class' => [],
								'autocomplete' => [],
								'aria-required' => [],
								'aria-invalid' => [],
								'placeholder' => [],
								'value' => [],
								'type' => [],
								'name' => [],
								'style' => [],
								'min' => [],
								'max' => [],
							],
							'textarea' => [
								'cols' => [],
								'rows' => [],
								'class' => [],
								'aria-invalid' => [],
								'placeholder' => [],
								'name' => [],
								'style' => [],
							],
							'select' => [
								'name' => [],
								'class' => [],
								'id' => [],
								'style' => [],
							],
							'option' => [
								'value' => [],
								'selected' => [],
								'disabled' => [],
								'label' => [],
								'class' => [],
								'style' => [],
							],
							'label' => [
								'for' => [],
								'style' => [],
							],
							'p' => [
								'role' => [],
								'aria-live' => [],
								'aria-atomic' => [],
								'style' => [],
							],
							'ul' => [],
							'li' => [],
							'br' => [],
						];
						$value = wp_kses( $value, $allowed_tags, $allowed_protocols  );
					}
				}
			}
			return $array_or_string;
		}


		/**
		 * Save settings.
		 *
		 * @return void
		 */
		public function scf_save_settings() {

			// Checks nonce.
			check_ajax_referer( 'scf_admin_nonce', 'nonce' );

			// Checks if user has permission.
			if ( ! current_user_can( 'manage_options' ) ) {
				wp_send_json_error( 'You do not have permission to do this.' );
				wp_die();
			}

			$options = \WPPOOL\SCF\App::options();

			$request = $this->sanitize_inputs( $_POST );
			$updated = [];

			foreach ( $options as $option_name => $option_value ) {
				$name  = 'scf_' . $option_name;
				$value = isset( $request['options'][ $option_name ] ) ? $request['options'][ $option_name ] : $option_value;

				update_option( $name, $value );

				$updated[ $name ] = $value;
			}

			$response = [
				'success' => true,
				'data'    => $request['options'],
				'updated' => $updated,
			];

			wp_send_json( $response );
			wp_die();
		}

		/**
		 * Get leads.
		 *
		 * @return void
		 */
		public function scf_get_leads() {

			// Checks nonce.
			check_ajax_referer( 'scf_admin_nonce', 'nonce' );

			// Checks if user has permission.
			if ( ! current_user_can( 'manage_options' ) ) {
				wp_send_json_error( 'You do not have permission to do this.' );
				wp_die();
			}

			$leads = Lead::get_all();

			if ( $leads ) {
				$response = [
					'success' => true,
					'data'    => $leads,
				];
			} else {
				$response = [
					'success' => false,
					'data'    => [],
				];
			}

			wp_send_json( $response );
		}

		/**
		 * Delete leads SCF.
		 *
		 * @return void
		 */
		public function scf_delete_leads() {
			// Checks nonce.
			check_ajax_referer( 'scf_admin_nonce', 'nonce' );

			// Checks if user has permission.
			if ( ! current_user_can( 'manage_options' ) ) {
				wp_send_json_error( 'You do not have permission to do this.' );
				wp_die();
			}

			$ids = isset( $_REQUEST['ids'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['ids'] ) ) : '';
			$id_array = explode( ',', $ids );
			$deleted = Lead::delete_multiple( $id_array );

			if ( $deleted ) {
				$response = [
					'success' => true,
					'data'    => $deleted,
				];
			} else {
				$response = [
					'success' => false,
					'data'    => [],
				];
			}

			wp_send_json( $response );
		}
		/**
		 * Delete leads CF7.
		 *
		 * @return void
		 */
		public function scf_delete_cf7_leads() {
			// Checks nonce.
			check_ajax_referer( 'scf_admin_nonce', 'nonce' );

			// Checks if user has permission.
			if ( ! current_user_can( 'manage_options' ) ) {
				wp_send_json_error( 'You do not have permission to do this.' );
				wp_die();
			}

			$ids = isset( $_REQUEST['ids'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['ids'] ) ) : '';
			$id_array = explode( ',', $ids );
			$deleted = Lead::delete_multiple_cf7( $id_array );

			if ( $deleted ) {
				$response = [
					'success' => true,
					'data'    => $deleted,
				];
			} else {
				$response = [
					'success' => false,
					'data'    => [],
				];
			}

			wp_send_json( $response );
		}

		/**
		 * This is scf_form_select.
		 *
		 * @return void
		 */
		public function scf_form_select() {
			// Checks nonce.
			check_ajax_referer( 'scf_admin_nonce', 'nonce' );
			$id = isset( $_REQUEST['id'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['id'] ) ) : '';

			if ( ! $id ) {
				die();
			}

			echo do_shortcode("[contact-form-7 id='$id']");
			die();
		}

		/**
		 * This is cf7_form_leads_select.
		 *
		 * @return void
		 */
		public function cf7_form_leads_select() {
			// Checks nonce.
			check_ajax_referer( 'scf_admin_nonce', 'nonce' );
			$id = isset( $_REQUEST['id'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['id'] ) ) : '';

			if ( ! $id ) {
				die();
			}

			global $wpdb;
			$table_name = $wpdb->prefix . 'cf7_leads';

			$cf7_leads = $wpdb->get_results(
				$wpdb->prepare(
					// phpcs:ignore
					"SELECT * FROM $table_name WHERE cf7_id IN (%s) AND `deleted_at` IS NULL ORDER BY id DESC",
					$id
				), ARRAY_A
			);

			if ( $cf7_leads ) {
				foreach ( $cf7_leads as &$lead ) {
					$lead['field'] = json_decode( $lead['field'], true );
					$lead['meta'] = json_decode( $lead['meta'], true );
				}
				$response = [
					'success' => true,
					'data'    => $cf7_leads,
				];
			} else {
				$response = [
					'success' => false,
					'data'    => [],
				];
			}

			wp_send_json( $response );
		}

		/**
		 * Check If CF7 installed and get list of form.
		 *
		 * @return void
		 */
		public function scf_is_cf7_installed() {

			$cf7_is_installed = false;

			if ( is_plugin_active('contact-form-7/wp-contact-form-7.php') ) {
				// Get all the forms created in CF7.
				$forms = WPCF7_ContactForm::find([
					'orderby' => 'ID',
					'order' => 'ASC',
				]);

				// Format the forms data for the REST API response.
				$formatted_forms = [];
				foreach ( $forms as $form ) {
					$formatted_forms[] = [
						'id' => $form->id(),
						'title' => $form->title(),
					];
				}
				wp_send_json($formatted_forms);
			} elseif ( file_exists(WP_PLUGIN_DIR . '/contact-form-7/wp-contact-form-7.php') ) {
				wp_send_json('inactive'); // 'Installed but not activated'
			} else {
				wp_send_json($cf7_is_installed); // 'Not installed'
			}

		}

		/**
		 * CF7 Download, install and activate.
		 *
		 * @return void
		 */
		public function cf7_plugin_installed() {
			// Check if Contact Form 7 plugin is already installed.
			$cf7_is_installed = is_plugin_active('contact-form-7/wp-contact-form-7.php');

			if ( true == $cf7_is_installed ) {
				wp_send_json('Contact Form 7 is already installed.');
			}

			if ( ! current_user_can('manage_options') ) {
				wp_send_json(false, __('You do not have permission to do this', 'social-contact-form'));
			} else {

				require_once( ABSPATH . 'wp-admin/includes/plugin-install.php' );
				require_once( ABSPATH . 'wp-admin/includes/class-wp-upgrader.php' );
				require_once ( ABSPATH . 'wp-admin/includes/class-plugin-upgrader.php' );

				$api = plugins_api( 'plugin_information', [
					'slug' => 'contact-form-7',
				] );

				$upgrader = new Plugin_Upgrader( new WP_Upgrader_Skin() );
				$install = $upgrader->install( $api->download_link );

				if ( is_wp_error( $install ) ) {
					wp_send_json( 'Failed to install Contact Form 7 plugin' );
				}

				activate_plugin( 'contact-form-7/wp-contact-form-7.php' );

				wp_send_json('Contact Form 7 plugin has been installed and activated successfully!');
			}

		}
	}

	// Initialize Ajax class. Only if doing ajax.
	if ( wp_doing_ajax() ) {
		new Ajax();
	}
endif;