<?php
if ( ! class_exists( 'TEMPLINES_Widget_Subtitles' ) ) {

/**
 * Plugin initializer class.
 *
 * @author  Jory Hogeveen <info@keraweb.nl>
 * @package Widget_Subtitles
 * @since   0.1.0
 * @version 1.2.1
 */
final class TEMPLINES_Widget_Subtitles
{
	/**
	 * The single instance of the class.
	 *
	 * @since  0.1.0
	 * @var    \TEMPLINES_Widget_Subtitles
	 */
	private static $_instance = null;

	/**
	 * The plugin basename.
	 *
	 * @since  1.1.4
	 * @var    string
	 */
	public static $_basename = '';

	/**
	 * The plugin i18n domain.
	 *
	 * @since  1.1.4.1
	 * @var    string
	 */
	public static $_domain = 'widget-subtitles';

	/**
	 * Possible locations of the subtitle.
	 *
	 * @since  0.1.0
	 * @var    array
	 */
	private $locations = array();

	/**
	 * The default subtitle location.
	 *
	 * @since  1.1.2
	 * @var    string
	 */
	private $default_location = 'after-inside';

	/**
	 * The capability required to modify subtitle locations.
	 *
	 * @since  1.1.2
	 * @var    string
	 */
	private $location_cap = 'edit_theme_options';

	/**
	 * Constructor.
	 *
	 * @since   0.1.0
	 * @access  private
	 */
	private function __construct() {
		self::$_instance = $this;
		self::$_basename = plugin_basename( __FILE__ );

		add_action( 'init', array( $this, 'init' ) );
	}

	/**
	 * Main Genesis Widget Subtitles.
	 *
	 * Ensures only one instance of Widget Subtitle is loaded or can be loaded.
	 *
	 * @since   0.1.0
	 * @access  public
	 * @static
	 * @see     TEMPLINES_widget_subtitles()
	 * @return  TEMPLINES_Widget_Subtitles
	 */
	public static function get_instance() {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;
	}

	/**
	 * Init function/action and register all used hooks and data.
	 *
	 * @since   0.1.0
	 * @access  public
	 * @return  void
	 */
	public function init() {

		/**
		 * Change the capability required to modify subtitle locations.
		 * Default: `edit_theme_options`.
		 *
		 * @since  1.1.2
		 * @param  string  $location_cap  The capability.
		 * @return string
		 */
		$this->location_cap = apply_filters( 'widget_subtitles_edit_location_capability', $this->location_cap );

		$loc['before']  = __( 'Before title', 'templines-helper-core' );
		$loc['after']   = __( 'After title', 'templines-helper-core' );
		$loc['outside'] = __( 'Outside heading', 'templines-helper-core' );
		$loc['inside']  = __( 'Inside heading', 'templines-helper-core' );

		/**
		 * Sets the default location for subtitles.
		 * This will be overwritten once a widget is saved.
		 *
		 * @since   1.1.2
		 *
		 * @param   string  $subtitle_location  The subtitle location (default: 'after-inside').
		 * @return  string  Options: 'after-inside', 'after-outside', 'before-inside', 'before-outside'.
		 */
		$this->default_location = apply_filters( 'widget_subtitles_default_location', $this->default_location );

		$default = explode( '-', $this->default_location );
		foreach ( $default as $key => $value ) {
			if ( isset( $loc[ $value ] ) && 2 === count( $default ) ) {
				$default[ $key ] = $loc[ $value ];
			}
		}
		$default = implode( ' - ', $default );

		$this->locations = array(
			''               => __( 'Default', 'templines-helper-core' ) . ' (' . $default . ')',
			// before title, outside title element.
			'before-outside' => $loc['before'] . ' - ' . $loc['outside'],
			// before title, inside title element
			'before-inside'  => $loc['before'] . ' - ' . $loc['inside'],
			// after title, outside title element
			'after-outside'  => $loc['after'] . ' - ' . $loc['outside'],
			// after title, inside title element
			'after-inside'   => $loc['after'] . ' - ' . $loc['inside'],
		);

		add_action( 'in_widget_form', array( $this, 'action_in_widget_form' ), 9, 3 );
		add_filter( 'widget_update_callback', array( $this, 'filter_widget_update_callback' ), 10, 4 );
		add_filter( 'dynamic_sidebar_params', array( $this, 'filter_dynamic_sidebar_params' ) );
	}

	/**
	 * Add a subtitle input field into the form.
	 *
	 * @since   0.1.0
	 * @since   1.2.0  Add `action_` prefix.
	 * @access  public
	 *
	 * @param   \WP_Widget  $widget
	 * @param   null        $return
	 * @param   array       $instance
	 * @return  null
	 */
	public function action_in_widget_form( $widget, $return, $instance ) {

		$instance = wp_parse_args(
			(array) $instance,
			array(
				'subtitle' => '',
				'subtitle_location' => '',
			)
		);

		$can_edit_location = current_user_can( $this->location_cap );
		?>

		<p>
			<label for="<?php echo $widget->get_field_id( 'subtitle' ); ?>"><?php esc_html_e( 'Subtitle', 'templines-helper-core' ); ?>:</label>
			<input class="widefat" id="<?php echo $widget->get_field_id( 'subtitle' ); ?>" name="<?php echo $widget->get_field_name( 'subtitle' ); ?>" type="text" value="<?php echo esc_attr( strip_tags( $instance['subtitle'] ) ); ?>"/>
		</p>

		<?php if ( $can_edit_location ) { ?>
		<p>
			<label for="<?php echo $widget->get_field_id( 'subtitle_location' ); ?>"><?php esc_html_e( 'Subtitle location', 'templines-helper-core' ); ?>:</label>
			<select name="<?php echo $widget->get_field_name( 'subtitle_location' ); ?>" id="<?php echo $widget->get_field_id( 'subtitle_location' ); ?>">
			<?php
			$locations = $this->get_available_subtitle_locations( $widget, $instance );
			foreach ( (array) $locations as $location_key => $location_name ) {
				?>
				<option value="<?php echo $location_key; ?>" <?php selected( $instance['subtitle_location'], $location_key, true ); ?>>
					<?php echo $location_name; ?>
				</option>
				<?php
			}
			?>
			</select>
		</p>
		<?php } else { ?>
			<input type="hidden" name="<?php echo $widget->get_field_name( 'subtitle_location' ); ?>" value="<?php echo $instance['subtitle_location']; ?>"/>
		<?php } ?>

		<script type="text/javascript">
			;( function( $ ) {
				var title     = '#<?php echo $widget->get_field_id( 'title' ); ?>',
					subtitle  = '#<?php echo $widget->get_field_id( 'subtitle' ); ?>',
					$title    = $( title ),
					$subtitle = $( subtitle );
				<?php if ( $can_edit_location ) { ?>
				var subtitle_location  = '#<?php echo $widget->get_field_id( 'subtitle_location' ); ?>',
					$subtitle_location = $( subtitle_location );

				// show/hide subtitle location input.
				if ( ! $subtitle.val() ) {
					$subtitle_location.parent().hide();
				}
				$subtitle.on( 'keyup', function() {
					if ( $(this).val() ) {
						$subtitle_location.parent().slideDown('fast');
					} else {
						$subtitle_location.parent().slideUp('fast');
					}
				} );
				<?php } ?>
				// Relocate subtitle input after title if available.
				if ( $title.length && $title.parent('p').length ) {
					$subtitle.parent('p').detach().insertAfter( $title.parent('p') );
					<?php if ( $can_edit_location ) { ?>
					$subtitle_location.parent('p').detach().insertAfter( $subtitle.parent('p') );
					<?php } ?>
				}
			} ) ( jQuery );
		</script>

		<?php
		return $return;
	}

	/**
	 * Filter the widget’s settings before saving, return false to cancel saving (keep the old settings if updating).
	 *
	 * @since   0.1.0
	 * @since   1.2.0  Add `filter_` prefix.
	 * @since   1.2.1  sanitize_text_field() to ensure same validation as default widget title.
	 * @access  public
	 *
	 * @param   array       $instance
	 * @param   array       $new_instance
	 * param   array       $old_instance
	 * param   \WP_Widget  $widget
	 * @return  array
	 */
	public function filter_widget_update_callback( $instance, $new_instance ) {
		unset( $instance['subtitle'] );
		unset( $instance['subtitle_location'] );

		if ( ! empty( $new_instance['subtitle'] ) ) {
			$instance['subtitle'] = sanitize_text_field( $new_instance['subtitle'] );

			if ( ! empty( $new_instance['subtitle_location'] ) && is_string( $new_instance['subtitle_location'] ) ) {
				//&& array_key_exists( $new_instance['subtitle_location'], $this->locations )
				$instance['subtitle_location'] = esc_attr( wp_strip_all_tags( $new_instance['subtitle_location'] ) );
			}
		}

		return $instance;
	}

	/**
	 * Gets called from within the dynamic_sidebar function which displays a widget container.
	 * This filter gets called for each widget instance in the sidebar.
	 *
	 * // Disable variable check because of global $wp_registered_widgets.
	 * @SuppressWarnings(PHPMD.LongVariables)
	 *
	 * @since   0.1.0
	 * @since   1.2.0  Add `filter_` prefix.
	 * @access  public
	 *
	 * @global  $wp_registered_widgets
	 * @param   array  $params
	 * @return  array
	 */
	public function filter_dynamic_sidebar_params( $params ) {
		global $wp_registered_widgets;

		if ( ! isset( $params[0]['widget_id'] ) ) {
			return $params;
		}
		$widget_id = $params[0]['widget_id'];

		if ( empty( $wp_registered_widgets[ $widget_id ] ) ) {
			return $params;
		}
		$widget = $wp_registered_widgets[ $widget_id ];

		// Get instance settings.
		if ( empty( $widget['callback'][0]->option_name ) ) {
			return $params;
		}
		/** @var \WP_Widget $widget_obj */
		$widget_obj = $widget['callback'][0];
		$instance = get_option( $widget['callback'][0]->option_name );

		// Check if there's an instance of the widget.
		if ( ! array_key_exists( $params[1]['number'], $instance ) ) {
			return $params;
		}
		$instance = $instance[ $params[1]['number'] ];

		// Add the subtitle.
		if ( ! empty( $instance['subtitle'] ) ) {

			$sidebar_id = $this->get_widget_sidebar_id( $widget_id );

			// Default.
			$subtitle_location = $this->default_location;
			// Available.
			$locations = $this->get_available_subtitle_locations( $widget_obj, $instance, $sidebar_id );

			// Get location value if it exists and is valid.
			if ( ! empty( $instance['subtitle_location'] ) && array_key_exists( $instance['subtitle_location'], $locations ) ) {
				$subtitle_location = $instance['subtitle_location'];
			}

			/**
			 * Filters subtitle element (default: span).
			 *
			 * @since  1.0.0
			 * @since  1.1.0  Add extra parameters.
			 * @since  1.1.3  Add WP_Widget instance parameter.
			 *
			 * @param  string      'span'       The HTML element.
			 * @param  string      $widget_id   The widget ID (widget name + instance number).
			 * @param  string      $sidebar_id  The sidebar ID where this widget is located.
			 * @param  array       $widget      All widget data.
			 * @param  \WP_Widget  $widget_obj  The Widget object.
			 * @return string  A valid HTML element.
			 */
			$subtitle_element = apply_filters( 'widget_subtitles_element', 'div', $widget_id, $sidebar_id, $widget, $widget_obj );

			$subtitle_classes = $this->get_subtitle_classes( $subtitle_location );
			/**
			 * Allow filter for subtitle classes to overwrite, remove or add classes.
			 *
			 * @since  1.0.0
			 * @since  1.1.0  Add extra parameters.
			 * @since  1.1.3  Add WP_Widget instance parameter.
			 *
			 * @param  array       $subtitle_classes  The default classes.
			 * @param  string      $widget_id         The widget ID (widget name + instance number).
			 * @param  string      $sidebar_id        The sidebar ID where this widget is located.
			 * @param  array       $widget            All widget data.
			 * @param  \WP_Widget  $widget_obj        The Widget object.
			 * @return array  An array of CSS classes.
			 */
			$subtitle_classes = apply_filters( 'widget_subtitles_classes', $subtitle_classes, $widget_id, $sidebar_id, $widget, $widget_obj );

			// Create class string to use.
			$subtitle_classes = is_array( $subtitle_classes ) ? '' . implode( ' ', $subtitle_classes ) . '' : '';

			// Start the output.
			$subtitle = '<' . $subtitle_element . ' class="' . $subtitle_classes . '">';
			/**
			 * Filters the widget subtitle.
			 *
			 * @since  1.2.0
			 *
			 * @param  string      $subtitle    The widget subtitle.
			 * @param  array       $instance    Array of settings for the current widget.
			 * @param  string      $widget_id   The widget ID (widget name + instance number).
			 * @param  string      $sidebar_id  The sidebar ID where this widget is located.
			 * @param  array       $widget      All widget data.
			 * @param  \WP_Widget  $widget_obj  The Widget object.
			 * @return string  The new subtitle.
			 */
			$subtitle .= apply_filters( 'widget_subtitle', $instance['subtitle'], $instance, $widget_id, $sidebar_id, $widget, $widget_obj );
			$subtitle .= '</' . $subtitle_element . '>';

			// Add the subtitle.
			$params[0] = $this->add_subtitle( $params[0], $subtitle, $subtitle_location, $widget, $sidebar_id );

		} // End if().

		return $params;
	}

	/**
	 * Add a subtitle in the widget parameters.
	 *
	 * @since   1.1.2
	 * @since   1.2.0  Added optional `$widget_obj` and `$sidebar_id` parameters.
	 * @access  public
	 * @param   array       $params             Widget parameters.
	 * @param   string      $subtitle           The subtitle, may contain HTML.
	 * @param   string      $subtitle_location  The subtitle location.
	 * @param   \WP_Widget  $widget_obj         The Widget object.
	 * @param   string      $sidebar_id         The sidebar ID where this widget is located.
	 * @return  array
	 */
	public function add_subtitle( $params, $subtitle, $subtitle_location = '', $widget_obj = null, $sidebar_id = '' ) {

		if ( empty( $subtitle_location ) ) {
			$subtitle_location = $this->default_location;
		}

		if ( empty( $params['before_title'] ) ) {
			$params['before_title'] = '';
		}
		if ( empty( $params['after_title'] ) ) {
			$params['after_title'] = '';
		}

		// Assign the output to the correct location in the correct order.
		switch ( $subtitle_location ) {

			case 'before-inside':
				// A space to separate subtitle from title.
				$params['before_title'] = $params['before_title'] . $subtitle . ' ';
				break;

			case 'before-outside':
				$params['before_title'] = $subtitle . $params['before_title'];
				break;

			case 'after-inside':
				// A space to separate subtitle from title.
				$params['after_title'] = ' ' . $subtitle . $params['after_title'];
				break;

			case 'after-outside':
				$params['after_title'] = $params['after_title'] . $subtitle;
				break;

			default:
				/**
				 * Add subtitle at a custom location
				 *
				 * @since  1.2.0
				 *
				 * @param  array       $params             Widget sidebar parameters from `dynamic_sidebar_params` filter.
				 * @param  string      $subtitle           The subtitle.
				 * @param  string      $subtitle_location  The selected subtitle location.
				 * @param  \WP_Widget  $widget_obj         The widget object.
				 * @param  string      $sidebar_id         The sidebar ID where this widget is located.
				 * @return array
				 */
				$params = apply_filters( 'widget_subtitles_add_subtitle', $params, $subtitle, $subtitle_location, $widget_obj, $sidebar_id );
				break;
		}

		return $params;
	}

	/**
	 * Get the sidebar ID related to a widget ID.
	 *
	 * @since   1.1.1
	 * @since   1.2.0   Make use of `wp_get_sidebars_widgets()`.
	 * @access  public
	 * @param   string  $widget_id  The widget identifier.
	 * @return  string
	 */
	public function get_widget_sidebar_id( $widget_id ) {
		//global $_wp_sidebars_widgets;
		$sidebars_widgets = wp_get_sidebars_widgets();
		$sidebar_id       = '';
		if ( is_array( $sidebars_widgets ) ) {
			foreach ( $sidebars_widgets as $key => $widgets ) {
				if ( is_array( $widgets ) && in_array( (string) $widget_id, array_map( 'strval', $widgets ), true ) ) {
					// Found!
					$sidebar_id = $key;
					break;
				}
			}
		}
		return $sidebar_id;
	}

	/**
	 * Get the subtitle classes.
	 *
	 * @since   1.1.1
	 * @access  public
	 * @param   string  $location  The subtitle location.
	 * @return  array
	 */
	public function get_subtitle_classes( $location ) {
		// Create subtitle classes.
		$subtitle_classes = array( 'widget--subtitle');
		// Add subtitle location classes.
		$subtitle_classes[] = 'subtitle-' . $location;
        // Templines Custom html CLass.
        $subtitle_classes[] = 'fl-font-style-lighter-than';
		// Convert to array.
		$location_classes = explode( '-', $location );
		// Prevent duplicated classes.
		if ( 1 < count( $location_classes ) ) {
			// Add location part classes.
			foreach ( $location_classes as $location_class ) {
				$subtitle_classes[] = 'subtitle-' . $location_class;
			}
		}
		return $subtitle_classes;
	}

	/**
	 * Get the available locations for a widget.
	 *
	 * @since   1.1.3
	 * @since   1.2.0  Optional `$sidebar_id` parameter.
	 * @param   \WP_Widget  $widget_obj
	 * @param   array       $instance
	 * @param   string      $sidebar_id  The sidebar ID where this widget is located.
	 * @return  array
	 */
	public function get_available_subtitle_locations( $widget_obj, $instance, $sidebar_id = '' ) {

		$widget_id = $widget_obj->id;
		if ( ! $sidebar_id ) {
			$sidebar_id = $this->get_widget_sidebar_id( $widget_id );
		}

		/**
		 * Filter the available locations.
		 * Note: Location keys must be strings!
		 *
		 * @since   1.1.3
		 * @since   1.2.0  Allow custom location + add `$widget_id` and `$sidebar_id` parameters.
		 *
		 * @param   string[]    $locations   The array of available locations.
		 * @param   \WP_Widget  $widget_obj  The widget object.
		 * @param   array       $instance    The widget instance.
		 * @param   string      $widget_id   The widget ID (widget name + instance number).
		 * @param   string      $sidebar_id  The sidebar ID where this widget is located.
		 * @return  string[]  $locations  The available locations: key => label.
		 */
		return (array) apply_filters( 'widget_subtitles_available_locations', $this->locations, $widget_obj, $instance, $widget_id, $sidebar_id );
	}

	/**
	 * Magic method to output a string if trying to use the object as a string.
	 *
	 * @since   0.1.0
	 * @access  public
	 * @return  string
	 */
	public function __toString() {
		return get_class( $this );
	}

	/**
	 * Magic method to keep the object from being cloned.
	 *
	 * @since   0.1.0
	 * @access  public
	 * @return  void
	 */
	public function __clone() {
		_doing_it_wrong(
			__FUNCTION__,
			esc_html( get_class( $this ) . ': ' . __( 'This class does not want to be cloned', 'templines-helper-core' ) ),
			null
		);
	}

	/**
	 * Magic method to keep the object from being unserialized.
	 *
	 * @since   0.1.0
	 * @access  public
	 * @return  void
	 */
	public function __wakeup() {
		_doing_it_wrong(
			__FUNCTION__,
			esc_html( get_class( $this ) . ': ' . __( 'This class does not want to wake up', 'templines-helper-core' ) ),
			null
		);
	}

	/**
	 * Magic method to prevent a fatal error when calling a method that doesn't exist.
	 *
	 * @since   0.1.0
	 * @access  public
	 * @param   string  $method
	 * @param   array   $args
	 * @return  null
	 */
	public function __call( $method = '', $args = array() ) {
		_doing_it_wrong(
			esc_html( get_class( $this ) . "::{$method}" ),
			esc_html__( 'Method does not exist.', 'templines-helper-core' ),
			null
		);
		unset( $method, $args );
		return null;
	}

}

/**
 * Main instance of Widget Subtitle.
 *
 * Returns the main instance of Widget_Subtitles to prevent the need to use globals.
 *
 * @since   0.1.0
 * @return  \TEMPLINES_Widget_Subtitles
 */
function templines_widget_subtitles() {
	return TEMPLINES_Widget_Subtitles::get_instance();
}
templines_widget_subtitles();

} // End if().
