<?php

use Elementor\Control_Media;
use Elementor\Group_Control_Image_Size;
use Elementor\Utils;
use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Repeater;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor hotspot Carousel widget.
 *
 * Elementor widget that displays a bullet list with any chosen icons and texts.
 *
 * @since 1.0.0
 */
class Templines_Elementor_Hotspot_Carousel extends Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve hotspot Carousel widget name.
	 *
	 * @return string Widget name.
	 * @since  1.0.0
	 * @access public
	 *
	 */
	public function get_name() {
		return 'templines-hotspot-carousel';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve hotspot Carousel widget title.
	 *
	 * @return string Widget title.
	 * @since  1.0.0
	 * @access public
	 *
	 */
	public function get_title() {
		return __( 'Hotspot Carousel', 'templines-helper-core' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve hotspot Carousel widget icon.
	 *
	 * @return string Widget icon.
	 * @since  1.0.0
	 * @access public
	 *
	 */
	public function get_icon() {
		return 'ip-image-box';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 */
    public function get_categories() {
        return array('templines-helper-core-elements');
    }

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @return array Widget keywords.
	 * @since  2.1.0
	 * @access public
	 *
	 */
	public function get_keywords() {
		return [ 'hotspot', 'hotspot Carousel', 'image', 'list' ];
	}

	/**
	 * Register hotspot Carousel widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since  1.0.0
	 * @access protected
	 */
	protected function register_controls() {
		$this->start_controls_section(
			'section_image',
			[
				'label' => __( 'Hotspot Carousel', 'templines-helper-core' ),
			]
		);

		$this->add_control(
			'arrows_color',
			[
				'label'   => __( 'Arrows color', 'templines-helper-core' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'dark',
				'options' => [
					'dark'  => __( 'Dark', 'templines-helper-core' ),
					'light' => __( 'Light', 'templines-helper-core' ),
				]
			]
		);

		$repeater = new Repeater();

		$repeater->add_control(
			'image',
			[
				'label'   => __( 'Choose Image', 'templines-helper-core' ),
				'type'    => Controls_Manager::MEDIA,
				'default' => [
					'url' => Utils::get_placeholder_image_src(),
				],
			]
		);

		$repeater->add_control(
			'hotspots',
			[
				'label' => __( 'Add / Edit Hotspots', 'templines-helper-core' ),
				'image' => 'image',
				'type'  => 'templines-hotspot'
			]
		);


		$this->add_control(
			'image_list',
			[
				'label'       => '',
				'type'        => Controls_Manager::REPEATER,
				'fields'      => $repeater->get_controls(),
			]
		);


		$this->end_controls_section();
	}

	/**
	 * Render hotspot Carousel widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since  1.0.0
	 * @access protected
	 */
    protected function render() {
        $settings = $this->get_settings_for_display();
        ?>
        <div class="c-hotspot-carousel">
            <div class="c-hotspot-carousel__list js-hotspot-carousel h-carousel h-carousel--flex h-carousel--dark h-carousel--dots-hide h-carousel--<?php echo esc_attr( $settings['arrows_color'] ); ?>">
                <?php
                    foreach ( $settings['image_list'] as $index => $item ) : ?>
                    <?php $hotspots = ! empty( $item['hotspots'] ) ? json_decode( $item['hotspots'], true ) : []; ?>
                    <div class="c-hotspot-carousel__item">

                        <?php if ( ! empty( $item['image']['id'] ) ) { ?>
                            <div class="c-hotspot-carousel__image-wrap">
                                <?php
                                if ( ! empty( $item['image']['id'] ) && ( $type = get_post_mime_type( $item['image']['id'] ) ) ) {
                                    if ( $type == 'image/svg+xml' ) {
                                        echo templines_get_inline_svg( $item['image']['id'], 'c-hotspot-carousel__image c-hotspot-carousel__image--svg' );
                                    } else {
                                        if ( $image_meta = templines_image_meta( $item['image']['id'], 'large' ) ) {
                                            $image_key = 'image-' . $index;
                                            $this->add_render_attribute( $image_key, 'class', 'c-hotspot-carousel__image' );
                                            if ( $image_meta['srcset'] && $image_meta['sizes'] ) {
                                                $this->add_render_attribute( $image_key, 'srcset', $image_meta['srcset'] );
                                                $this->add_render_attribute( $image_key, 'sizes', $image_meta['sizes'] );
                                            }
                                            $this->add_render_attribute( $image_key, 'src', $image_meta['src'] );
                                            $this->add_render_attribute( $image_key, 'alt', $image_meta['alt'] );

                                            $this->add_render_attribute( $image_key, 'loading', 'lazy' );


                                            echo '<img ' . $this->get_render_attribute_string( $image_key ) . '>';
                                        }
                                    }
                                } ?>
                                <?php if ( is_array( $hotspots ) ) { ?>
                                    <?php foreach ( $hotspots as $point ) { ?>
                                        <div class="c-hotspot-carousel__point js-carousel-point"
                                             data-left="<?php echo esc_attr( $point['x'] ); ?>"
                                             data-top="<?php echo esc_attr( $point['y'] ); ?>" data-rounded-children="enabled">
                                            <div class="syn-hotspot-item syn-hotspot-itemfront syn-elem-isbxsh">
                                                <div class="syn-hotspot-marker syn-elem-idrounded">
                                                    <div class="syn-hotspot-markerinsider elementor-hotspot-marker">+</div>
                                                </div>
                                            </div>
                                            <div class="c-hotspot-carousel__point-popup"><?php echo esc_html( $point['title'] ); ?></div>
                                        </div>
                                    <?php } ?>
                                <?php } ?>

                            </div>
                        <?php } ?>

                    </div>
                <?php
                endforeach;
                ?>
            </div>
        </div>
        <?php
    }

/**
 * Render hotspot Carousel widget output in the editor.
 *
 * Written as a Backbone JavaScript template and used to generate the live preview.
 *
 * @since  1.0.0
 * @access protected
 */
	protected function _content_template() {
	}
}
