<?php


namespace Elementor;


if (!defined('ABSPATH'))
    exit; // Exit if accessed directly


class BDDEX_Gallery_Widget extends Widget_Base {

    public function get_name() {
        return 'lae-gallery';
    }

    public function get_title() {
        return __('Avas Image / Video', 'avas');
    }

    public function get_icon() {
        return 'eicon-gallery-grid';
    }

    public function get_categories() {
        return array('bddex');
    }

    public function get_script_depends() {
        return [
            'lae-widgets-scripts',
            'lae-frontend-scripts',
            'jquery-magnific-popup',
            'isotope.pkgd',
            'imagesloaded.pkgd'
        ];
    }

    protected function _register_controls() {

        $this->start_controls_section(
            'section_gallery',
            [
                'label' => __('Gallery', 'avas'),
            ]
        );


        $this->add_control(
            'heading',
            [
                'label' => __('Heading for the grid', 'avas'),
                'type' => Controls_Manager::TEXT,
                'placeholder' => __('Title Here', 'avas'),
                'default' => __('Title Here', 'avas'),
            ]
        );


        $this->add_control(
            'gallery_items_heading',
            [
                'label' => __('Gallery Items', 'avas'),
                'type' => Controls_Manager::HEADING,
            ]
        );


        $this->add_control(
            'gallery_items',
            [
                'type' => Controls_Manager::REPEATER,
                'fields' => [

                    [
                        "type" => Controls_Manager::SELECT,
                        "name" => "item_type",
                        "label" => __("Item Type", "avas"),
                        'options' => array(
                            'image' => __('Image', 'avas'),
                            'youtube' => __('YouTube Video', 'avas'),
                            'vimeo' => __('Vimeo Video', 'avas'),
                        ),
                        'default' => 'image',
                    ],

                    [
                        'name' => 'item_name',
                        'label' => __('Item Label', 'avas'),
                        'type' => Controls_Manager::TEXT,
                        'description' => __('The label or name for the gallery item.', 'avas'),
                    ],

                    [
                        'name' => 'item_image',
                        'label' => __('Gallery Image', 'avas'),
                        'description' => __('The image for the gallery item. If item type chosen is YouTube or Vimeo video, the image will be used as a placeholder image for video.', 'avas'),
                        'type' => Controls_Manager::MEDIA,
                        'default' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'label_block' => true,
                    ],

                    [
                        'name' => 'item_tags',
                        'label' => __('Item Tag(s)', 'avas'),
                        'type' => Controls_Manager::TEXT,
                        'description' => __('One or more comma separated tags for the gallery item. Will be used as filters for the items.', 'avas'),
                    ],

                    [
                        'name' => 'item_link',
                        'label' => __('Item Link', 'avas'),
                        'description' => __('The URL of the page to which the image gallery item points to (optional).', 'avas'),
                        'type' => Controls_Manager::URL,
                        'label_block' => true,
                        'default' => [
                            'url' => '',
                            'is_external' => 'false',
                        ],
                        'placeholder' => __('http://your-link.com', 'avas'),
                        'condition' => [
                            'item_type' => ['image'],
                        ],
                    ],

                    [
                        'name' => 'video_link',
                        'label' => __('Video URL', 'avas'),
                        'description' => __('The URL of the YouTube or Vimeo video.', 'avas'),
                        'type' => Controls_Manager::TEXT,
                        'condition' => [
                            'item_type' => ['youtube', 'vimeo'],
                        ],
                    ],

                ],
               // 'title_field' => '{{{ item_name }}}',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_settings',
            [
                'label' => __('General', 'avas'),
                'tab' => Controls_Manager::TAB_SETTINGS,
            ]
        );

        $this->add_control(
            'filterable',
            [
                'label' => __('Filterable?', 'avas'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => __('Yes', 'avas'),
                'label_off' => __('No', 'avas'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );


        $this->add_control(
            'per_line',
            [
                'label' => __('Columns per row', 'avas'),
                'type' => Controls_Manager::NUMBER,
                'min' => 1,
                'max' => 5,
                'step' => 1,
                'default' => 4,
            ]
        );

        $this->add_control(
            'layout_mode',
            [
                'type' => Controls_Manager::SELECT,
                'label' => __('Choose a layout for the grid', 'avas'),
                'options' => array(
                    'fitRows' => __('Fit Rows', 'avas'),
                    'masonry' => __('Masonry', 'avas'),
                ),
                'default' => 'fitRows',
            ]
        );

        $this->add_control(
            'enable_lightbox',
            [
                'type' => Controls_Manager::SWITCHER,
                'label_off' => __('No', 'avas'),
                'label_on' => __('Yes', 'avas'),
                'return_value' => 'yes',
                'default' => 'yes',
                'label' => __('Enable Lightbox Gallery?', 'avas'),
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_pagination',
            [
             //  'label' => __('Pagination', 'avas'),
              // 'tab' => Controls_Manager::TAB_SETTINGS,
            ]
        );


        $this->add_control(
            'pagination',
            [
               'type' => Controls_Manager::SELECT,
              //  'label' => __('Pagination', 'avas'),
                // 'description' => __('Choose pagination type or choose None if no pagination is desired. Make sure you enter the items per page value in the option \'Number of items to be displayed per page and on each load more invocation\' field below to control number of items to display per page.', 'avas'),
                'options' => array(
                //    'none' => __('None', 'avas'),
                  //  'paged' => __('Paged', 'avas'),
                  //  'load_more' => __('Load More', 'avas'),
                ),
                'default' => 'none',
            ]
        );


        // $this->add_control(
        //     'show_remaining',
        //     [
        //         // 'label' => __('Display count of items yet to be loaded with the load more button?', 'avas'),
        //         // 'type' => Controls_Manager::SWITCHER,
        //         // 'label_on' => __('Yes', 'avas'),
        //         // 'label_off' => __('No', 'avas'),
        //         // 'return_value' => 'yes',
        //         // 'default' => 'yes',
        //         // 'condition' => [
        //         //     'pagination' => 'load_more',
        //         // ],
        //     ]
        // );


        $this->add_control(
            'items_per_page',
            [
                // 'label' => __('Number of items to be displayed per page and on each load more invocation.', 'avas'),
                // 'type' => Controls_Manager::NUMBER,
                'default' => -1,
                'condition' => [
                    'pagination' => ['load_more', 'paged'],
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_heading_styling',
            [
                'label' => __('Gallery Heading', 'avas'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );


        $this->add_control(
            'heading_tag',
            [
                'label' => __( 'Heading HTML Tag', 'avas' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'h1' => __( 'H1', 'avas' ),
                    'h2' => __( 'H2', 'avas' ),
                    'h3' => __( 'H3', 'avas' ),
                    'h4' => __( 'H4', 'avas' ),
                    'h5' => __( 'H5', 'avas' ),
                    'h6' => __( 'H6', 'avas' ),
                    'div' => __( 'div', 'avas' ),
                ],
                'default' => 'h3',
            ]
        );

        $this->add_control(
            'heading_color',
            [
                'label' => __( 'Heading Color', 'avas' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .lae-gallery-wrap .lae-heading' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'heading_typography',
                'selector' => '{{WRAPPER}} .lae-gallery-wrap .lae-heading',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_filters_styling',
            [
                'label' => __('Gallery Filters', 'avas'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'filter_color',
            [
                'label' => __( 'Filter Color', 'avas' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .lae-gallery-wrap .lae-taxonomy-filter .lae-filter-item a' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'filter_hover_color',
            [
                'label' => __( 'Filter Hover Color', 'avas' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .lae-gallery-wrap .lae-taxonomy-filter .lae-filter-item a:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'filter_active_border',
            [
                'label' => __( 'Active Filter Border Color', 'avas' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .lae-gallery-wrap .lae-taxonomy-filter .lae-filter-item.lae-active' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'filter_typography',
                'selector' => '{{WRAPPER}} .lae-gallery-wrap .lae-taxonomy-filter .lae-filter-item a',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_thumbnail_styling',
            [
                'label' => __('Gallery Thumbnail', 'avas'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );


        $this->add_control(
            'thumbnail_hover_bg_color',
            [
                'label' => __( 'Thumbnail Hover Background Color', 'avas' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .lae-gallery-wrap .lae-gallery .lae-gallery-item .lae-project-image .lae-image-overlay' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'thumbnail_hover_opacity',
            [
                'label' => __( 'Thumbnail Hover Opacity (%)', 'avas' ),
                'type' => Controls_Manager::SLIDER,
                'default' => [
                    'size' => 0.5,
                ],
                'range' => [
                    'px' => [
                        'max' => 1,
                        'min' => 0.10,
                        'step' => 0.01,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .lae-gallery-wrap .lae-gallery .lae-gallery-item .lae-project-image:hover .lae-image-overlay' => 'opacity: {{SIZE}};',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_item_title_styling',
            [
                'label' => __('Gallery Item Title', 'avas'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'item_title_tag',
            [
                'label' => __( 'Title HTML Tag', 'avas' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'h1' => __( 'H1', 'avas' ),
                    'h2' => __( 'H2', 'avas' ),
                    'h3' => __( 'H3', 'avas' ),
                    'h4' => __( 'H4', 'avas' ),
                    'h5' => __( 'H5', 'avas' ),
                    'h6' => __( 'H6', 'avas' ),
                    'div' => __( 'div', 'avas' ),
                ],
                'default' => 'h3',
            ]
        );

        $this->add_control(
            'item_title_color',
            [
                'label' => __( 'Title Color', 'avas' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .lae-gallery-wrap .lae-gallery .lae-gallery-item .lae-project-image .lae-image-info .lae-entry-title a' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'title_hover_border_color',
            [
                'label' => __( 'Title Hover Border Color', 'avas' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .lae-gallery-wrap .lae-gallery .lae-gallery-item .lae-project-image .lae-image-info .lae-entry-title a:hover' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'item_title_typography',
                'selector' => '{{WRAPPER}} .lae-gallery-wrap .lae-gallery .lae-gallery-item .lae-project-image .lae-image-info .lae-entry-title',
            ]
        );



        $this->end_controls_section();

        $this->start_controls_section(
            'section_item_tags',
            [
                'label' => __('Gallery Item Tags', 'avas'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'item_tags_color',
            [
                'label' => __( 'Item Tags Color', 'avas' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .lae-gallery-wrap .lae-gallery .lae-gallery-item .lae-project-image .lae-image-info .lae-terms' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'item_tags_typography',
                'selector' => '{{WRAPPER}} .lae-gallery-wrap .lae-gallery .lae-gallery-item .lae-project-image .lae-image-info .lae-terms',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_pagination_styling',
            [
                'label' => __('Pagination', 'avas'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );


        $this->add_control(
            'pagination_border_color',
            [
                'label' => __( 'Border Color', 'avas' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .lae-gallery-wrap .lae-pagination .lae-page-nav, {{WRAPPER}} .lae-gallery-wrap .lae-pagination .lae-page-nav:first-child' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'pagination_hover_bg_color',
            [
                'label' => __( 'Hover Background Color', 'avas' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .lae-gallery-wrap .lae-pagination .lae-page-nav:hover, {{WRAPPER}} .lae-gallery-wrap .lae-pagination .lae-page-nav.lae-current-page' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'pagination_nav_icon_color',
            [
                'label' => __( 'Nav Icon Color', 'avas' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .lae-gallery-wrap .lae-pagination .lae-page-nav i' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'pagination_disabled_nav_icon_color',
            [
                'label' => __( 'Disabled Nav Icon Color', 'avas' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .lae-gallery-wrap .lae-pagination .lae-page-nav.lae-disabled i' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'heading_nav_text',
            [
                'label' => __( 'Navigation text', 'avas' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'pagination_text_color',
            [
                'label' => __( 'Nav Text Color', 'avas' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .lae-gallery-wrap .lae-pagination .lae-page-nav' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'label' => __( 'Nav Text Typography', 'avas' ),
                'name' => 'pagination_text_typography',
                'selector' => '{{WRAPPER}} .lae-gallery-wrap .lae-pagination .lae-page-nav',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_responsive',
            [
                'label' => __('Gutter Options', 'avas'),
                'tab' => Controls_Manager::TAB_SETTINGS,
            ]
        );

        $this->add_control(
            'heading_desktop',
            [
                'label' => __( 'Desktop', 'avas' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'after',
            ]
        );


        $this->add_control(
            'gutter',
            [
                'label' => __('Gutter', 'avas'),
                'description' => __('Space between columns in the grid.', 'avas'),
                'type' => Controls_Manager::NUMBER,
                'default' => 20,
                'selectors' => [
                    '{{WRAPPER}} .lae-gallery' => 'margin-left: -{{VALUE}}px; margin-right: -{{VALUE}}px;',
                    '{{WRAPPER}} .lae-gallery .lae-gallery-item' => 'padding: {{VALUE}}px;',
                ]
            ]
        );

        $this->add_control(
            'heading_tablet',
            [
                'label' => __( 'Tablet', 'avas' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'after',
            ]
        );


        $this->add_control(
            'tablet_gutter',
            [
                'label' => __('Gutter', 'avas'),
                'description' => __('Space between columns.', 'avas'),
                'type' => Controls_Manager::NUMBER,
                'default' => 10,
                'selectors' => [
                    '(tablet-){{WRAPPER}} .lae-gallery' => 'margin-left: -{{VALUE}}px; margin-right: -{{VALUE}}px;',
                    '(tablet-){{WRAPPER}} .lae-gallery .lae-gallery-item' => 'padding: {{VALUE}}px;',
                ]
            ]
        );

        $this->add_control(
            'tablet_width',
            [
                'label' => __('Tablet Resolution', 'avas'),
                'description' => __('The resolution to treat as a tablet resolution.', 'avas'),
                'type' => Controls_Manager::NUMBER,
                'default' => 800,
            ]
        );

        $this->add_control(
            'heading_mobile',
            [
                'label' => __( 'Mobile Phone', 'avas' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'after',
            ]
        );

        $this->add_control(
            'mobile_gutter',
            [
                'label' => __('Gutter', 'avas'),
                'description' => __('Space between columns.', 'avas'),
                'type' => Controls_Manager::NUMBER,
                'default' => 10,
                'selectors' => [
                    '(mobile-){{WRAPPER}} .lae-gallery' => 'margin-left: -{{VALUE}}px; margin-right: -{{VALUE}}px;',
                    '(mobile-){{WRAPPER}} .lae-gallery .lae-gallery-item' => 'padding: {{VALUE}}px;',
                ]
            ]
        );

        $this->add_control(
            'mobile_width',
            [
                'label' => __('Tablet Resolution', 'avas'),
                'description' => __('The resolution to treat as a tablet resolution.', 'avas'),
                'type' => Controls_Manager::NUMBER,
                'default' => 480,
            ]
        );

        $this->end_controls_section();

    }

    protected function render() {

        $settings = $this->get_settings();
        $items = $settings['gallery_items'];
        unset($settings['gallery_items']); // exclude items from settings

        if (!empty($items)) :

            $terms = bddex_get_gallery_terms($items);
            $max_num_pages = ceil(count($items) / $settings['items_per_page']);

            ?>

            <div class="lae-gallery-wrap lae-container"
                 data-settings='<?php echo wp_json_encode($settings); ?>'
                 data-items='<?php echo wp_json_encode($items); ?>'
                 data-maxpages='<?php echo $max_num_pages; ?>'
                 data-total='<?php echo count($items); ?>'
                 data-current='1'>

                <?php if (!empty($settings['heading']) || $settings['filterable'] == 'yes'): ?>

                    <div class="lae-gallery-header">

                        <?php if (!empty($settings['heading'])) : ?>

                            <<?php echo $settings['heading_tag']; ?> class="lae-heading"><?php echo wp_kses_post($settings['heading']); ?></<?php echo $settings['heading_tag']; ?>>

                        <?php endif; ?>

                        <?php

                        if ($settings['filterable'] == 'yes')
                            echo bddex_get_gallery_terms_filter($terms);

                        ?>

                    </div>

                <?php endif; ?>

                <div id="<?php echo uniqid('lae-gallery'); ?>"
                     class="lae-gallery js-isotope lae-<?php echo esc_attr($settings['layout_mode']); ?>"
                     data-isotope-options='{ "itemSelector": ".lae-gallery-item", "layoutMode": "<?php echo esc_attr($settings['layout_mode']); ?>" }'>

                    <?php bddex_display_gallery($items, $settings, 1); ?>

                </div><!-- Isotope items -->

                <?php echo bddex_paginate_gallery($items, $settings); ?>

            </div>

            <?php

        endif;
    }

    protected function content_template() {
    }

}

Plugin::instance()->widgets_manager->register_widget_type( new BDDEX_Gallery_Widget() );


add_action('wp_ajax_lae_load_gallery_items', 'bddex_load_gallery_items_callback');

add_action('wp_ajax_nopriv_lae_load_gallery_items', 'bddex_load_gallery_items_callback');

function bddex_load_gallery_items_callback() {
    $items = bddex_parse_items($_POST['items']);
    $settings = bddex_parse_gallery_settings($_POST['settings']);
    $paged = intval($_POST['paged']);

    bddex_display_gallery($items, $settings, $paged);

    wp_die();

}

function bddex_parse_items($items) {

    return $items;
}

function bddex_parse_gallery_settings($settings) {

    $s = $settings;

    $s['filterable'] = filter_var($s['filterable'], FILTER_VALIDATE_BOOLEAN);
    $s['per_line'] = filter_var($s['per_line'], FILTER_VALIDATE_INT);
    $s['items_per_page'] = filter_var($s['items_per_page'], FILTER_VALIDATE_INT);

    return $s;
}

function bddex_display_gallery($items, $settings, $paged = 1) {


    $num_of_columns = intval($settings['per_line']);
    $items_per_page = intval($settings['items_per_page']); ?>

    <?php $column_style = bddex_get_column_class($num_of_columns); ?>

    <?php
    // If pagination option is chosen, filter the items for the current page
    if ($settings['pagination'] != 'none')
        $items = bddex_get_items_to_display($items, $paged, $items_per_page);
    ?>

    <?php foreach ($items as $item): ?>

        <?php

        $style = '';
        if (!empty($item['item_tags'])) {
            $terms = explode(',', $item['item_tags']);

            foreach ($terms as $term) {
                // Get rid of spaces before adding the term
                $style .= ' term-' . preg_replace('/\s+/', '-', $term);
            }
        }
        ?>

        <?php

        $item_type = $item['item_type'];
        $item_class = 'lae-' . $item_type . '-type';

        ?>

        <div
            class="lae-gallery-item <?php echo $style; ?> <?php echo $column_style; ?> <?php echo $item_class; ?> lae-zero-margin">

            <div class="lae-project-image">

                <?php if ($item_type == 'image' && !empty($item['item_image']) && !empty($item['item_link']['url'])): ?>

                    <a href="<?php echo esc_url($item['item_link']['url']); ?>"
                       title="<?php echo esc_html($item['item_name']); ?>"><?php echo wp_get_attachment_image($item['item_image']['id'], 'large', false, array('class' => 'lae-image large', 'alt' => $item['item_name'])); ?> </a>

                <?php else: ?>

                    <?php echo wp_get_attachment_image($item['item_image']['id'], 'large', false, array('class' => 'lae-image large', 'alt' => $item['item_name'])); ?>

                <?php endif; ?>

                <div class="lae-image-info">

                    <div class="lae-entry-info">

                        <<?php echo $settings['item_title_tag']; ?> class="lae-entry-title">

                            <?php if ($item_type == 'image' && !empty($item['item_link']['url'])): ?>

                                <?php $target = $item['item_link']['is_external'] ? 'target="_blank"' : ''; ?>

                                <a href="<?php echo esc_url($item['item_link']['url']); ?>"
                                   title="<?php echo esc_html($item['item_name']); ?>"
                                   <?php echo $target; ?>><?php echo esc_html($item['item_name']); ?></a>

                            <?php else: ?>

                                <?php echo esc_html($item['item_name']); ?>

                            <?php endif; ?>

                        </<?php echo $settings['item_title_tag']; ?>>

                        <?php if ($item_type == 'youtube' || $item_type == 'vimeo') : ?>

                            <?php
                            $video_url = $item['video_link'];
                            ?>
                            <?php if (!empty($video_url)) : ?>

                                <a class="lae-video-lightbox"
                                   href="<?php echo $video_url; ?>"
                                   title="<?php echo esc_html($item['item_name']); ?>"><i
                                        class="lae-icon-video-play"></i></a>

                            <?php endif; ?>

                        <?php endif; ?>

                        <span class="lae-terms"><?php echo esc_html($item['item_tags']); ?></span>

                    </div>

                    <?php if ($item_type == 'image' && !empty($item['item_image']) && $settings['enable_lightbox']) : ?>

                        <?php
                        $image_data = wp_get_attachment_image_src($item['item_image']['id'], 'full');
                        ?>
                        <?php if ($image_data) : ?>

                            <?php $image_src = $image_data[0]; ?>

                            <a class="lae-lightbox-item"
                               href="<?php echo $image_src; ?>"
                               title="<?php echo esc_html($item['item_name']); ?>"><i
                                    class="lae-icon-full-screen"></i></a>

                        <?php endif; ?>

                    <?php endif; ?>


                </div>

                <div class="lae-image-overlay"></div>

            </div>

        </div>

        <?php

    endforeach;

}

function bddex_get_gallery_terms($items) {

    $tags = $terms = array();

    foreach ($items as $item) {
        $tags = array_merge($tags, explode(',', $item['item_tags']));
    }
    $terms = array_values(array_unique($tags));

    return $terms;

}

function bddex_get_items_to_display($items, $paged, $items_per_page) {

    $offset = $items_per_page * ($paged - 1);

    $items = array_slice($items, $offset, $items_per_page);

    return $items;
}

function bddex_paginate_gallery($items, $settings) {

    $pagination_type = $settings['pagination'];

    // no pagination required if option is not chosen by user or if all posts are already displayed
    if ($pagination_type == 'none' || count($items) <= $settings['items_per_page'])
        return;

    $max_num_pages = ceil(count($items) / $settings['items_per_page']);

    $output = '<div class="lae-pagination">';

    if ($pagination_type == 'load_more') {
        $output .= '<a href="#" class="lae-load-more lae-button">';
        $output .= __('Load More', 'livemesh-el-addons');
        if ($settings['show_remaining'])
            $output .= ' - ' . '<span>' . (count($items) - $settings['items_per_page']) . '</span>';
        $output .= '</a>';
    }
    else {
        $page_links = array();

        for ($n = 1; $n <= $max_num_pages; $n++) :
            $page_links[] = '<a class="lae-page-nav' . ($n == 1 ? ' lae-current-page' : '') . '" href="#" data-page="' . $n . '">' . number_format_i18n($n) . '</a>';
        endfor;

        $r = join("\n", $page_links);

        if (!empty($page_links)) {
            $prev_link = '<a class="lae-page-nav lae-disabled" href="#" data-page="prev"><i class="lae-icon-arrow-left3"></i></a>';
            $next_link = '<a class="lae-page-nav" href="#" data-page="next"><i class="lae-icon-arrow-right3"></i></a>';

            $output .= $prev_link . "\n" . $r . "\n" . $next_link;
        }
    }

    $output .= '<span class="lae-loading"></span>';

    $output .= '</div>';

    return $output;

}

/** Isotope filtering support for Gallery * */

function bddex_get_gallery_terms_filter($terms) {

    $output = '';

    if (!empty($terms)) {

        $output .= '<div class="lae-taxonomy-filter">';

        $output .= '<div class="lae-filter-item segment-0 lae-active"><a data-value="*" href="#">' . esc_html__('All', 'livemesh-el-addons') . '</a></div>';

        $segment_count = 1;
        foreach ($terms as $term) {

            $output .= '<div class="lae-filter-item segment-' . intval($segment_count) . '"><a href="#" data-value=".term-' . preg_replace('/\s+/', '-', $term) . '" title="' . esc_html__('View all items filed under ', 'livemesh-el-addons') . esc_attr($term) . '">' . esc_html($term) . '</a></div>';

            $segment_count++;
        }

        $output .= '</div>';

    }

    return $output;
}