<?php
namespace SLElementorWidgets\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Scheme_Color;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * @since 1.1.0
 */
class Luxus_User_Login extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.1.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'luxus-user-login';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.1.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'User Avatar', 'luxus' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.1.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'sl-theme-widget-icon user-avatar';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.1.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'luxus-widgets' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.1.0
	 *
	 * @access protected
	 */
	protected function register_controls() {

		$this->luxus_genetal_layout_options();
	}

	// Content Layout Options
	private function luxus_genetal_layout_options() {

		$this->start_controls_section(
			'section_layout_options',
			[
				'label' => __( 'Layout', 'luxus' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'user_login_text',
			[
				'label' => __( 'User Login Text', 'luxus' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				'default' => __( 'Login', 'luxus' ),
				'placeholder' => __( 'User login text here', 'luxus' ),
			]
		);
		
		$this->add_control(
			'user_login_text_color',
			[
				'type'      => Controls_Manager::COLOR,
				'label'     => __( 'User Login Text Color', 'luxus' ),
				'default' => '#00bbff',
				'separator' => '',
				'selectors' => [
					'{{WRAPPER}} .sl-user-login-container .sl-ajax-login' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'line_height',
			[
				'label' => __( 'Line Height', 'luxus' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px' ],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 200,
						'step' => 5,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 80,
				],
				'selectors' => [
					'{{WRAPPER}} .sl-user-login-container .user-loged-in' => 'line-height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'user_login_align',
			[
				'label' => __( 'Avatar Alignment', 'luxus' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'left' => [
						'title' => __( 'Left', 'luxus' ),
						'icon' => 'fa fa-align-left',
					],
					'center' => [
						'title' => __( 'Center', 'luxus' ),
						'icon' => 'fa fa-align-center',
					],
					'right' => [
						'title' => __( 'Right', 'luxus' ),
						'icon' => 'fa fa-align-right',
					],
				],
				'default' => 'center',
				'selectors' => [
					'{{WRAPPER}} .sl-user-login-container' => 'text-align: {{VALUE}};',
				],
				'separator' => 'before',
			]
		);

		$this->end_controls_section();

	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.1.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
		$user_login_text = $settings['user_login_text'];

		?>
			<div class="sl-user-login-container">
				<?php

                //CHECKING THE USER LOGED IN OR USER EXISTS OR NOT IN DATABASE
                if( is_user_logged_in() ):

                    $user = wp_get_current_user();

                    //GETTING PROFILE IMAGE OF USER
                    $user_meta_img_id = get_user_meta( $user->ID, 'profile_image_id', true );
                    $user_meta_img_url = !$user_meta_img_id == null ? wp_get_attachment_url( $user_meta_img_id ) : null;
                    $avatar_placeholder = esc_url(get_template_directory_uri()) . '/assets/images/luxus-avatar.png';
                    
                 ?>
                    <div class="user-loged-in">
                        <img src="<?php echo !$user_meta_img_url == null ? $user_meta_img_url : $avatar_placeholder; ?>" class="user-pic" alt="<?php esc_attr('Avatar'); ?>">
                        <div class="quick-links">
                            <div class="user-name"><?php echo esc_html( $user->display_name ); ?></div>
                            <ul>
                        	<?php if ( current_user_can('administrator') || current_user_can('editor') || current_user_can('author') || current_user_can('contributor') || current_user_can('subscriber') ) { ?>

                        		<li><a href="<?php echo esc_url(home_url('/wp-admin/')); ?>"><i class="fas fa-list"></i><?php esc_html_e('Dashboard', 'luxus'); ?></a></li>
                            	<li><a href="<?php echo esc_url(home_url('/wp-admin/profile.php')); ?>"><i class="far fa-user"></i><?php esc_html_e('Profile settings', 'luxus'); ?></a></li>

                            <?php } else { ?>

								<li><a href="<?php echo esc_url(home_url('/user-dashboard')); ?>"><i class="fas fa-list"></i><?php esc_html_e('Dashboard', 'luxus'); ?></a></li>
                            	<li><a href="<?php echo esc_url(home_url('/my-profile')); ?>"><i class="far fa-user"></i><?php esc_html_e('Profile settings', 'luxus'); ?></a></li>

                            <?php } ?>

                                <li><a href="<?php echo esc_url( wp_logout_url( home_url() ) ); ?>"><i class="fas fa-sign-out-alt"></i><?php esc_html_e('Logout', 'luxus'); ?></a></li>
                            </ul>
                        </div>
                    </div>

                <?php else: ?>
					<a class="sl-ajax-login" href="">

						<i class="sl-icon sl-avatar-o"></i> <?php echo !empty( $user_login_text ) ? $user_login_text : esc_html__( 'Login', 'luxus'); ?>
					</a>
                <?php endif;

				?>
			</div>
    	<?php
	}
}