<?php
namespace SLElementorWidgets\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Scheme_Color;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * @since 1.1.0
 */
class Luxus_Nav_Menu extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.1.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'luxus-nav-menu';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.1.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Nav Menu', 'luxus' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.1.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'sl-theme-widget-icon nav-menu';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.1.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'luxus-widgets' ];
	}

	// Get Menus
	private function luxus_nav_menus() {
		
		$options = array();

        $sl_menus = get_terms( 'nav_menu', array( 'hide_empty' => false ) );

        if ( is_array( $sl_menus ) && ! empty( $sl_menus ) ) {

            foreach ( $sl_menus as $menu ) {

                if ( is_object( $menu ) && isset( $menu->name, $menu->term_id ) ) {
                    $options[ $menu->term_id ] = $menu->name;
                }

            }
        }

		return $options;

	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.1.0
	 *
	 * @access protected
	 */
	protected function register_controls() {

		$this->luxus_genetal_layout_options();
		$this->luxus_style_main_menu_options();

	}

	// Content Layout Options
	private function luxus_genetal_layout_options() {

		$this->start_controls_section(
			'section_layout_options',
			[
				'label' => __( 'Layout', 'luxus' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'select_nav_menu',
			[
				'label'       => __( 'Select Nav Menu', 'luxus' ),
				'label_block' => true,
				'type'        => Controls_Manager::SELECT,
				'options'     => $this->luxus_nav_menus(),
				'description' => empty( $this->luxus_nav_menus() ) ? sprintf( esc_html__( 'Menus not found. Please visit %sAppearance > Menus%s page to create new menu.', 'luxus' ), '<b>', '</b>' ) : esc_html__( 'Select menu to display.', 'luxus' ),
			]
		);

		$this->add_control(
			'line_height',
			[
				'label' => __( 'Line Height', 'luxus' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px' ],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 200,
						'step' => 5,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 80,
				],
				'selectors' => [
					'{{WRAPPER}} .sl-nav-menu-container .main-menu' => 'line-height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'nav_menu_align',
			[
				'label' => __( 'Menu Alignment', 'luxus' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'left' => [
						'title' => __( 'Left', 'luxus' ),
						'icon' => 'fa fa-align-left',
					],
					'center' => [
						'title' => __( 'Center', 'luxus' ),
						'icon' => 'fa fa-align-center',
					],
					'right' => [
						'title' => __( 'Right', 'luxus' ),
						'icon' => 'fa fa-align-right',
					],
				],
				'default' => 'center',
				'selectors' => [
					'{{WRAPPER}} .sl-nav-menu-container .main-menu' => 'text-align: {{VALUE}};',
				],
				'separator' => 'before',
			]
		);

		$this->end_controls_section();

	}

	private function luxus_style_main_menu_options() {

		$this->start_controls_section(
			'section_main_menu_style',
			[
				'label' => __( 'Menu Style', 'luxus' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name'      => 'nav_submenu_box_shadow',
				'selector'  => '{{WRAPPER}} .sl-nav-menu-container .main-menu ul li ul.sub-menu',
			]
		);

		$this->start_controls_tabs( 'nav_menu_tabs' );

		// Normal tab.
		$this->start_controls_tab(
			'nav_menu_normal',
			[
				'label'     => __( 'Normal', 'luxus' ),
			]
		);

		$this->add_control(
			'nav_main_menu_color_normal',
			[
				'type'      => Controls_Manager::COLOR,
				'label'     => __( 'Main Menu Color', 'luxus' ),
				'default' => '#00bbff',
				'separator' => '',
				'selectors' => [
					'{{WRAPPER}} .sl-nav-menu-container .sl-menu ul li a' => 'color: {{VALUE}} !important;',
				],
			]
		);

		$this->add_control(
			'nav_sub_menu_color_normal',
			[
				'type'      => Controls_Manager::COLOR,
				'label'     => __( 'Sub Menu Color', 'luxus' ),
				'default' => '#00bbff',
				'separator' => '',
				'selectors' => [
					'{{WRAPPER}} .sl-nav-menu-container .sl-menu ul.sub-menu li a' => 'color: {{VALUE}} !important;',
				],
			]
		);

		$this->add_control(
			'nav_sub_menu_bg_color_normal',
			[
				'type'      => Controls_Manager::COLOR,
				'label'     => __( 'Sub Menu Background', 'luxus' ),
				'default' => '#ffffff',
				'separator' => '',
				'selectors' => [
					'{{WRAPPER}} .sl-nav-menu-container .sl-menu ul.sub-menu li' => 'background: {{VALUE}} !important;',
				],
			]
		);

		$this->end_controls_tab();

		// Hover tab.
		$this->start_controls_tab(
			'nav_menu_hover',
			[
				'label'     => __( 'Hover', 'luxus' ),
			]
		);

		$this->add_control(
			'nav_main_menu_color_hover',
			[
				'type'      => Controls_Manager::COLOR,
				'label'     => __( 'Main Menu Color', 'luxus' ),
				'default' => '#f0055',
				'separator' => '',
				'selectors' => [
					'{{WRAPPER}} .sl-nav-menu-container .sl-menu ul li a:hover' => 'color: {{VALUE}} !important;',
					'{{WRAPPER}} .sl-nav-menu-container .sl-menu ul li.current-menu-item>a' => 'color: {{VALUE}} !important;',
					'{{WRAPPER}} .sl-nav-menu-container .sl-menu ul li.current-menu-ancestor>a' => 'color: {{VALUE}} !important;',
				],
			]
		);

		$this->add_control(
			'nav_sub_menu_color_hover',
			[
				'type'      => Controls_Manager::COLOR,
				'label'     => __( 'Sub Menu Color', 'luxus' ),
				'default' => '#00bbff',
				'separator' => '',
				'selectors' => [
					'{{WRAPPER}} .sl-nav-menu-container .sl-menu ul.sub-menu li a:hover' => 'color: {{VALUE}} !important;',
					'{{WRAPPER}} .sl-nav-menu-container .sl-menu ul.sub-menu li.current-menu-item>a' => 'color: {{VALUE}} !important;',
					'{{WRAPPER}} .sl-nav-menu-container .sl-menu ul.sub-menu li.current-menu-ancestor>a' => 'color: {{VALUE}} !important;',
				],
			]
		);
		$this->add_control(
			'nav_sub_menu_bg_color_hover',
			[
				'type'      => Controls_Manager::COLOR,
				'label'     => __( 'Sub Menu Background', 'luxus' ),
				'default' => '#f5f5ff',
				'separator' => '',
				'selectors' => [
					'{{WRAPPER}} .sl-nav-menu-container .sl-menu ul.sub-menu li:hover' => 'background: {{VALUE}} !important;',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->end_controls_section();

	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.1.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
		$select_nav_menu = $settings['select_nav_menu'];

		?>
			<div class="sl-nav-menu-container">
				<div class="sl-menu main-menu">
		            <?php
		                wp_nav_menu( array(
		                	'theme_location' => 'primary-menu',
		                    'menu' => $select_nav_menu,
		                    'container'	=> '',
		              	) );
		            ?>
		        </div>
			</div>
    	<?php
	}
}
