<?php
/**
 * WooCommerce Compatibility File
 *
 * @link https://woocommerce.com/
 *
 * @package Applic
 */

/**
 * WooCommerce setup function.
 *
 * @link https://docs.woocommerce.com/document/third-party-custom-theme-compatibility/
 * @link https://github.com/woocommerce/woocommerce/wiki/Enabling-product-gallery-features-(zoom,-swipe,-lightbox)-in-3.0.0
 *
 * @return void
 */
function applic_woocommerce_setup() {
	add_theme_support( 'woocommerce' );
	add_theme_support( 'wc-product-gallery-zoom' );
	add_theme_support( 'wc-product-gallery-lightbox' );
	add_theme_support( 'wc-product-gallery-slider' );
}
add_action( 'after_setup_theme', 'applic_woocommerce_setup' );

/**
 * Add 'woocommerce-active' class to the body tag.
 *
 * @param  array $classes CSS classes applied to the body tag.
 * @return array $classes modified to include 'woocommerce-active' class.
 */
function applic_woocommerce_active_body_class( $classes ) {
	$classes[] = 'woocommerce-active';

	return $classes;
}
add_filter( 'body_class', 'applic_woocommerce_active_body_class' );

/**
 * Products per page.
 *
 * @return integer number of products.
 */
function applic_woocommerce_products_per_page() {
	return 9;
}
add_filter( 'loop_shop_per_page', 'applic_woocommerce_products_per_page' );

/**
 * Product gallery thumnbail columns.
 *
 * @return integer number of columns.
 */
function applic_woocommerce_thumbnail_columns() {
	return 3;
}
add_filter( 'woocommerce_product_thumbnails_columns', 'applic_woocommerce_thumbnail_columns' );

/**
 * Default loop columns on product archives.
 *
 * @return integer products per row.
 */
function applic_woocommerce_loop_columns() {
	return 3;
}
add_filter( 'loop_shop_columns', 'applic_woocommerce_loop_columns' );

/**
 * Related Products Args.
 *
 * @param array $args related products args.
 * @return array $args related products args.
 */
function applic_woocommerce_related_products_args( $args ) {
	$defaults = array(
		'posts_per_page' => 3,
		'columns'        => 3,
	);

	$args = wp_parse_args( $defaults, $args );

	return $args;
}
add_filter( 'woocommerce_output_related_products_args', 'applic_woocommerce_related_products_args' );

if ( ! function_exists( 'applic_woocommerce_product_columns_wrapper' ) ) {
	/**
	 * Product columns wrapper.
	 *
	 * @return  void
	 */
	function applic_woocommerce_product_columns_wrapper() {
		$columns = applic_woocommerce_loop_columns();
		echo '<div class="columns-' . absint( $columns ) . '">';
	}
}
add_action( 'woocommerce_before_shop_loop', 'applic_woocommerce_product_columns_wrapper', 40 );

if ( ! function_exists( 'applic_woocommerce_product_columns_wrapper_close' ) ) {
	/**
	 * Product columns wrapper close.
	 *
	 * @return  void
	 */
	function applic_woocommerce_product_columns_wrapper_close() {
		echo '</div>';
	}
}
add_action( 'woocommerce_after_shop_loop', 'applic_woocommerce_product_columns_wrapper_close', 40 );

/**
 * Remove default WooCommerce wrapper.
 */
remove_action( 'woocommerce_before_main_content', 'woocommerce_breadcrumb', 20 );

if ( ! function_exists( 'applic_woocommerce_cart_link_fragment' ) ) {
	/**
	 * Cart Fragments.
	 *
	 * Ensure cart contents update when products are added to the cart via AJAX.
	 *
	 * @param array $fragments Fragments to refresh via AJAX.
	 * @return array Fragments to refresh via AJAX.
	 */
	function applic_woocommerce_cart_link_fragment( $fragments ) {
		ob_start();
		applic_woocommerce_cart_link();
		$fragments['a.cart-contents'] = ob_get_clean();

		return $fragments;
	}
}
add_filter( 'woocommerce_add_to_cart_fragments', 'applic_woocommerce_cart_link_fragment' );

if ( ! function_exists( 'applic_woocommerce_cart_link' ) ) {
	/**
	 * Cart Link.
	 *
	 * Displayed a link to the cart including the number of items present and the cart total.
	 *
	 * @return void
	 */
	function applic_woocommerce_cart_link() {
		?>
		<a class="cart-contents" href="<?php echo esc_url( wc_get_cart_url() ); ?>" title="<?php esc_attr_e( 'View your shopping cart', 'applic' ); ?>">
			<?php
			$item_count_text = sprintf(
				/* translators: number of items in the mini cart. */
				_n( '%d item', '%d items', WC()->cart->get_cart_contents_count(), 'applic' ),
				WC()->cart->get_cart_contents_count()
			);
			?>
			<span class="amount"><?php echo wp_kses_data( WC()->cart->get_cart_subtotal() ); ?></span> <span class="count"><?php echo esc_html( $item_count_text ); ?></span>
		</a>
		<?php
	}
}

if ( ! function_exists( 'applic_woocommerce_header_cart' ) ) {
	/**
	 * Display Header Cart.
	 *
	 * @return void
	 */
	function applic_woocommerce_header_cart() {
		if ( is_cart() ) {
			$class = 'current-menu-item';
		} else {
			$class = '';
		}
		?>
		<ul id="site-header-cart" class="site-header-cart">
			<li class="<?php echo esc_attr( $class ); ?>">
				<?php applic_woocommerce_cart_link(); ?>
			</li>
			<li>
				<?php
				$instance = array(
					'title' => '',
				);

				the_widget( 'WC_Widget_Cart', $instance );
				?>
			</li>
		</ul>
		<?php
	}
}

/**
 * Add Custom Plus and Minus Button On WooCommerce Quantity.
 */
if ( !function_exists('applic_quantity_fields_script') ) {
	function applic_quantity_fields_script(){
	    ?>
	    <script type='text/javascript'>
	    jQuery( function( $ ) {
	        if ( ! String.prototype.getDecimals ) {
	            String.prototype.getDecimals = function() {
	                var num = this,
	                    match = ('' + num).match(/(?:\.(\d+))?(?:[eE]([+-]?\d+))?$/);
	                if ( ! match ) {
	                    return 0;
	                }
	                return Math.max( 0, ( match[1] ? match[1].length : 0 ) - ( match[2] ? +match[2] : 0 ) );
	            }
	        }
	        // Quantity "plus" and "minus" buttons
	        $( document.body ).on( 'click', '.plus, .minus', function() {
	            var $qty        = $( this ).closest( '.quantity' ).find( '.qty'),
	                currentVal  = parseFloat( $qty.val() ),
	                max         = parseFloat( $qty.attr( 'max' ) ),
	                min         = parseFloat( $qty.attr( 'min' ) ),
	                step        = $qty.attr( 'step' );

	            // Format values
	            if ( ! currentVal || currentVal === '' || currentVal === 'NaN' ) currentVal = 0;
	            if ( max === '' || max === 'NaN' ) max = '';
	            if ( min === '' || min === 'NaN' ) min = 0;
	            if ( step === 'any' || step === '' || step === undefined || parseFloat( step ) === 'NaN' ) step = 1;

	            // Change the value
	            if ( $( this ).is( '.plus' ) ) {
	                if ( max && ( currentVal >= max ) ) {
	                    $qty.val( max );
	                } else {
	                    $qty.val( ( currentVal + parseFloat( step )).toFixed( step.getDecimals() ) );
	                }
	            } else {
	                if ( min && ( currentVal <= min ) ) {
	                    $qty.val( min );
	                } else if ( currentVal > 0 ) {
	                    $qty.val( ( currentVal - parseFloat( step )).toFixed( step.getDecimals() ) );
	                }
	            }

	            // Trigger change event
	            $qty.trigger( 'change' );

	            return false;
	        });
	    });
	    </script>
	    <?php
	}
}
add_action( 'wp_footer' , 'applic_quantity_fields_script' );