<?php
/**
 * Plugin Name: Applic Core
 * Plugin URI: https://themeforest.net/user/quomodotheme/portfolio
 * Description: This plugin adds the core features to the Applic WordPress theme. You must have to install this plugin to work with this theme.
 * Version: 2.0
 * Author: QuomodoThemes
 * Author URI: https://themeforest.net/user/quomodotheme/portfolio
 * Text domain: applic-core
 */

if ( !defined('ABSPATH') )
    die('-1');


// Make sure the same class is not loaded twice in free/premium versions.
if ( !class_exists( 'Applic_core' ) ) {
	/**
	 * Main Applic Core Class
	 *
	 * The main class that initiates and runs the Applic Core plugin.
	 *
	 * @since 1.7.0
	 */
	final class Applic_core {
		/**
		 * Applic Core Version
		 *
		 * Holds the version of the plugin.
		 *
		 * @since 1.7.0
		 * @since 1.7.1 Moved from property with that name to a constant.
		 *
		 * @var string The plugin version.
		 */
		const  VERSION = '1.0' ;
		/**
		 * Minimum Elementor Version
		 *
		 * Holds the minimum Elementor version required to run the plugin.
		 *
		 * @since 1.7.0
		 * @since 1.7.1 Moved from property with that name to a constant.
		 *
		 * @var string Minimum Elementor version required to run the plugin.
		 */
		const  MINIMUM_ELEMENTOR_VERSION = '1.7.0';
		/**
		 * Minimum PHP Version
		 *
		 * Holds the minimum PHP version required to run the plugin.
		 *
		 * @since 1.7.0
		 * @since 1.7.1 Moved from property with that name to a constant.
		 *
		 * @var string Minimum PHP version required to run the plugin.
		 */
		const  MINIMUM_PHP_VERSION = '5.4' ;
		/**
		 * Instance
		 *
		 * Holds a single instance of the `Press_Elements` class.
		 *
		 * @since 1.7.0
		 *
		 * @access private
		 * @static
		 *
		 * @var Press_Elements A single instance of the class.
		 */
		private static  $_instance = null ;
		/**
		 * Instance
		 *
		 * Ensures only one instance of the class is loaded or can be loaded.
		 *
		 * @since 1.7.0
		 *
		 * @access public
		 * @static
		 *
		 * @return Press_Elements An instance of the class.
		 */
		public static function instance() {
			if ( is_null( self::$_instance ) ) {
				self::$_instance = new self();
			}
			return self::$_instance;
		}

		/**
		 * Clone
		 *
		 * Disable class cloning.
		 *
		 * @since 1.7.0
		 *
		 * @access protected
		 *
		 * @return void
		 */
		public function __clone() {
			// Cloning instances of the class is forbidden
			_doing_it_wrong( __FUNCTION__, esc_html__( 'Cheatin&#8217; huh?', 'applic-core' ), '1.7.0' );
		}

		/**
		 * Wakeup
		 *
		 * Disable unserializing the class.
		 *
		 * @since 1.7.0
		 *
		 * @access protected
		 *
		 * @return void
		 */
		public function __wakeup() {
			// Unserializing instances of the class is forbidden.
			_doing_it_wrong( __FUNCTION__, esc_html__( 'Cheatin&#8217; huh?', 'applic-core' ), '1.7.0' );
		}

		/**
		 * Constructor
		 *
		 * Initialize the Applic Core plugins.
		 *
		 * @since 1.7.0
		 *
		 * @access public
		 */
		public function __construct() {
			$this->core_includes();
			$this->init_hooks();
			do_action( 'press_elements_loaded' );
		}

		/**
		 * Include Files
		 *
		 * Load core files required to run the plugin.
		 *
		 * @since 1.7.0
		 *
		 * @access public
		 */
		public function core_includes() {
			// Blog page post like function
			require_once __DIR__ . '/inc/post-like.php';

			// Metaboxes
			require_once __DIR__ . '/inc/metaboxes.php';

            /**
             * Register widget area.
             *
             */
            require_once __DIR__ . '/wp-widgets/widgets.php';

            // Instagram Widget
            require_once __DIR__ . '/wp-widgets/instagram/instagram-api.php';
            require_once __DIR__ . '/wp-widgets/instagram/instagram-settings.php';
            require_once __DIR__ . '/wp-widgets/instagram/instagram-widget.php';
		}

		/**
		 * Init Hooks
		 *
		 * Hook into actions and filters.
		 *
		 * @since 1.7.0
		 *
		 * @access private
		 */
		private function init_hooks() {
			add_action( 'init', [ $this, 'i18n' ] );
			add_action( 'plugins_loaded', [ $this, 'init' ] );
		}

		/**
		 * Load Textdomain
		 *
		 * Load plugin localization files.
		 *
		 * @since 1.7.0
		 *
		 * @access public
		 */
		public function i18n() {
			load_plugin_textdomain( 'appland-core', false, plugin_basename( dirname( __FILE__ ) ) . '/languages' );
		}

		/**
		 * Init Applic Core
		 *
		 * Load the plugin after Elementor (and other plugins) are loaded.
		 *
		 * @since 1.0.0
		 * @since 1.7.0 The logic moved from a standalone function to this class method.
		 *
		 * @access public
		 */
		public function init() {

			if ( !did_action( 'elementor/loaded' ) ) {
				add_action( 'admin_notices', [ $this, 'admin_notice_missing_main_plugin' ] );
				return;
			}

			// Check for required Elementor version

			if ( !version_compare( ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=' ) ) {
				add_action( 'admin_notices', [ $this, 'admin_notice_minimum_elementor_version' ] );
				return;
			}

			// Check for required PHP version

			if ( version_compare( PHP_VERSION, self::MINIMUM_PHP_VERSION, '<' ) ) {
				add_action( 'admin_notices', [ $this, 'admin_notice_minimum_php_version' ] );
				return;
			}

			// Add new Elementor Categories
			add_action( 'elementor/init', [ $this, 'add_elementor_category' ] );
			// Register Widget Scripts
			add_action( 'elementor/frontend/after_register_scripts', [ $this, 'register_widget_scripts' ] );
			// Register Widget Styles
			add_action( 'admin_enqueue_scripts', [ $this, 'register_admin_styles' ] );
			add_action( 'elementor/frontend/after_enqueue_styles', [ $this, 'register_widget_styles' ] );
			add_action( 'elementor/frontend/after_register_styles', [ $this, 'register_widget_styles' ] );
			add_action( 'elementor/editor/before_enqueue_scripts', [ $this, 'register_widget_styles' ] );

			// Register New Widgets
			add_action( 'elementor/widgets/widgets_registered', [ $this, 'on_widgets_registered' ] );
		}

		/**
		 * Admin notice
		 *
		 * Warning when the site doesn't have Elementor installed or activated.
		 *
		 * @since 1.1.0
		 * @since 1.7.0 Moved from a standalone function to a class method.
		 *
		 * @access public
		 */
		public function admin_notice_missing_main_plugin() {
			$message = sprintf(
			/* translators: 1: Applic Core 2: Elementor */
				esc_html__( '"%1$s" requires "%2$s" to be installed and activated.', 'applic-core' ),
				'<strong>' . esc_html__( 'Applic core', 'applic-core' ) . '</strong>',
				'<strong>' . esc_html__( 'Elementor', 'applic-core' ) . '</strong>'
			);
			printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );
		}

		/**
		 * Admin notice
		 *
		 * Warning when the site doesn't have a minimum required Elementor version.
		 *
		 * @since 1.1.0
		 * @since 1.7.0 Moved from a standalone function to a class method.
		 *
		 * @access public
		 */
		public function admin_notice_minimum_elementor_version() {
			$message = sprintf(
			/* translators: 1: Applic Core 2: Elementor 3: Required Elementor version */
				esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'applic-core' ),
				'<strong>' . esc_html__( 'Applic Core', 'applic-core' ) . '</strong>',
				'<strong>' . esc_html__( 'Elementor', 'applic-core' ) . '</strong>',
				self::MINIMUM_ELEMENTOR_VERSION
			);
			printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );
		}

		/**
		 * Admin notice
		 *
		 * Warning when the site doesn't have a minimum required PHP version.
		 *
		 * @access public
		 */
		public function admin_notice_minimum_php_version() {
			$message = sprintf(
			/* translators: 1: Applic Core 2: PHP 3: Required PHP version */
				esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'applic-core' ),
				'<strong>' . esc_html__( 'Applic Core', 'applic-core' ) . '</strong>',
				'<strong>' . esc_html__( 'PHP', 'applic-core' ) . '</strong>',
				self::MINIMUM_PHP_VERSION
			);
			printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );
		}

		/**
		 * Add new Elementor Categories
		 *
		 * Register new widget categories for Applic Core widgets.
		 *
		 * @access public
		 */
		public function add_elementor_category() {
			\Elementor\Plugin::instance()->elements_manager->add_category( 'applic-elements', [
				'title' => __( 'Applic Elements', 'applic-core' ),
			], 1 );
		}

		/**
		 * Register Widget Scripts
		 *
		 * Register custom scripts required to run Applic Core.
		 *
		 * @access public
		 */
		public function register_widget_scripts() {
			// Typing Effect
			//wp_register_script( 'typedjs', plugins_url( 'press-elements/libs/typed/typed.js' ), array( 'jquery' ) );
		}

		/**
		 * Register Widget Styles
		 *
		 * Register custom styles required to run Applic Core.
		 *
		 * @access public
		 */
		public function register_widget_styles() {
			// Typing Effect
			wp_enqueue_style( 'applic-elementor-edit', plugins_url( 'assets/css/elementor-edit.css', __FILE__ ) );
		}


		/**
		 * Register Admin Styles
		 *
		 * Register custom styles required to Applic WordPress Admin Dashboard.
		 *
		 * @access public
		 */
		public function register_admin_styles() {
            wp_enqueue_style( 'applic-admin', plugins_url( 'assets/css/applic-core-admin.css', __FILE__ ) );
		}

		/**
		 * Register New Widgets
		 *
		 * Include Applic Core widgets files and register them in Elementor.
		 *
		 * @since 1.0.0
		 * @since 1.7.1 The method moved to this class.
		 *
		 * @access public
		 */
		public function on_widgets_registered() {
			$this->include_widgets();
			$this->register_widgets();
		}

		/**
		 * Include Widgets Files
		 *
		 * Load Applic Core widgets files.
		 *
		 * @since 1.0.0
		 * @since 1.7.1 The method moved to this class.
		 *
		 * @access private
		 */
		private function include_widgets() {
			require_once __DIR__ . '/widgets/Applic_hero.php';
			require_once __DIR__ . '/widgets/Applic_features.php';
			require_once __DIR__ . '/widgets/Applic_about_area.php';
			require_once __DIR__ . '/widgets/Applic_process_area.php';
			require_once __DIR__ . '/widgets/Applic_goal_area.php';
			require_once __DIR__ . '/widgets/Applic_video_area.php';
			require_once __DIR__ . '/widgets/Applic_screenshot_area.php';
			require_once __DIR__ . '/widgets/Applic_testimonials.php';
			require_once __DIR__ . '/widgets/Applic_pricing.php';
			require_once __DIR__ . '/widgets/Applic_faq.php';
			require_once __DIR__ . '/widgets/Applic_blog.php';
			require_once __DIR__ . '/widgets/Applic_contact.php';
			require_once __DIR__ . '/widgets/Applic_team.php';
		}

		/**
		 * Register Widgets
		 *
		 * Register Applic Core widgets.
		 *
		 * @since 1.0.0
		 * @since 1.7.1 The method moved to this class.
		 *
		 * @access private
		 */
		private function register_widgets() {
			// Site Elements
			\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \ApplicCore\Widgets\Applic_hero() );
			\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \ApplicCore\Widgets\Applic_features() );
			\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \ApplicCore\Widgets\Applic_about_area() );
			\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \ApplicCore\Widgets\Applic_process_area() );
			\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \ApplicCore\Widgets\Applic_goal_area() );
			\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \ApplicCore\Widgets\Applic_video_area() );
			\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \ApplicCore\Widgets\Applic_screenshot_area() );
			\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \ApplicCore\Widgets\Applic_testimonials() );
			\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \ApplicCore\Widgets\Applic_pricing() );
			\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \ApplicCore\Widgets\Applic_faq() );
			\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \ApplicCore\Widgets\Applic_blog() );
			\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \ApplicCore\Widgets\Applic_contact() );
			\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \ApplicCore\Widgets\Applic_team() );
		}

	}
}
// Make sure the same function is not loaded twice in free/premium versions.

if ( !function_exists( 'applic_core_load' ) ) {
	/**
	 * Load Applic Core
	 *
	 * Main instance of Press_Elements.
	 *
	 * @since 1.0.0
	 * @since 1.7.0 The logic moved from this function to a class method.
	 */
	function applic_core_load() {
		return Applic_core::instance();
	}

	// Run Applic Core
	applic_core_load();
}

add_action( 'wp_enqueue_scripts', function() {
	wp_dequeue_style('elementor-global');
});


add_action('admin_enqueue_script', '');